// Copyright (C) 2010 Ben Asselstine
//
//  This program is free software; you can redistribute it and/or modify
//  it under the terms of the GNU General Public License as published by
//  the Free Software Foundation; either version 3 of the License, or
//  (at your option) any later version.
//
//  This program is distributed in the hope that it will be useful,
//  but WITHOUT ANY WARRANTY; without even the implied warranty of
//  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
//  GNU Library General Public License for more details.
//
//  You should have received a copy of the GNU General Public License
//  along with this program; if not, write to the Free Software
//  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 
//  02110-1301, USA.

#include "listerola-options.h"

const double default_num_rows = 4.0;
const double default_duration = 1.5;
const double default_interval = 0.5;
const double default_flip_interval = 5.0;

ListerolaOptions::ListerolaOptions()
  : num_rows(default_num_rows), duration(default_duration), 
    interval(default_interval), flip_interval(default_flip_interval),
    listing_file(""), listing_kind(SOURCE_NONE)
{
}
	
ListerolaOptions::ListerolaOptions(ListerolaOptions &orig)
  : num_rows(orig.num_rows), duration(orig.duration), interval(orig.interval),
    flip_interval(orig.flip_interval), listing_file(orig.listing_file),
    listing_kind(orig.listing_kind)
{
}

ListerolaOptions::~ListerolaOptions()
{
}

void ListerolaOptions::set_num_rows(double rows)
{
  num_rows = rows; 
  num_rows_changed.emit();
}

void ListerolaOptions::set_duration(double hours)
{
  duration = hours; 
  duration_changed.emit();
}

void ListerolaOptions::set_interval(double hours)
{
  interval = hours; 
  interval_changed.emit();
}

void ListerolaOptions::set_flipahead_interval(double mins)
{
  flip_interval = mins; 
  flipahead_interval_changed.emit();
}
        
void ListerolaOptions::set_xmltv_listing_file(std::string file)
{
  listing_file = file;
  listing_kind = SOURCE_XMLTV;
  listing_file_changed.emit();
}
        
void ListerolaOptions::set_mrbs_listing_file(std::string file)
{
  listing_file = file;
  listing_kind = SOURCE_MRBS;
  listing_file_changed.emit();
}
        
void ListerolaOptions::set_random_listing_file(std::string file)
{
  listing_file = file;
  listing_kind = SOURCE_RANDOM;
  listing_file_changed.emit();
}

void ListerolaOptions::clear_listing_file()
{
  listing_file = "";
  listing_kind = SOURCE_NONE;
  listing_file_changed.emit();
}

std::list<std::string> ListerolaOptions::get_numbered_hosts()
{
  std::list<std::string> h;
  std::map<std::string, int>::iterator it = host_numbers.begin();
  for (; it != host_numbers.end(); it++)
    h.push_back((*it).first);
  return h;
}

int ListerolaOptions::get_host_number(std::string host)
{
  std::map<std::string, int>::iterator it;
  it = host_numbers.find(host);
  if (it == host_numbers.end())
    return -1;
  return (*it).second;
}
        
void ListerolaOptions::set_host_number(std::string host, int number)
{
  std::map<std::string, int>::iterator it;
  it = host_numbers.find(host);
  if (it != host_numbers.end())
    (*it).second = number;
  else
    host_numbers[host] = number;
  host_numbers_changed.emit();
}
        
void ListerolaOptions::clear_host_number(std::string host)
{
  std::map<std::string, int>::iterator it;
  it = host_numbers.find(host);
  if (it != host_numbers.end())
    {
      host_numbers.erase(it);
      host_numbers_changed.emit();
    }
}

std::list<std::string> ListerolaOptions::get_named_hosts()
{
  std::list<std::string> h;
  std::map<std::string, std::string>::iterator it = host_names.begin();
  for (; it != host_names.end(); it++)
    h.push_back((*it).first);
  return h;
}

std::string ListerolaOptions::get_host_name(std::string host)
{
  std::map<std::string, std::string>::iterator it;
  it = host_names.find(host);
  if (it == host_names.end())
    return "";
  return (*it).second;
}
        
void ListerolaOptions::set_host_name(std::string host, std::string name)
{
  std::map<std::string, std::string>::iterator it;
  it = host_names.find(host);
  if (it != host_names.end())
    (*it).second = name;
  else
    host_names[host] = name;
  host_names_changed.emit();
}

void ListerolaOptions::clear_host_name(std::string host)
{
  std::map<std::string, std::string>::iterator it;
  it = host_names.find(host);
  if (it != host_names.end())
    {
      host_names.erase(it);
      host_names_changed.emit();
    }
}
