/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006, 2007, 2008  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.
  

  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <string.h>

#include "ker.h"
#include "ker-internal.h"

int
_lw6ker_team_init (lw6ker_team_t * team, lw6ker_map_struct_t * map_struct,
		   lw6map_options_t * options)
{
  int ret = 0;
  int i;

  team->active = 0;
  team->map_struct = map_struct;
  team->gradient =
    (lw6ker_zone_state_t *) LW6SYS_CALLOC (map_struct->nb_zones *
					   sizeof (lw6ker_zone_state_t));
  team->cursor_ref_pot = options->cursor_pot_init;
  team->last_spread_dir = LW6KER_DIR_NNE;
  lw6ker_cursor_array_clear (&(team->cursor_array));

  if (team->gradient)
    {
      for (i = 0; i < map_struct->nb_zones; ++i)
	{
	  /*
	   * Set direction to "uncalculated"
	   */
	  team->gradient[i].direction_to_cursor = -1;
	  /*
	   * Set closest cursor to the center of the square
	   */
	  team->gradient[i].closest_cursor_pos.x =
	    map_struct->zones[i].pos.x + map_struct->zones[i].size / 2;
	  team->gradient[i].closest_cursor_pos.y =
	    map_struct->zones[i].pos.y + map_struct->zones[i].size / 2;
	}
      ret = 1;
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING, _("unable to allocate gradient"));
    }

  return ret;
}

void
_lw6ker_team_clear (lw6ker_team_t * team)
{
  if (team->gradient)
    {
      LW6SYS_FREE (team->gradient);
    }

  memset (team, 0, sizeof (lw6ker_team_t));
}

int
_lw6ker_team_copy (lw6ker_team_t * dst, lw6ker_team_t * src)
{
  int ret = 0;

  if (dst && src && dst->map_struct && src->map_struct
      && dst->map_struct == src->map_struct)
    {
      dst->active = src->active;
      memcpy (dst->gradient, src->gradient,
	      src->map_struct->nb_zones * sizeof (lw6ker_zone_state_t));
      dst->cursor_ref_pot = src->cursor_ref_pot;
      dst->last_spread_dir = src->last_spread_dir;
      dst->cursor_array = src->cursor_array;
      ret = 1;
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING,
		  _
		  ("team_copy only works if dst and src point to the same map_struct"));
    }

  return ret;
}

void
_lw6ker_team_update_checksum (lw6ker_team_t * team, u_int32_t * checksum)
{
  int i;

  lw6sys_checksum_update_int32 (checksum, team->active);
  // map_struct checksumed elsewhere
  for (i = 0; i < team->map_struct->nb_zones; ++i)
    {
      _lw6ker_zone_state_update_checksum (&(team->gradient[i]), checksum);
    }
  lw6sys_checksum_update_int32 (checksum, team->cursor_ref_pot);
  lw6sys_checksum_update_int32 (checksum, team->last_spread_dir);
  _lw6ker_cursor_array_update_checksum (&(team->cursor_array), checksum);
}

void
lw6ker_team_activate (lw6ker_team_t * team, int32_t nb_cursors,
		      lw6sys_xy_t pos)
{
  team->active = 1;
  _lw6ker_cursor_array_activate (&(team->cursor_array), nb_cursors, pos);
}

void
lw6ker_team_unactivate (lw6ker_team_t * team)
{
  team->active = 0;
  lw6ker_cursor_array_clear (&(team->cursor_array));
}

void
lw6ker_team_apply_cursors (lw6ker_team_t * team, lw6map_options_t * options)
{
  int32_t i;
  int32_t zone_id;
  int32_t max_pot = 0;
  int32_t round_delta = options->round_delta;

  /*
   * Wizardry to determine a new cursor_ref_pot. In most cases
   * it will end with max_pot being the old cursor_ref_pot+1
   * and delta being 1. But it's there for special cases, mostly
   * at game startup and when one fiddles arround with cursors
   * pot_offset fields..
   */
  max_pot = team->cursor_ref_pot;
  for (i = 0; i < team->cursor_array.nb_cursors; ++i)
    {
      zone_id =
	lw6ker_map_struct_get_zone_id (team->map_struct,
				       team->cursor_array.cursors[i].pos.x,
				       team->cursor_array.cursors[i].pos.y);
      if (zone_id < 0)
	{
	  zone_id =
	    lw6ker_map_struct_get_zone_id (team->map_struct,
					   team->cursor_array.cursors[i].
					   last_applied_pos.x,
					   team->cursor_array.cursors[i].
					   last_applied_pos.y);
	}
      if (zone_id >= 0)
	{
	  max_pot = lw6sys_max (max_pot, team->gradient[zone_id].potential);

	  round_delta =
	    lw6sys_max (round_delta,
			team->cursor_ref_pot +
			team->cursor_array.cursors[i].pot_offset -
			team->gradient[zone_id].potential);
	  round_delta = lw6sys_min (round_delta, options->max_round_delta);
	}
    }

  /*
   * Following line is essential, it ensures that cursor potentials
   * will increase at each game turn. A consequence is that one
   * shouldn't call apply_cursors twice per game round. Indeed,
   * calling apply_cursors changes the map state (cursor_ref_pot)
   * so calling it twice isn't the same as calling it only once.
   */
  team->cursor_ref_pot = max_pot + round_delta;

  if (team->cursor_ref_pot + options->max_cursor_pot_offset >
      options->max_cursor_pot)
    {
      lw6ker_team_normalize_pot (team, options);
    }

  /*
   * We actually apply the cursors
   */
  for (i = 0; i < team->cursor_array.nb_cursors; ++i)
    {
      zone_id =
	lw6ker_map_struct_get_zone_id (team->map_struct,
				       team->cursor_array.cursors[i].pos.x,
				       team->cursor_array.cursors[i].pos.y);
      if (zone_id >= 0)
	{
	  /*
	   * Beware of multiple cursors, we check again if changing
	   * the value is needed.
	   */
	  if (team->gradient[zone_id].potential <
	      team->cursor_ref_pot + team->cursor_array.cursors[i].pot_offset)
	    {
	      team->gradient[zone_id].potential =
		team->cursor_ref_pot +
		team->cursor_array.cursors[i].pot_offset;
	      _lw6ker_cursor_update_last_applied (&
						  (team->cursor_array.
						   cursors[i]));
	      team->gradient[zone_id].closest_cursor_pos =
		team->cursor_array.cursors[i].pos;
	    }
	}
      else
	{
	  zone_id =
	    lw6ker_map_struct_get_zone_id (team->map_struct,
					   team->cursor_array.cursors[i].
					   last_applied_pos.x,
					   team->cursor_array.cursors[i].
					   last_applied_pos.y);
	  if (zone_id >= 0)
	    {
	      /*
	       * Beware of multiple cursors, we check again if changing
	       * the value is needed.
	       */
	      if (team->gradient[zone_id].potential <
		  team->cursor_ref_pot +
		  team->cursor_array.cursors[i].pot_offset)
		{
		  team->gradient[zone_id].potential =
		    team->cursor_ref_pot +
		    team->cursor_array.cursors[i].pot_offset;
		}
	    }
	  else
	    {
	      lw6sys_log (LW6SYS_LOG_WARNING,
			  _
			  ("unable to apply cursor, neither pos (%d,%d) nor last_applied_pos (%d,%d) seem to point on a valid zone"),
			  team->cursor_array.cursors[i].pos.x,
			  team->cursor_array.cursors[i].pos.y,
			  team->cursor_array.cursors[i].last_applied_pos.x,
			  team->cursor_array.cursors[i].last_applied_pos.y);
	    }
	}
    }
}

static int32_t
next_dir (int32_t dir)
{
  int ret;
  /*
   * Arbitrary hard-coded order for directions. (diags = 1 4 7 10)
   * 0 5 1
   * 3 8 4
   * 6 11 7
   * 9 2 10
   */
  static int corres[LW6KER_NB_DIRS] =
    { 5, 3, 10, 8, 6, 1, 11, 9, 4, 2, 0, 7 };

  ret = corres[dir];

  /*
     ret = dir + (LW6KER_NB_DIRS / 2 + 1);
     ret = ret % LW6KER_NB_DIRS;
   */

  return ret;
}

/*
 * Highest potential = closest to the cursor
 */
void
lw6ker_team_spread_gradient (lw6ker_team_t * team)
{
  int32_t i;
  int32_t n;
  int32_t zone_id;
  int32_t new_potential;
  lw6ker_zone_struct_t *zone_structs;
  lw6ker_zone_state_t *zone_states;
  int32_t dir;

  n = team->map_struct->nb_zones;
  zone_structs = team->map_struct->zones;
  zone_states = team->gradient;

  team->last_spread_dir = next_dir (team->last_spread_dir);
  dir = team->last_spread_dir;

  switch (dir)
    {
    case LW6KER_DIR_ENE:
    case LW6KER_DIR_ESE:
    case LW6KER_DIR_SE:
    case LW6KER_DIR_SSE:
    case LW6KER_DIR_SSW:
    case LW6KER_DIR_SW:
      for (i = 0; i < n; ++i)
	{
	  if (((zone_id = zone_structs[i].link[dir]) >= 0)
	      && (zone_states[zone_id].potential
		  < (new_potential =
		     zone_states[i].potential - zone_structs[i].size)))
	    {
	      zone_states[zone_id].potential = new_potential;
	      zone_states[zone_id].direction_to_cursor = -1;
	    }
	}
      break;
    case LW6KER_DIR_WSW:
    case LW6KER_DIR_WNW:
    case LW6KER_DIR_NW:
    case LW6KER_DIR_NNW:
    case LW6KER_DIR_NNE:
    case LW6KER_DIR_NE:
      for (i = n - 1; i >= 0; --i)
	{
	  if (((zone_id = zone_structs[i].link[dir]) >= 0)
	      && (zone_states[zone_id].potential
		  < (new_potential =
		     zone_states[i].potential - zone_structs[i].size)))
	    {
	      zone_states[zone_id].potential = new_potential;
	      zone_states[zone_id].direction_to_cursor = -1;
	    }
	}
      break;
    default:
      lw6sys_log (LW6SYS_LOG_WARNING,
		  _("unable to spread gradient, incorrect dir=%d"), dir);
      break;
    }
}

void
lw6ker_team_normalize_pot (lw6ker_team_t * team, lw6map_options_t * options)
{
  int32_t i;
  int32_t n;
  u_int32_t min_pot = options->max_cursor_pot;
  u_int32_t max_pot = 0;
  u_int32_t delta;
  lw6ker_zone_state_t *zone_states = team->gradient;

  n = team->map_struct->nb_zones;
  for (i = 0; i < n; ++i)
    {
      min_pot = lw6sys_min (min_pot, zone_states[i].potential);
      max_pot = lw6sys_max (max_pot, zone_states[i].potential);
    }

  /*
   * Theorically, we could use delta=min_pot, but in the
   * case of an extremely big map, for which even after
   * long spreading, there are still pot=0 points, we
   * fallback on max_pot/2, that is we divide by 2
   * the cursor potential.
   */
  delta = lw6sys_max (min_pot, max_pot / 2);

  for (i = 0; i < n; ++i)
    {
      zone_states[i].potential -= delta;
      if (zone_states[i].potential <= 0
	  || zone_states[i].potential > options->max_cursor_pot)
	{
	  zone_states[i].potential = options->cursor_pot_init;
	}
      zone_states[i].direction_to_cursor = -1;
    }

  team->cursor_ref_pot = max_pot - delta;
  if (team->cursor_ref_pot <= 0 || team->cursor_ref_pot > max_pot)
    {
      lw6sys_log (LW6SYS_LOG_WARNING,
		  _("inconsistent cursor_ref_pot=%d (max_pot=%d, delta=%d)"),
		  team->cursor_ref_pot, max_pot, delta);
      team->cursor_ref_pot = options->cursor_pot_init;
    }
}
