/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006, 2007, 2008  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.
  

  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <string.h>

#include "hlp.h"

/**
 * lw6hlp_is_listed
 *
 * @keyword: the keyword we want to check out
 *
 * Checks wether a given keyword is listed or not, in any of the
 * know options list. In fact, the answer is just "does this mean
 * anything to Liquid War 6 at all".
 *
 * Return value: 1 if listed, 0 if not.
 */
int
lw6hlp_is_listed (char *keyword)
{
  int ret = 0;
  lw6sys_list_t *list = NULL;

  list = lw6hlp_list ();
  if (list)
    {
      ret = lw6hlp_is_listed_here (list, keyword);
      lw6sys_list_free (list);
    }

  return ret;
}

/**
 * lw6hlp_is_listed_here
 *
 * @here: the list of keywords we want to check
 * @keyword: the keyword we want to check out
 *
 * Checks wether a given keyword is present in the given list or not.
 *
 * Return value: 1 if listed, 0 if not.
 */
int
lw6hlp_is_listed_here (lw6sys_list_t * list, char *keyword)
{
  int ret = 0;
  char *entry = NULL;
  lw6sys_list_t *pos;

  pos = list;
  for (pos = list; (!ret) && (pos != NULL) && (pos->data != NULL);
       pos = lw6sys_list_next (pos))
    {
      entry = (char *) pos->data;
      if (lw6hlp_match (entry, keyword))
	{
	  ret = 1;
	  break;
	}
    }

  return ret;
}

static int
update_quick (lw6sys_list_t ** list)
{
  lw6sys_list_push_back (list, "help");
  lw6sys_list_push_back (list, "copyright");
  lw6sys_list_push_back (list, "version");
  lw6sys_list_push_back (list, "test");
  lw6sys_list_push_back (list, "pedigree");
  lw6sys_list_push_back (list, "audit");
  lw6sys_list_push_back (list, "modules");
  lw6sys_list_push_back (list, "reset");
  lw6sys_list_push_back (list, "defaults");
  lw6sys_list_push_back (list, "list");
  lw6sys_list_push_back (list, "about");
  lw6sys_list_push_back (list, "debug");

  return ((*list) != NULL);
}

lw6sys_list_t *
lw6hlp_list_quick ()
{
  lw6sys_list_t *list = NULL;

  list = lw6sys_list_new (NULL);
  if (list)
    {
      update_quick (&list);
    }

  return list;
}

static int
update_doc (lw6sys_list_t ** list)
{
  lw6sys_list_push_back (list, "list-quick");
  lw6sys_list_push_back (list, "list-doc");
  lw6sys_list_push_back (list, "list-show");
  lw6sys_list_push_back (list, "list-path");
  lw6sys_list_push_back (list, "list-players");
  lw6sys_list_push_back (list, "list-input");
  lw6sys_list_push_back (list, "list-graphics");
  lw6sys_list_push_back (list, "list-sound");
  lw6sys_list_push_back (list, "list-network");
  lw6sys_list_push_back (list, "list-map");
  lw6sys_list_push_back (list, "list-map-options");
  lw6sys_list_push_back (list, "list-map-style");
  lw6sys_list_push_back (list, "list-funcs");
  lw6sys_list_push_back (list, "list-hooks");
  lw6sys_list_push_back (list, "list-tuning");
  lw6sys_list_push_back (list, "list-aliases");
  lw6sys_list_push_back (list, "example-options-xml");
  lw6sys_list_push_back (list, "example-style-xml");

  return ((*list) != NULL);
}

lw6sys_list_t *
lw6hlp_list_doc ()
{
  lw6sys_list_t *list = NULL;

  list = lw6sys_list_new (NULL);
  if (list)
    {
      update_doc (&list);
    }

  return list;
}

static int
update_show (lw6sys_list_t ** list)
{
  lw6sys_list_push_back (list, "show-build-package-tarname");
  lw6sys_list_push_back (list, "show-build-package-name");
  lw6sys_list_push_back (list, "show-build-package-string");
  lw6sys_list_push_back (list, "show-build-version");
  lw6sys_list_push_back (list, "show-build-codename");
  lw6sys_list_push_back (list, "show-build-stamp");
  lw6sys_list_push_back (list, "show-build-md5sum");
  lw6sys_list_push_back (list, "show-build-copyright");
  lw6sys_list_push_back (list, "show-build-license");
  lw6sys_list_push_back (list, "show-build-configure-args");
  lw6sys_list_push_back (list, "show-build-gcc-version");
  lw6sys_list_push_back (list, "show-build-cflags");
  lw6sys_list_push_back (list, "show-build-ldflags");
  lw6sys_list_push_back (list, "show-build-hostname");
  lw6sys_list_push_back (list, "show-build-date");
  lw6sys_list_push_back (list, "show-build-time");
  lw6sys_list_push_back (list, "show-build-target-cpu");
  lw6sys_list_push_back (list, "show-build-target-os");
  lw6sys_list_push_back (list, "show-build-ms-windows");
  lw6sys_list_push_back (list, "show-build-top-srcdir");
  lw6sys_list_push_back (list, "show-build-prefix");
  lw6sys_list_push_back (list, "show-build-datadir");
  lw6sys_list_push_back (list, "show-build-libdir");
  lw6sys_list_push_back (list, "show-build-includedir");
  lw6sys_list_push_back (list, "show-build-localedir");
  lw6sys_list_push_back (list, "show-build-docdir");
  lw6sys_list_push_back (list, "show-build-enable-console");
  lw6sys_list_push_back (list, "show-build-enable-mod-gl");
  lw6sys_list_push_back (list, "show-build-enable-mod-csound");
  lw6sys_list_push_back (list, "show-build-enable-mod-ogg");
  lw6sys_list_push_back (list, "show-build-enable-mod-http");
  lw6sys_list_push_back (list, "show-build-enable-allinone");
  lw6sys_list_push_back (list, "show-build-enable-fullstatic");
  lw6sys_list_push_back (list, "show-build-enable-gprof");
  lw6sys_list_push_back (list, "show-build-enable-valgrind");
  lw6sys_list_push_back (list, "show-default-user-dir");
  lw6sys_list_push_back (list, "show-default-config-file");
  lw6sys_list_push_back (list, "show-default-log-file");
  lw6sys_list_push_back (list, "show-default-prefix");
  lw6sys_list_push_back (list, "show-default-mod-dir");
  lw6sys_list_push_back (list, "show-default-data-dir");
  lw6sys_list_push_back (list, "show-default-map-dir");
  lw6sys_list_push_back (list, "show-default-map-path");
  lw6sys_list_push_back (list, "show-default-script-file");
  lw6sys_list_push_back (list, "show-cwd");
  lw6sys_list_push_back (list, "show-run-dir");
  lw6sys_list_push_back (list, "show-user-dir");
  lw6sys_list_push_back (list, "show-config-file");
  lw6sys_list_push_back (list, "show-log-file");
  lw6sys_list_push_back (list, "show-prefix");
  lw6sys_list_push_back (list, "show-mod-dir");
  lw6sys_list_push_back (list, "show-data-dir");
  lw6sys_list_push_back (list, "show-map-dir");
  lw6sys_list_push_back (list, "show-map-path");
  lw6sys_list_push_back (list, "show-script-file");

  return ((*list) != NULL);
}

lw6sys_list_t *
lw6hlp_list_show ()
{
  lw6sys_list_t *list = NULL;

  list = lw6sys_list_new (NULL);
  if (list)
    {
      update_show (&list);
    }

  return list;
}

static int
update_path (lw6sys_list_t ** list)
{
  lw6sys_list_push_back (list, "run-dir");
  lw6sys_list_push_back (list, "user-dir");
  lw6sys_list_push_back (list, "config-file");
  lw6sys_list_push_back (list, "log-file");
  lw6sys_list_push_back (list, "prefix");
  lw6sys_list_push_back (list, "mod-dir");
  lw6sys_list_push_back (list, "data-dir");
  lw6sys_list_push_back (list, "map-dir");
  lw6sys_list_push_back (list, "map-path");
  lw6sys_list_push_back (list, "script-file");

  return ((*list) != NULL);
}

lw6sys_list_t *
lw6hlp_list_path ()
{
  lw6sys_list_t *list = NULL;

  list = lw6sys_list_new (NULL);
  if (list)
    {
      update_path (&list);
    }

  return list;
}

static int
update_players (lw6sys_list_t ** list)
{
  lw6sys_list_push_back (list, "player1-name");
  lw6sys_list_push_back (list, "player2-name");
  lw6sys_list_push_back (list, "player3-name");
  lw6sys_list_push_back (list, "player4-name");

  return ((*list) != NULL);
}

lw6sys_list_t *
lw6hlp_list_players ()
{
  lw6sys_list_t *list = NULL;

  list = lw6sys_list_new (NULL);
  if (list)
    {
      update_players (&list);
    }

  return list;
}

static int
update_input (lw6sys_list_t ** list)
{
  return ((*list) != NULL);
}

lw6sys_list_t *
lw6hlp_list_input ()
{
  lw6sys_list_t *list = NULL;

  list = lw6sys_list_new (NULL);
  if (list)
    {
      update_input (&list);
    }

  return list;
}

static int
update_graphics (lw6sys_list_t ** list)
{
  lw6sys_list_push_back (list, "width");
  lw6sys_list_push_back (list, "height");
  lw6sys_list_push_back (list, "fullscreen");
  lw6sys_list_push_back (list, "display-fps");

  return ((*list) != NULL);
}

lw6sys_list_t *
lw6hlp_list_graphics ()
{
  lw6sys_list_t *list = NULL;

  list = lw6sys_list_new (NULL);
  if (list)
    {
      update_graphics (&list);
    }

  return list;
}

static int
update_sound (lw6sys_list_t ** list)
{
  lw6sys_list_push_back (list, "sound-volume");
  lw6sys_list_push_back (list, "music-volume");

  return ((*list) != NULL);
}

lw6sys_list_t *
lw6hlp_list_sound ()
{
  lw6sys_list_t *list = NULL;

  list = lw6sys_list_new (NULL);
  if (list)
    {
      update_sound (&list);
    }

  return list;
}

static int
update_network (lw6sys_list_t ** list)
{
  return ((*list) != NULL);
}

lw6sys_list_t *
lw6hlp_list_network ()
{
  lw6sys_list_t *list = NULL;

  list = lw6sys_list_new (NULL);
  if (list)
    {
      update_network (&list);
    }

  return list;
}

static int
update_map (lw6sys_list_t ** list)
{
  lw6sys_list_push_back (list, "chosen-map");
  lw6sys_list_push_back (list, "use-map-options");
  lw6sys_list_push_back (list, "use-map-style");
  lw6sys_list_push_back (list, "force-time");
  lw6sys_list_push_back (list, "force-size");
  lw6sys_list_push_back (list, "force-background");
  lw6sys_list_push_back (list, "force-hud");
  lw6sys_list_push_back (list, "force-menu");
  lw6sys_list_push_back (list, "force-view");
  lw6sys_list_push_back (list, "force-team-colors");

  return ((*list) != NULL);
}

lw6sys_list_t *
lw6hlp_list_map ()
{
  lw6sys_list_t *list = NULL;

  list = lw6sys_list_new (NULL);
  if (list)
    {
      update_map (&list);
    }

  return list;
}

static int
update_map_options (lw6sys_list_t ** list)
{
  lw6sys_list_push_back (list, "total-time");
  lw6sys_list_push_back (list, "respawn-team");
  lw6sys_list_push_back (list, "moves-per-round");
  lw6sys_list_push_back (list, "spreads-per-round");
  lw6sys_list_push_back (list, "rounds-per-sec");
  lw6sys_list_push_back (list, "fighter-attack");
  lw6sys_list_push_back (list, "fighter-defense");
  lw6sys_list_push_back (list, "fighter-new-health");
  lw6sys_list_push_back (list, "side-attack-factor");
  lw6sys_list_push_back (list, "side-defense-factor");
  lw6sys_list_push_back (list, "nb-move-tries");
  lw6sys_list_push_back (list, "nb-attack-tries");
  lw6sys_list_push_back (list, "nb-defense-tries");
  lw6sys_list_push_back (list, "single-army-surface");
  lw6sys_list_push_back (list, "total-armies-surface");
  lw6sys_list_push_back (list, "max-nb-teams");
  lw6sys_list_push_back (list, "x-polarity");
  lw6sys_list_push_back (list, "y-polarity");
  lw6sys_list_push_back (list, "max-zone-size");
  lw6sys_list_push_back (list, "min-map-width");
  lw6sys_list_push_back (list, "max-map-width");
  lw6sys_list_push_back (list, "min-map-height");
  lw6sys_list_push_back (list, "max-map-height");
  lw6sys_list_push_back (list, "min-map-surface");
  lw6sys_list_push_back (list, "max-map-surface");
  lw6sys_list_push_back (list, "bot-wait-between-moves");
  lw6sys_list_push_back (list, "bot-nb-move-steps");
  lw6sys_list_push_back (list, "round-delta");
  lw6sys_list_push_back (list, "max-round-delta");
  lw6sys_list_push_back (list, "max-cursor-pot");
  lw6sys_list_push_back (list, "cursor-pot-init");
  lw6sys_list_push_back (list, "max-cursor-pot-offset");

  return ((*list) != NULL);
}

lw6sys_list_t *
lw6hlp_list_map_options ()
{
  lw6sys_list_t *list = NULL;

  list = lw6sys_list_new (NULL);
  if (list)
    {
      update_map_options (&list);
    }

  return list;
}

static int
update_map_style (lw6sys_list_t ** list)
{
  lw6sys_list_push_back (list, "keep-ratio");
  lw6sys_list_push_back (list, "zoom");
  lw6sys_list_push_back (list, "fighter-scale");
  lw6sys_list_push_back (list, "use-texture");
  lw6sys_list_push_back (list, "background-style");
  lw6sys_list_push_back (list, "hud-style");
  lw6sys_list_push_back (list, "menu-style");
  lw6sys_list_push_back (list, "view-style");
  lw6sys_list_push_back (list, "background-density");
  lw6sys_list_push_back (list, "background-speed");
  lw6sys_list_push_back (list, "colorize-background");
  lw6sys_list_push_back (list, "background-color-auto");
  lw6sys_list_push_back (list, "hud-color-auto");
  lw6sys_list_push_back (list, "menu-color-auto");
  lw6sys_list_push_back (list, "view-color-auto");
  lw6sys_list_push_back (list, "system-color-auto");
  lw6sys_list_push_back (list, "background-color-base-bg");
  lw6sys_list_push_back (list, "background-color-base-fg");
  lw6sys_list_push_back (list, "background-color-alternate-bg");
  lw6sys_list_push_back (list, "background-color-alternate-fg");
  lw6sys_list_push_back (list, "hud-color-frame-bg");
  lw6sys_list_push_back (list, "hud-color-frame-fg");
  lw6sys_list_push_back (list, "hud-color-text-bg");
  lw6sys_list_push_back (list, "hud-color-text-fg");
  lw6sys_list_push_back (list, "menu-color-default-bg");
  lw6sys_list_push_back (list, "menu-color-default-fg");
  lw6sys_list_push_back (list, "menu-color-selected-bg");
  lw6sys_list_push_back (list, "menu-color-selected-fg");
  lw6sys_list_push_back (list, "menu-color-disabled-bg");
  lw6sys_list_push_back (list, "menu-color-disabled-fg");
  lw6sys_list_push_back (list, "view-color-cursor-bg");
  lw6sys_list_push_back (list, "view-color-cursor-fg");
  lw6sys_list_push_back (list, "view-color-map-bg");
  lw6sys_list_push_back (list, "view-color-map-fg");
  lw6sys_list_push_back (list, "system-color-bg");
  lw6sys_list_push_back (list, "system-color-fg");
  lw6sys_list_push_back (list, "team-color-red");
  lw6sys_list_push_back (list, "team-color-green");
  lw6sys_list_push_back (list, "team-color-blue");
  lw6sys_list_push_back (list, "team-color-yellow");
  lw6sys_list_push_back (list, "team-color-cyan");
  lw6sys_list_push_back (list, "team-color-magenta");
  lw6sys_list_push_back (list, "team-color-orange");
  lw6sys_list_push_back (list, "team-color-lightblue");
  lw6sys_list_push_back (list, "team-color-purple");
  lw6sys_list_push_back (list, "team-color-pink");

  return ((*list) != NULL);
}

lw6sys_list_t *
lw6hlp_list_map_style ()
{
  lw6sys_list_t *list = NULL;

  list = lw6sys_list_new (NULL);
  if (list)
    {
      update_map_style (&list);
    }

  return list;
}

static int
update_funcs (lw6sys_list_t ** list)
{
  return ((*list) != NULL);
}

lw6sys_list_t *
lw6hlp_list_funcs ()
{
  lw6sys_list_t *list = NULL;

  list = lw6sys_list_new (NULL);
  if (list)
    {
      update_funcs (&list);
    }

  return list;
}

static int
update_hooks (lw6sys_list_t ** list)
{
  return ((*list) != NULL);
}

lw6sys_list_t *
lw6hlp_list_hooks ()
{
  lw6sys_list_t *list = NULL;

  list = lw6sys_list_new (NULL);
  if (list)
    {
      update_hooks (&list);
    }

  return list;
}

static int
update_tuning (lw6sys_list_t ** list)
{
  lw6sys_list_push_back (list, "log-level");
  lw6sys_list_push_back (list, "max-frames-per-sec");
  lw6sys_list_push_back (list, "max-logic-per-sec");
  lw6sys_list_push_back (list, "max-io-per-sec");
  lw6sys_list_push_back (list, "max-logic-per-display");
  lw6sys_list_push_back (list, "max-io-per-display");
  lw6sys_list_push_back (list, "checkpoint-period");
  lw6sys_list_push_back (list, "loader-sleep");
  lw6sys_list_push_back (list, "memory-bazooka-size");
  lw6sys_list_push_back (list, "memory-bazooka-eraser");

  return ((*list) != NULL);
}

lw6sys_list_t *
lw6hlp_list_tuning ()
{
  lw6sys_list_t *list = NULL;

  list = lw6sys_list_new (NULL);
  if (list)
    {
      update_tuning (&list);
    }

  return list;
}

static int
update_aliases (lw6sys_list_t ** list)
{
  lw6sys_list_push_back (list, "h");
  lw6sys_list_push_back (list, "?");
  lw6sys_list_push_back (list, "v");
  lw6sys_list_push_back (list, "copyleft");
  lw6sys_list_push_back (list, "copy");
  lw6sys_list_push_back (list, "c");

  return ((*list) != NULL);
}

lw6sys_list_t *
lw6hlp_list_aliases ()
{
  lw6sys_list_t *list = NULL;

  list = lw6sys_list_new (NULL);
  if (list)
    {
      update_aliases (&list);
    }

  return list;
}

/**
 * lw6hlp_list
 *
 * Returns a list of all available keywords.
 *
 * Return value: a list containing all the keywords. Strings are not
 *   dynamically allocated, you can't modify them.
 */
lw6sys_list_t *
lw6hlp_list ()
{
  lw6sys_list_t *list = NULL;
  int ok = 0;

  list = lw6sys_list_new (NULL);
  if (list)
    {
      ok = update_quick (&list) && update_doc (&list) && update_show (&list)
	&& update_path (&list) && update_players (&list)
	&& update_input (&list) && update_graphics (&list)
	&& update_sound (&list) && update_network (&list)
	&& update_map (&list) && update_map_options (&list)
	&& update_map_style (&list) && update_funcs (&list)
	&& update_hooks (&list) && update_tuning (&list)
	&& update_aliases (&list);
    }

  if (!ok)
    {
      lw6sys_log (LW6SYS_LOG_WARNING,
		  _("problem generating help keywords list"));
      if (list)
	{
	  lw6sys_list_free (list);
	  list = NULL;
	}
    }

  return list;
}
