/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006, 2007  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.
  

  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#include <stdlib.h>
#include <string.h>
#include <stdio.h>

#include "config.h"
#include "sys.h"
#include "sys-internal.h"

#define THREAD_JOIN_SLEEP 0.1f

static int thread_create_counter = 0;
static int thread_join_counter = 0;

static void
thread_callback (void *thread_handler)
{
  _LW6SYS_THREAD_HANDLER *th;
  th = (_LW6SYS_THREAD_HANDLER *) thread_handler;

  if (th)
    {
      lw6sys_log (LW6SYS_LOG_INFO, "sys", _("begin thread id=%d"), th->id);
      if (th->callback_func)
	{
	  th->callback_func (th->callback_data);
	}
      /* 
       * callback is over, we signal it to the caller, if needed
       */
      th->callback_done = 1;
      while (!th->can_join)
	{
	  /*
	   * Now the caller is supposed to set "can_join"
	   * to allow this thread to actually finish.
	   */
	  lw6sys_sleep (THREAD_JOIN_SLEEP);
	}
      if (th->callback_join)
	{
	  th->callback_join (th->callback_data);
	}
      lw6sys_log (LW6SYS_LOG_INFO, "sys", _("end thread id=%d"), th->id);
      pthread_exit (NULL);
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING, "sys",
		  _("can't call thread_callback on NULL thread_handler"));
    }
}

void *
lw6sys_thread_create (void (*callback_func) (void *callback_data),
		      void (*callback_join) (void *callback_data),
		      void *callback_data, int flag)
{
  _LW6SYS_THREAD_HANDLER *thread_handler = NULL;

  thread_handler =
    (_LW6SYS_THREAD_HANDLER *)
    LW6SYS_CALLOC (sizeof (_LW6SYS_THREAD_HANDLER));
  if (thread_handler)
    {
      // callback_done & the rest already set to 0
      thread_handler->id = (++thread_create_counter);
      lw6sys_log (LW6SYS_LOG_INFO, "sys", _("creating thread id=%d"),
		  thread_handler->id);
      thread_handler->flag = flag;
      thread_handler->callback_func = callback_func;
      thread_handler->callback_join = callback_join;
      thread_handler->callback_data = callback_data;
      if (!pthread_create (&(thread_handler->thread), NULL,
			   (void *) thread_callback, (void *) thread_handler))
	{
	  // OK, thread created
	}
      else
	{
	  lw6sys_log (LW6SYS_LOG_WARNING, "sys", _("can't start thread"));
	  /*
	   * Better do a thread_create_counter-- here, this way we
	   * can use a "very probably" atomic operation to increase
	   * it *before* thread creation, ensuring the id is properly
	   * set. Not perfect, but better and simpler than many other
	   * "solutions".
	   */
	  thread_create_counter--;
	  LW6SYS_FREE (thread_handler);
	  thread_handler = NULL;
	}
    }

  return (void *) thread_handler;
}

int
lw6sys_thread_is_callback_done (void *thread_handler)
{
  int ret = 0;

  if (thread_handler)
    {
      _LW6SYS_THREAD_HANDLER *th;
      th = (_LW6SYS_THREAD_HANDLER *) thread_handler;

      ret = th->callback_done;
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING, "sys",
		  _("can't call is_callback_done on NULL thread_handler"));
    }

  return ret;
}

int
lw6sys_thread_get_id (void *thread_handler)
{
  int ret = 0;

  if (thread_handler)
    {
      _LW6SYS_THREAD_HANDLER *th;
      th = (_LW6SYS_THREAD_HANDLER *) thread_handler;

      ret = th->id;
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING, "sys",
		  _("can't call get_id on NULL thread_handler"));
    }

  return ret;
}

void *
lw6sys_thread_get_data (void *thread_handler)
{
  void *ret = NULL;

  if (thread_handler)
    {
      _LW6SYS_THREAD_HANDLER *th;
      th = (_LW6SYS_THREAD_HANDLER *) thread_handler;

      ret = th->callback_data;
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING, "sys",
		  _("can't call get_data on NULL thread_handler"));
    }

  return ret;
}

int
lw6sys_thread_get_flag (void *thread_handler)
{
  int ret = 0;

  if (thread_handler)
    {
      _LW6SYS_THREAD_HANDLER *th;
      th = (_LW6SYS_THREAD_HANDLER *) thread_handler;

      ret = th->flag;
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING, "sys",
		  _("can't call get_flag on NULL thread_handler"));
    }

  return ret;
}

void
lw6sys_thread_join (void *thread_handler)
{
  if (thread_handler)
    {
      _LW6SYS_THREAD_HANDLER *th;
      th = (_LW6SYS_THREAD_HANDLER *) thread_handler;

      lw6sys_log (LW6SYS_LOG_INFO, "sys", _("joining thread id=%d"), th->id);

      th->can_join = 1;

      if (!pthread_join (th->thread, NULL))
	{
	  LW6SYS_FREE (th);
	  thread_join_counter++;
	}
      else
	{
	  lw6sys_log (LW6SYS_LOG_WARNING, "sys", _("can't join thread id=%d"),
		      th->id);
	}
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING, "sys",
		  _("can't call join on NULL thread_handler"));
    }
}

int
lw6sys_get_thread_create_count ()
{
  return thread_create_counter;
}

int
lw6sys_get_thread_join_count ()
{
  return thread_join_counter;
}

int
lw6sys_check_thread_count ()
{
  int ret = 1;

  if (thread_create_counter != thread_join_counter)
    {
      ret = 0;
      lw6sys_log (LW6SYS_LOG_WARNING, "sys",
		  _
		  ("possible thread problem, %d threads have been started, but only %d threads have been joined"),
		  thread_create_counter, thread_join_counter);
    }

  return ret;
}
