/*
Liquid War 6 is a unique multiplayer wargame.
Copyright (C)  2005, 2006, 2007  Christian Mauduit <ufoot@ufoot.org>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.


Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
Contact author        : ufoot@ufoot.org
*/

#include <stdio.h>
#include <locale.h>
#include <string.h>

#include "config.h"
#include "sys.h"

#define CHECKSUM_INT32_DATA 1234
#define CHECKSUM_INT32_RESULT 0x5d3d512e
#define COLOR_ASCII "#12345678"
#define COLOR_8_R 0x12
#define COLOR_8_G 0x34
#define COLOR_8_B 0x56
#define COLOR_8_A 0x78
#define FILE_EXISTS_FILE1 "/etc/hosts"
#define FILE_EXISTS_FILE2 "silver_bullet.txt"
#define DIR_EXISTS_DIR1 "/etc"
#define DIR_EXISTS_DIR2 "/silver/bullet"
#define CREATE_DIR_DIR "/tmp/foo/bar"
#define CREATE_DIR_FILE "void.txt"
#define MALLOC_SIZE 1048576
#define THREAD_N 5
#define THREAD_SLEEP_MAIN 0.66f
#define THREAD_SLEEP_CALLBACK 0.2f
#define THREAD_TEXT "foo bar"
#define SLEEP_TIME 3
#define BLANK_STR "\t "

#define FUNCTION_BEGIN { lw6sys_log(LW6SYS_LOG_DEBUG,"sys-test",_("running tests in \"%s\""),__FUNCTION__); }
#define FUNCTION_END   if (ret) { lw6sys_log(LW6SYS_LOG_DEBUG,"sys-test",_("tests in \"%s\" OK"),__FUNCTION__); } else { lw6sys_log(LW6SYS_LOG_WARNING,"sys-test",_("tests in \"%s\" failed"),__FUNCTION__); }

/*
 * Used to test lw6sys_assoc_map.
 */
static void
assoc_map_func (void *func_data, char *key, void *value)
{
  int *ret = (int *) func_data;
  int *i = (int *) value;

  lw6sys_log (LW6SYS_LOG_DEBUG, "sys-test",
	      _("assoc item ret=%d, key=\"%s\", value=%d"), *ret, key, *i);
}

/*
 * Testing functions in assoc.c
 */
static int
test_assoc ()
{
  int ret = 1;
  LW6SYS_ASSOC *assoc;
  FUNCTION_BEGIN;

  {
    lw6sys_log (LW6SYS_LOG_DEBUG, "sys-test", _("new/delete on assoc"));
    assoc = lw6sys_assoc_new (NULL);
    lw6sys_assoc_free (assoc);
  }

  {
    int a = 3, b = 5, c = 7;

    lw6sys_log (LW6SYS_LOG_DEBUG, "sys-test",
		_("set/unset/has_key/get/map on int assoc"));
    assoc = lw6sys_assoc_new (NULL);
    lw6sys_assoc_set (&assoc, "a", (void *) &a);
    lw6sys_assoc_set (&assoc, "b", (void *) &b);
    lw6sys_assoc_set (&assoc, "c", (void *) &a);
    lw6sys_assoc_set (&assoc, "c", (void *) &c);
    lw6sys_assoc_unset (assoc, "b");
    lw6sys_assoc_map (assoc, assoc_map_func, &ret);
    ret = ret && lw6sys_assoc_has_key (assoc, "a");
    ret = ret && !lw6sys_assoc_has_key (assoc, "b");
    ret = ret && !lw6sys_assoc_has_key (assoc, "this key does not exist");
    ret = ret && (*((int *) lw6sys_assoc_get (assoc, "a")) == 3);
    ret = ret && ((int *) lw6sys_assoc_get (assoc, "b") == NULL);
    ret = ret && (*((int *) lw6sys_assoc_get (assoc, "c")) == 7);
    lw6sys_assoc_free (assoc);
  }

  {
    char *str1, *str2, *str3;

    lw6sys_log (LW6SYS_LOG_DEBUG, "sys-test",
		_("testing free_func callback on string assoc"));
    assoc = lw6sys_assoc_new (&lw6sys_free_callback);
    str1 = LW6SYS_MALLOC (5);
    strncpy (str1, "str1", 5);
    lw6sys_assoc_set (&assoc, "1", (void *) str1);
    str2 = LW6SYS_MALLOC (5);
    strncpy (str2, "str2", 5);
    lw6sys_assoc_set (&assoc, "2", (void *) str2);
    str3 = LW6SYS_MALLOC (5);
    strncpy (str3, "str3", 5);
    lw6sys_assoc_set (&assoc, "3", (void *) str3);
    lw6sys_assoc_unset (assoc, "1");
    lw6sys_assoc_unset (assoc, "3");
    ret = ret && !lw6sys_assoc_has_key (assoc, "1");
    ret = ret && lw6sys_assoc_has_key (assoc, "2");
    ret = ret && !lw6sys_assoc_has_key (assoc, "3");
    ret = ret && strcmp ((char *) lw6sys_assoc_get (assoc, "2"), "str2") == 0;
    lw6sys_assoc_free (assoc);
  }

  {
    int a = 3, b = 5;
    LW6SYS_LIST *keys;

    lw6sys_log (LW6SYS_LOG_DEBUG, "sys-test", _("testing keys on int assoc"));
    assoc = lw6sys_assoc_new (NULL);
    lw6sys_assoc_set (&assoc, "a", (void *) &a);
    lw6sys_assoc_set (&assoc, "b", (void *) &b);
    lw6sys_assoc_set (&assoc, "b", (void *) &b);
    keys = lw6sys_assoc_keys (assoc);
    lw6sys_assoc_free (assoc);
    ret = ret && lw6sys_list_length (keys) == 2;
    lw6sys_list_free (keys);
  }

  FUNCTION_END;
  return ret;
}

/*
 * Testing functions in checksum.c
 */
static int
test_checksum ()
{
  int ret = 1;
  FUNCTION_BEGIN;

  {
    LW6SYS_UINT32 checksum = 0;

    lw6sys_checksum_update_int32 (&checksum, CHECKSUM_INT32_DATA);
    lw6sys_log (LW6SYS_LOG_DEBUG, "sys-test",
		_("checksum for %d is %08x (should be %08x)"),
		CHECKSUM_INT32_DATA, checksum, CHECKSUM_INT32_RESULT);
    if (checksum != CHECKSUM_INT32_RESULT)
      {
	ret = 0;
      }
  }

  FUNCTION_END;
  return ret;
}

/*
 * Testing functions in color.c
 */
static int
test_color ()
{
  int ret = 1;
  FUNCTION_BEGIN;

  {
    LW6SYS_COLOR_8 color_8;

    lw6sys_log (LW6SYS_LOG_DEBUG, "sys-test",
		_("interpreting color \"%s\""), COLOR_ASCII);
    color_8 = lw6sys_color_a_to_8 (COLOR_ASCII);
    if (color_8.r != COLOR_8_R || color_8.g != COLOR_8_G
	|| color_8.b != COLOR_8_B || color_8.a != COLOR_8_A)
      {
	ret = 0;
      }
  }

  FUNCTION_END;
  return ret;
}

/*
 * Testing functions in i18n.c
 */
static int
test_i18n ()
{
  int ret = 1;
  FUNCTION_BEGIN;

  // Todo...

  FUNCTION_END;
  return ret;
}

/*
 * Used to test lw6sys_list_map.
 */
static void
list_map_func (void *func_data, void *data)
{
  int *ret = (int *) func_data;
  int *i = (int *) data;

  lw6sys_log (LW6SYS_LOG_DEBUG, "sys-test", _("list item ret=%d, data=%d"),
	      *ret, *i);
}

/*
 * Testing functions in list.c
 */
static int
test_list ()
{
  int ret = 1;
  LW6SYS_LIST *list;
  FUNCTION_BEGIN;

  {
    lw6sys_log (LW6SYS_LOG_DEBUG, "sys-test",
		_("new/delete/is_empty on list"));
    list = lw6sys_list_new (NULL);
    ret = ret && lw6sys_list_is_empty (list);
    lw6sys_list_free (list);
  }

  {
    int a = 3, b = 5, c = 7;

    lw6sys_log (LW6SYS_LOG_DEBUG, "sys-test",
		_("push/pop/length/map on int list"));
    list = lw6sys_list_new (NULL);
    lw6sys_list_push (&list, (void *) &a);
    lw6sys_list_push (&list, (void *) &b);
    lw6sys_list_push (&list, (void *) &c);
    lw6sys_list_map (list, list_map_func, &ret);
    ret = ret && lw6sys_list_length (list) == 3;
    ret = ret && (*((int *) lw6sys_list_pop (&list)) == 7);
    ret = ret && (*((int *) lw6sys_list_pop (&list)) == 5);
    ret = ret && (*((int *) lw6sys_list_pop (&list)) == 3);
    lw6sys_list_free (list);
  }

  {
    char *str1, *str2, *str2a;

    lw6sys_log (LW6SYS_LOG_DEBUG, "sys-test",
		_("testing free_func callback on string list"));
    list = lw6sys_list_new (&lw6sys_free_callback);
    str1 = LW6SYS_MALLOC (5);
    strncpy (str1, "str1", 5);
    lw6sys_list_push (&list, (void *) str1);
    str2 = LW6SYS_MALLOC (5);
    strncpy (str2, "str2", 5);
    lw6sys_list_push (&list, (void *) str2);
    str2a = ((char *) lw6sys_list_pop (&list));
    LW6SYS_FREE (str2a);
    ret = ret && lw6sys_list_length (list) == 1;
    lw6sys_list_free (list);
  }

  FUNCTION_END;
  return ret;
}

/*
 * Testing functions in log.c
 */
static int
test_log ()
{
  int ret = 1;
  FUNCTION_BEGIN;

  lw6sys_log (LW6SYS_LOG_DEBUG, "sys-test", _("testing log_debug %s=%d"),
	      "toto", 0);
  lw6sys_log (LW6SYS_LOG_INFO, "sys-test", _("testing log_info %s=%d"),
	      "toto", 0);
  lw6sys_log (LW6SYS_LOG_WARNING, "sys-test", _("testing log_warning %s=%d"),
	      "toto", 0);
  lw6sys_log (LW6SYS_LOG_ERROR, "sys-test", _("testing log_error %s=%d"),
	      "toto", 0);

  FUNCTION_END;
  return ret;
}

/*
 * Testing functions in mem.c
 */
static int
test_mem ()
{
  int ret = 1;
  char *ptr = NULL;
  FUNCTION_BEGIN;

  lw6sys_log (LW6SYS_LOG_DEBUG, "sys-test",
	      _("testing MALLOC/FREE (%d bytes)"), MALLOC_SIZE);
  ptr = (char *) LW6SYS_MALLOC (MALLOC_SIZE * sizeof (char));
  ret = ret && (ptr != NULL);
  LW6SYS_FREE (ptr);

  FUNCTION_END;
  return ret;
}

/*
 * Testing functions in mutex.c
 */
static int
test_mutex ()
{
  int ret = 0;
  void *mutex = NULL;
  FUNCTION_BEGIN;

  {
    lw6sys_log (LW6SYS_LOG_DEBUG, "sys-test", _("testing mutex functions"));
    mutex = lw6sys_mutex_create ();
    if (mutex)
      {
	if (lw6sys_mutex_lock (mutex))
	  {
	    if (lw6sys_mutex_trylock (mutex))
	      {
		lw6sys_log (LW6SYS_LOG_WARNING, "sys-test",
			    _
			    ("strange, trylock on locked mutex should return 0"));
	      }
	    else
	      {
		if (lw6sys_mutex_unlock (mutex))
		  {
		    ret = 1;
		  }
	      }
	  }
	lw6sys_mutex_destroy (mutex);
      }
  }

  FUNCTION_END;
  return ret;
}

/*
 * Helper function to test file_exists.
 */
static int
test_path_file_exists (char *filename)
{
  int ret = 0;
  FUNCTION_BEGIN;

  if (lw6sys_file_exists (filename))
    {
      ret = 1;
      lw6sys_log (LW6SYS_LOG_DEBUG, "sys-test", _("file \"%s\" exists"),
		  filename);
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_DEBUG, "sys-test",
		  _("file \"%s\" does not exist"), filename);
    }

  FUNCTION_END;
  return ret;
}

/*
 * Helper function to test dir_exists.
 */
static int
test_path_dir_exists (char *dirname)
{
  int ret = 0;
  FUNCTION_BEGIN;

  if (lw6sys_dir_exists (dirname))
    {
      ret = 1;
      lw6sys_log (LW6SYS_LOG_DEBUG, "sys-test", _("dir \"%s\" exists"),
		  dirname);
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_DEBUG, "sys-test",
		  _("dir \"%s\" does not exist"), dirname);
    }

  FUNCTION_END;
  return ret;
}

/*
 * Testing functions in path.c
 */
static int
test_path ()
{
  int ret = 1;
  FUNCTION_BEGIN;

  ret = ret &&
    test_path_file_exists (FILE_EXISTS_FILE1) &&
    !test_path_file_exists (FILE_EXISTS_FILE2) &&
    test_path_dir_exists (DIR_EXISTS_DIR1) &&
    !test_path_dir_exists (DIR_EXISTS_DIR2);

  lw6sys_log (LW6SYS_LOG_DEBUG, "sys-test",
	      _("create_dir_for_file on \"%s/%s\""), CREATE_DIR_DIR,
	      CREATE_DIR_FILE);
  ret = ret && lw6sys_create_dir_for_file (CREATE_DIR_DIR "/" CREATE_DIR_FILE)
    && lw6sys_dir_exists (CREATE_DIR_DIR);

  FUNCTION_END;
  return ret;
}

/*
 * Testing functions in str.c
 */
static int
test_str ()
{
  int ret = 1;
  FUNCTION_BEGIN;

  lw6sys_log (LW6SYS_LOG_DEBUG, "sys-test", _("testing new_sprintf %s %d %f"),
	      "foo", 1, 3.14159);
  lw6sys_log (LW6SYS_LOG_DEBUG, "sys-test", _("testing is_blank on \"%s\""),
	      BLANK_STR);
  ret = ret && lw6sys_str_is_blank (BLANK_STR);

  FUNCTION_END;
  return ret;
}

static void
thread_func (void *callback_data)
{
  int i = 0;
  char *text = NULL;

  text = (char *) callback_data;
  lw6sys_log (LW6SYS_LOG_DEBUG, "sys-test", _("thread_callback text=\"%s\""),
	      text);
  for (i = 0; i < THREAD_N; ++i)
    {
      lw6sys_log (LW6SYS_LOG_DEBUG, "sys-test", _("thread_callback step %d"),
		  i + 1);
      lw6sys_sleep (THREAD_SLEEP_CALLBACK);
    }
}

static void
thread_join (void *callback_data)
{
  /*
   * A more complex example would process more complex
   * freeing of resources here.
   */
  LW6SYS_FREE (callback_data);
}

/*
 * Testing functions in thread.c
 */
static int
test_thread ()
{
  int ret = 1;
  FUNCTION_BEGIN;

  {
    void *thread_handler = NULL;
    char *text = lw6sys_str_copy (THREAD_TEXT);
    int i = 0;

    ret = 0;
    if (text)
      {
	thread_handler =
	  lw6sys_thread_create (&thread_func, &thread_join, (void *) text, 0);
	if (thread_handler)
	  {
	    for (i = 0; i < THREAD_N; ++i)
	      {
		lw6sys_log (LW6SYS_LOG_DEBUG, "sys-test",
			    _("thread_main step %d, callback_done=%d"), i + 1,
			    lw6sys_thread_is_callback_done (thread_handler));
		lw6sys_sleep (THREAD_SLEEP_MAIN);
	      }

	    lw6sys_thread_join (thread_handler);
	    ret = 1;
	  }
      }
  }

  FUNCTION_END;
  return ret;
}

/*
 * Testing functions in time.c
 */
static int
test_time ()
{
  int ret = 1;
  FUNCTION_BEGIN;

  lw6sys_log (LW6SYS_LOG_DEBUG, "sys-test", _("timestamp %lld"),
	      lw6sys_timestamp ());
  lw6sys_log (LW6SYS_LOG_DEBUG, "sys-test", _("sleep %d seconds"),
	      SLEEP_TIME);
  lw6sys_sleep (SLEEP_TIME);

  FUNCTION_END;
  return ret;
}

/*
 * Launch tests
 */
int
main (int argc, char *argv[])
{
  int ret = 1;

  setlocale (LC_ALL, "");
  bindtextdomain (PACKAGE_TARNAME, LW6_LOCALE_DIR);
  textdomain (PACKAGE_TARNAME);

  ret = test_assoc () && test_checksum () && test_color () &&
    test_i18n () &&
    test_list () && test_log () && test_mem () && test_mutex ()
    && test_path () && test_str () && test_thread () && test_time ();

  ret = lw6sys_check_malloc_free_count () && ret;
  ret = lw6sys_check_thread_count () && ret;

  return (!ret);
}
