/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006, 2007  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.
  

  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#include <stdlib.h>
#include <string.h>

#include "config.h"
#include "sys.h"

/*
 * Creates an empty assoc. There's a difference between NULL and an
 * empty assoc. The empty assoc would (in Scheme) be '() whereas
 * NULL corresponds to undefined "is not a assoc and will generate
 * errors if you ever call assoc functions on it".
 */
LW6SYS_ASSOC *
lw6sys_assoc_new (void (*free_func) (void *value))
{
  LW6SYS_ASSOC *ret = NULL;

  ret = LW6SYS_MALLOC (sizeof (LW6SYS_ASSOC));
  if (ret)
    {
      memset (ret, 0, sizeof (LW6SYS_ASSOC));
      ret->free_func = free_func;
    }

  return ret;
}

/*
 * Delete a assoc, this will cascade delete all the following
 * items in the assoc.
 */
void
lw6sys_assoc_free (LW6SYS_ASSOC * assoc)
{
  if (assoc)
    {
      /*
       * Keep a copy of next_item for we are about to
       * free the pointer to it.
       */
      LW6SYS_ASSOC *next_item = (void *) assoc->next_item;

      if (assoc->key)
	{
	  LW6SYS_FREE (assoc->key);
	}

      /*
       * It's legal to have free_func or value set to NULL,
       * we simply need to avoid the core dump, but one can
       * legitimately not need any peculiar free function,
       * or desire to store a NULL content in a valid assoc.
       */
      if (assoc->free_func && assoc->value)
	{
	  assoc->free_func (assoc->value);
	}

      LW6SYS_FREE (assoc);

      if (next_item)
	{
	  /*
	   * This should be the last call of the function.
	   * Hopefully the compiler will find this out and 
	   * optimize and *not* generate hudge stacks with
	   * return addresses which are of no use. At least
	   * the compiler *could* do it 8-) Recursion recursion...
	   */
	  lw6sys_assoc_free ((LW6SYS_ASSOC *) next_item);
	}
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING, "sys", _("trying to free NULL assoc"));
    }
}

/*
 * Returns true if the key exists.
 */
int
lw6sys_assoc_has_key (LW6SYS_ASSOC * assoc, char *key)
{
  int exists = 0;

  if (assoc)
    {
      while (assoc)
	{
	  if (assoc->key && strcmp (assoc->key, key) == 0)
	    {
	      exists = 1;
	    }
	  assoc = assoc->next_item;
	}
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING, "sys",
		  _("calling has_key on NULL assoc"));
    }

  return exists;
}

/*
 * Returns the value associated to a given key, NULL if the
 * key does not exist.
 */
void *
lw6sys_assoc_get (LW6SYS_ASSOC * assoc, char *key)
{
  void *value = NULL;

  if (assoc)
    {
      while (assoc)
	{
	  if (assoc->key && strcmp (assoc->key, key) == 0)
	    {
	      value = assoc->value;
	    }
	  assoc = assoc->next_item;
	}
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING, "sys", _("calling get on NULL assoc"));
    }

  return value;
}

/*
 * Sets a value in an associative array. The key pointer need
 * not be persistent, it can be freed after affectation. In
 * fact a new string will be created internally. This is not
 * true for the value, it's hard to find way to copy "any object".
 * So if you want an associative array of strings, key can
 * disappear after calling this function, but not value. The
 * function passed as free_func when creating the assoc will
 * be used to free stuff whenever needed (unset or free).
 */
void
lw6sys_assoc_set (LW6SYS_ASSOC ** assoc, char *key, void *value)
{
  int exists = 0;

  if (assoc && *assoc)
    {
      LW6SYS_ASSOC *search = *assoc;
      while (search)
	{
	  if (search->key && strcmp (search->key, key) == 0)
	    {
	      exists = 1;
	      if (search->free_func && search->value)
		{
		  search->free_func (search->value);
		}
	      search->value = value;
	    }
	  search = search->next_item;
	}
      if (!exists)
	{
	  LW6SYS_ASSOC *new_assoc = NULL;

	  new_assoc = LW6SYS_MALLOC (sizeof (LW6SYS_ASSOC));
	  if (new_assoc)
	    {
	      new_assoc->next_item = (void *) (*assoc);
	      new_assoc->key = lw6sys_str_copy (key);
	      new_assoc->value = value;
	      new_assoc->free_func = (*assoc)->free_func;
	      (*assoc) = new_assoc;
	    }
	}
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING, "sys", _("calling set on NULL assoc"));
    }
}

/*
 * Clears an entry in an associative array.
 */
void
lw6sys_assoc_unset (LW6SYS_ASSOC * assoc, char *key)
{
  if (assoc)
    {
      while (assoc)
	{
	  if (assoc->key && strcmp (assoc->key, key) == 0)
	    {
	      if (assoc->key)
		{
		  LW6SYS_FREE (assoc->key);
		}
	      if (assoc->free_func && assoc->value)
		{
		  assoc->free_func (assoc->value);
		}
	      if (assoc->next_item)
		{
		  LW6SYS_ASSOC *to_free;

		  to_free = assoc->next_item;
		  memcpy (assoc, assoc->next_item, sizeof (LW6SYS_ASSOC));
		  LW6SYS_FREE (to_free);
		}
	      else
		{
		  memset (assoc, 0, sizeof (LW6SYS_ASSOC));
		}
	    }
	  assoc = assoc->next_item;
	}
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING, "sys",
		  _("calling unset on NULL assoc"));
    }
}

/*
 * Returns a list containing all the keys of the assoc. The
 * list must be free with lw6sys_list_free by the caller.
 * This lisst copies all the keys of the assoc, so it is
 * safe to use it once the assoc is deleted. However the
 * keys will of course be of little interest in this case.
 * But the program won't segfault.
 */
LW6SYS_LIST *
lw6sys_assoc_keys (LW6SYS_ASSOC * assoc)
{
  LW6SYS_LIST *keys = NULL;

  if (assoc)
    {
      keys = lw6sys_list_new (lw6sys_free_callback);
      if (keys)
	{
	  while (assoc)
	    {
	      if (assoc->key)
		{
		  lw6sys_list_push (&keys, lw6sys_str_copy (assoc->key));
		}
	      assoc = assoc->next_item;
	    }
	}
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING, "sys", _("calling keys on NULL assoc"));
    }

  return keys;
}

/*
 * Executes a function on all assoc items.
 * The func_data parameter allows you to pass extra values to
 * the function, such as a file handler or any variable which
 * can not be inferred from list item values, and you of course
 * do not want to make global...
 */
void
lw6sys_assoc_map (LW6SYS_ASSOC * assoc,
		  void (*func) (void *func_data, char *key, void *value),
		  void *func_data)
{
  if (assoc)
    {
      while (assoc)
	{
	  /*
	   * To check if the assoc item is defined we call
	   * assoc->next_item which is false only at the end of
	   * the list. Other indicators such as key are "not as
	   * good".
	   */
	  if (assoc->next_item)
	    {
	      if (assoc->key)
		{
		  func (func_data, assoc->key, assoc->value);
		}
	      else
		{
		  lw6sys_log (LW6SYS_LOG_WARNING, "sys",
			      _("assoc has a NULL key"));
		}
	    }
	  assoc = assoc->next_item;
	}
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING, "sys", _("calling map on NULL assoc"));
    }
}
