/*
Liquid War 6 is a unique multiplayer wargame.
Copyright (C)  2005, 2006, 2007  Christian Mauduit <ufoot@ufoot.org>

This program is free software; you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.


Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
Contact author        : ufoot@ufoot.org
*/

#include <stdlib.h>
#include <locale.h>

#include "config.h"
#include "ker.h"

#define TEST_MAP_WIDTH 320
#define TEST_MAP_HEIGHT 240
#define TEST_NB_ROUNDS 100
#define TEST_GAME_STRUCT_CHECKSUM 0xae0f4e38
#define TEST_GAME_STATE_CHECKSUM 0xaa80f234
#define TEST_GAME_STATE_POPULATE_CHECKSUM 0x08f679d2
#define TEST_GAME_STATE_ALGORITHM_CHECKSUM 0xc25e5a7c

#define FUNCTION_BEGIN { lw6sys_log(LW6SYS_LOG_DEBUG,"ker-test",_("running tests in \"%s\""),__FUNCTION__); }
#define FUNCTION_END   if (ret) { lw6sys_log(LW6SYS_LOG_DEBUG,"ker-test",_("tests in \"%s\" OK"),__FUNCTION__); } else { lw6sys_log(LW6SYS_LOG_WARNING,"ker-test",_("tests in \"%s\" failed"),__FUNCTION__); }

static int
test_struct ()
{
  int ret = 1;
  FUNCTION_BEGIN;

  {
    LW6MAP_MAP *map;
    LW6KER_GAME_STRUCT *game_struct;
    char *repr;
    int checksum;

    ret = 0;
    map = lw6ker_map_new_rectangle (TEST_MAP_WIDTH, TEST_MAP_HEIGHT);
    if (map)
      {
	game_struct = lw6ker_game_struct_new (map);
	if (game_struct)
	  {
	    repr = lw6ker_game_struct_repr (game_struct);
	    if (repr)
	      {
		lw6sys_log (LW6SYS_LOG_DEBUG, "ker-test",
			    _("game_struct repr is \"%s\""), repr);
		LW6SYS_FREE (repr);
	      }
	    checksum = lw6ker_game_struct_checksum (game_struct);
	    lw6sys_log (LW6SYS_LOG_DEBUG, "ker-test",
			_
			("game struct checksum is %08x and should be %08x"),
			checksum, TEST_GAME_STRUCT_CHECKSUM);
	    lw6ker_game_struct_free (game_struct);
	    ret = (checksum == TEST_GAME_STRUCT_CHECKSUM);
	  }
	lw6ker_map_free_rectangle (map);
      }
  }

  FUNCTION_END;
  return ret;
}

static int
test_state ()
{
  int ret = 1;
  FUNCTION_BEGIN;

  {
    LW6MAP_MAP *map;
    LW6KER_GAME_STRUCT *game_struct;
    LW6KER_GAME_STATE *game_state;
    char *repr;
    int checksum;

    ret = 0;
    map = lw6ker_map_new_rectangle (TEST_MAP_WIDTH, TEST_MAP_HEIGHT);
    if (map)
      {
	game_struct = lw6ker_game_struct_new (map);
	if (game_struct)
	  {
	    repr = lw6ker_game_struct_repr (game_struct);
	    if (repr)
	      {
		lw6sys_log (LW6SYS_LOG_DEBUG, "ker-test",
			    _("game_struct repr is \"%s\""), repr);
		LW6SYS_FREE (repr);
	      }

	    game_state = lw6ker_game_state_new (game_struct);
	    if (game_state)
	      {
		repr = lw6ker_game_state_repr (game_state);
		if (repr)
		  {
		    lw6sys_log (LW6SYS_LOG_DEBUG, "ker-test",
				_("game_state repr is \"%s\""), repr);
		    LW6SYS_FREE (repr);
		  }
		lw6ker_map_state_print_debug (&(game_state->map));
		checksum = lw6ker_game_state_checksum (game_state);
		lw6sys_log (LW6SYS_LOG_DEBUG, "ker-test",
			    _
			    ("game state checksum is %08x and should be %08x"),
			    checksum, TEST_GAME_STATE_CHECKSUM);
		ret = lw6ker_map_state_sanity_check (&(game_state->map))
		  && (checksum == TEST_GAME_STATE_CHECKSUM);
		lw6ker_game_state_free (game_state);
	      }
	    lw6ker_game_struct_free (game_struct);
	  }
	lw6ker_map_free_rectangle (map);
      }
  }

  FUNCTION_END;
  return ret;
}

static int
test_population ()
{
  int ret = 1;
  FUNCTION_BEGIN;

  {
    LW6MAP_MAP *map;
    LW6KER_GAME_STRUCT *game_struct;
    LW6KER_GAME_STATE *game_state;
    char *repr;
    int checksum;

    ret = 0;
    map = lw6ker_map_new_rectangle (TEST_MAP_WIDTH, TEST_MAP_HEIGHT);
    if (map)
      {
	game_struct = lw6ker_game_struct_new (map);
	if (game_struct)
	  {
	    repr = lw6ker_game_struct_repr (game_struct);
	    if (repr)
	      {
		lw6sys_log (LW6SYS_LOG_DEBUG, "ker-test",
			    _("game_struct repr is \"%s\""), repr);
		LW6SYS_FREE (repr);
	      }

	    game_state = lw6ker_game_state_new (game_struct);
	    if (game_state)
	      {
		repr = lw6ker_game_state_repr (game_state);
		if (repr)
		  {
		    lw6sys_log (LW6SYS_LOG_DEBUG, "ker-test",
				_("game_state repr is \"%s\""), repr);
		    LW6SYS_FREE (repr);
		  }
		lw6ker_map_state_print_debug (&(game_state->map));
		lw6ker_game_state_add_team (game_state, 0, 1);
		lw6ker_game_state_add_team (game_state, 1, 1);
		lw6ker_game_state_add_team (game_state, 2, 1);
		lw6ker_game_state_add_team (game_state, 3, 1);
		lw6ker_game_state_add_team (game_state, 4, 1);
		lw6ker_game_state_add_team (game_state, 5, 1);
		lw6ker_game_state_add_team (game_state, 6, 1);
		lw6ker_game_state_add_team (game_state, 7, 1);
		lw6ker_map_state_print_debug (&(game_state->map));
		lw6ker_game_state_remove_team (game_state, 2);
		lw6ker_game_state_remove_team (game_state, 3);
		lw6ker_map_state_print_debug (&(game_state->map));
		checksum = lw6ker_game_state_checksum (game_state);
		lw6sys_log (LW6SYS_LOG_DEBUG, "ker-test",
			    _
			    ("game state checksum is %08x and should be %08x"),
			    checksum, TEST_GAME_STATE_POPULATE_CHECKSUM);
		ret = lw6ker_map_state_sanity_check (&(game_state->map))
		  && (checksum == TEST_GAME_STATE_POPULATE_CHECKSUM);
		lw6ker_game_state_free (game_state);
	      }
	    lw6ker_game_struct_free (game_struct);
	  }
	lw6ker_map_free_rectangle (map);
      }
  }

  FUNCTION_END;
  return ret;
}

static int
test_algorithm ()
{
  int ret = 1;
  FUNCTION_BEGIN;

  {
    LW6MAP_MAP *map;
    LW6KER_GAME_STRUCT *game_struct;
    LW6KER_GAME_STATE *game_state;
    char *repr;
    int checksum;
    int i;

    ret = 0;
    map = lw6ker_map_new_rectangle (TEST_MAP_WIDTH, TEST_MAP_HEIGHT);
    if (map)
      {
	game_struct = lw6ker_game_struct_new (map);
	if (game_struct)
	  {
	    repr = lw6ker_game_struct_repr (game_struct);
	    if (repr)
	      {
		lw6sys_log (LW6SYS_LOG_DEBUG, "ker-test",
			    _("game_struct repr is \"%s\""), repr);
		LW6SYS_FREE (repr);
	      }

	    game_state = lw6ker_game_state_new (game_struct);
	    if (game_state)
	      {
		repr = lw6ker_game_state_repr (game_state);
		if (repr)
		  {
		    lw6sys_log (LW6SYS_LOG_DEBUG, "ker-test",
				_("game_state repr is \"%s\""), repr);
		    LW6SYS_FREE (repr);
		  }
		lw6ker_game_state_add_team (game_state, 0, 1);
		lw6ker_game_state_add_team (game_state, 1, 1);
		lw6ker_game_state_add_team (game_state, 2, 1);
		lw6ker_map_state_print_debug (&(game_state->map));
		for (i = 0; i < TEST_NB_ROUNDS; ++i)
		  {
		    lw6ker_game_state_do_round (game_state);
		    lw6sys_log (LW6SYS_LOG_DEBUG, "ker-test",
				_("round %d, game_state checksum=%08x"),
				game_state->rounds,
				lw6ker_game_state_checksum (game_state));
		  }
		checksum = lw6ker_game_state_checksum (game_state);
		lw6sys_log (LW6SYS_LOG_DEBUG, "ker-test",
			    _
			    ("game state checksum is %08x and should be %08x"),
			    checksum, TEST_GAME_STATE_ALGORITHM_CHECKSUM);
		ret = lw6ker_map_state_sanity_check (&(game_state->map))
		  && (checksum == TEST_GAME_STATE_ALGORITHM_CHECKSUM);
		lw6ker_game_state_free (game_state);
	      }
	    lw6ker_game_struct_free (game_struct);
	  }
	lw6ker_map_free_rectangle (map);
      }
  }

  FUNCTION_END;
  return ret;
}

int
main (int argc, char *argv[])
{
  int ret = 1;

  setlocale (LC_ALL, "");
  bindtextdomain (PACKAGE_TARNAME, LW6_LOCALE_DIR);
  textdomain (PACKAGE_TARNAME);

  ret = test_struct () && test_state () && test_population ()
    && test_algorithm ();

  ret = lw6sys_check_malloc_free_count () && ret;
  ret = lw6sys_check_thread_count () && ret;

  return (!ret);
}
