/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006, 2007  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation, either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program.  If not, see <http://www.gnu.org/licenses/>.
  

  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#include <stdlib.h>
#include <string.h>
#include <locale.h>

#include <expat.h>

#include "config.h"
#include "cfg-internal.h"

void
lw6cfg_read_xml_int (char *xml_key, char *xml_value, char *target_key,
		     int *value)
{
  if (!strcmp (xml_key, target_key))
    {
      if (value)
	{
	  (*value) = strtol (xml_value, NULL, 10);
	}
    }
}

void
lw6cfg_read_xml_bool (char *xml_key, char *xml_value, char *target_key,
		      int *value)
{
  if (!strcmp (xml_key, target_key))
    {
      if (value)
	{
	  (*value) = strtol (xml_value, NULL, 10);
	  if (!(*value))
	    {
	      /*
	       * OK, trying to convert the value to a number gives 0,
	       * we just try if it could be a string like "true", "yes"
	       * on on.
	       * Note that prefered value is "true" 8-)
	       */
	      (*value) = (strcmp (xml_value, "true") == 0
			  || strcmp (xml_value, "yes") == 0
			  || strcmp (xml_value, "on") == 0);
	    }
	}
    }
}

void
lw6cfg_read_xml_float (char *xml_key, char *xml_value, char *target_key,
		       float *value)
{
  if (!strcmp (xml_key, target_key))
    {
      if (value)
	{
	  char *locale;
	  char *old_locale;

	  /*
	   * Important to set locale to C when calling atof,
	   * otherwise it returns garbage. Of course the XML
	   * file needs to be the same on every computer,
	   * and does not depend on locale.
	   */
	  locale = setlocale (LC_ALL, NULL);
	  if (locale)
	    {
	      /*
	       * We do need to make a copy in a separate buffer,
	       * otherwise the content pointed by *locale
	       * might change dynamically when calling setlocale
	       */
	      old_locale = lw6sys_str_copy (locale);

	      setlocale (LC_ALL, "C");
	      /*
	       * For some reason strtof doesn't work...
	       */
	      //(*value)=strtof(xml_value,NULL);
	      (*value) = atof (xml_value);
	      setlocale (LC_ALL, old_locale);
	      if (old_locale)
		{
		  LW6SYS_FREE (old_locale);
		}
	    }
	}
    }
}

void
lw6cfg_read_xml_string (char *xml_key, char *xml_value, char *target_key,
			char **value)
{
  if (!strcmp (xml_key, target_key))
    {
      if (value)
	{
	  if (*value)
	    {
	      /*
	       * This implies that *value is zeroed at initialisation,
	       * or filled with a dynamically allocated string.
	       */
	      LW6SYS_FREE (*value);
	      (*value) = NULL;
	    }
	  (*value) = lw6sys_str_copy (xml_value);
	}
    }
}

void
lw6cfg_read_xml_color (char *xml_key, char *xml_value,
		       char *target_key, LW6SYS_COLOR_F * value)
{
  if (!strcmp (xml_key, target_key))
    {
      LW6SYS_COLOR_8 color_8;

      color_8 = lw6sys_color_a_to_8 (xml_value);
      if (value)
	{
	  lw6sys_color_8_to_f (value, color_8);
	}
    }
}

static void XMLCALL
element_start (void *data, const char *element, const char **attr)
{
  _LW6CFG_XML_CALLBACK *callback;
  char *key = NULL;
  char *value = NULL;
  char **at;

  callback = (_LW6CFG_XML_CALLBACK *) data;

  if (callback && callback->func)
    {
      for (at = (char **) attr; (*at) != NULL; at += 2)
	{
	  if (strcmp ("key", (*at)) == 0)
	    {
	      key = *(at + 1);
	    }
	  if (strcmp ("value", (*at)) == 0)
	    {
	      value = *(at + 1);
	    }
	}
      if (key && value)
	{
	  callback->func (callback->data, (char *) element, key, value);
	}
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING, "sys",
		  _("XML callback function is NULL"));
    }
}

static void XMLCALL
element_end (void *data, const char *el)
{

}

int
lw6cfg_read_key_value_xml_file (char *filename,
				void (*callback_func) (void *callback_data,
						       char *element,
						       char *key,
						       char *value),
				void *callback_data)
{
  int ret = 0;
  char *content = NULL;
  _LW6CFG_XML_CALLBACK *callback;

  callback = LW6SYS_MALLOC (sizeof (_LW6CFG_XML_CALLBACK));
  if (callback)
    {
      callback->func = callback_func;
      callback->data = callback_data;

      content = lw6sys_read_file_content (filename);
      if (content)
	{
	  int length = strlen (content);
	  XML_Parser parser;

	  parser = XML_ParserCreate (NULL);
	  if (parser)
	    {
	      XML_SetElementHandler (parser, element_start, element_end);
	      XML_SetUserData (parser, (void *) callback);

	      if (XML_Parse (parser, content, length, 1) != XML_STATUS_ERROR)
		{
		  ret = 1;
		}
	      else
		{
		  lw6sys_log (LW6SYS_LOG_WARNING, "sys",
			      _
			      ("parse error reading XML file \"%s\" at line %d: \"%s\""),
			      filename,
			      XML_GetCurrentLineNumber (parser),
			      XML_ErrorString (XML_GetErrorCode (parser)));
		}
	      XML_ParserFree (parser);
	    }
	  LW6SYS_FREE (content);
	}
      LW6SYS_FREE (callback);
    }

  return ret;
}
