/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#include <stdlib.h>
#include <string.h>

#include "config.h"
#include "sys.h"

/*
 * Creates an empty list. There's a difference between NULL and an
 * empty list. The empty list would (in Scheme) be '() whereas
 * NULL corresponds to undefined "is not a list and will generate
 * errors if you ever call list functions on it".
 */
LW6SYS_LIST *
lw6sys_list_new (void (*free_func) (void *data))
{
  LW6SYS_LIST *ret = NULL;

  ret = LW6SYS_MALLOC (sizeof (LW6SYS_LIST));
  if (ret)
    {
      memset (ret, 0, sizeof (LW6SYS_LIST));
      ret->free_func = free_func;
    }

  return ret;
}

/*
 * Delete a list, this will cascade delete all the following
 * items in the list.
 */
void
lw6sys_list_free (LW6SYS_LIST * list)
{
  if (list)
    {
      /*
       * Keep a copy of next_item for we are about to
       * free the pointer to it.
       */
      LW6SYS_LIST *next_item = (void *) list->next_item;

      /*
       * It's legal to have free_func or data set to NULL,
       * we simply need to avoid the core dump, but one can
       * legitimately not need any peculiar free function,
       * or desire to store a NULL content in a valid list.
       */
      if (list->free_func && list->data)
	{
	  list->free_func (list->data);
	}

      LW6SYS_FREE (list);

      if (next_item)
	{
	  /*
	   * This should be the last call of the function.
	   * Hopefully the compiler will find this out and 
	   * optimize and *not* generate hudge stacks with
	   * return addresses which are of no use. At least
	   * the compiler *could* do it 8-) Recursion recursion...
	   */
	  lw6sys_list_free ((LW6SYS_LIST *) next_item);
	}
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING, "sys", _("trying to free NULL list"));
    }
}

/*
 * It's safer to call this rather than dig right into
 * the internals of the list.
 */
LW6SYS_LIST *
lw6sys_list_next (LW6SYS_LIST * list)
{
  LW6SYS_LIST *ret = NULL;

  if (list)
    {
      ret = list->next_item;
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING, "sys", _("calling next on NULL list"));
    }

  return ret;
}

/*
 * Returns true if the list is empty.
 */
int
lw6sys_list_is_empty (LW6SYS_LIST * list)
{
  int empty = 1;

  if (list)
    {
      empty = (list->next_item == NULL);
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING, "sys",
		  _("calling is_empty on NULL list"));
    }

  return empty;
}

/*
 * Returns the length of the list. This is a performance killer
 * for lists are inadapted to this. But it can still be usefull.
 */
int
lw6sys_list_length (LW6SYS_LIST * list)
{
  int ret = 0;

  if (list)
    {
      while (list->next_item)
	{
	  ret++;
	  list = (LW6SYS_LIST *) list->next_item;
	}
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING, "sys",
		  _("calling length on NULL list"));
    }

  return ret;
}

/*
 * Pushes data on the list. The free_func function is copied
 * from the previous element. The pointer on the list is
 * changed "in place" (in/out).
 */
void
lw6sys_list_push (LW6SYS_LIST ** list, void *data)
{
  LW6SYS_LIST *new_list = NULL;

  if (list && *list)
    {
      new_list = LW6SYS_MALLOC (sizeof (LW6SYS_LIST));
      if (new_list)
	{
	  new_list->next_item = (void *) (*list);
	  new_list->data = data;
	  new_list->free_func = (*list)->free_func;
	  (*list) = new_list;
	}
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING, "sys", _("calling push on NULL list"));
    }
}

/*
 * Pops data from the list, the returned value is what
 * was passed to list_push. The pointer on the list is
 * changed "in place" (in/out). When data is popped,
 * that needs some freeing (i.e. free_func was not NULL
 * when creating the list) then it's the responsibility
 * of the caller to free it when popping it. One popped
 * it's not freed, but it's out of the list scope. Of
 * course the LW6SYS_LIST is freed, but not the data.
 */
void *
lw6sys_list_pop (LW6SYS_LIST ** list)
{
  void *data = NULL;

  if (list && *list)
    {
      LW6SYS_LIST *new_list;
      data = (*list)->data;
      new_list = (*list)->next_item;

      /*
       * Here we do *NOT* call free_func, or else the
       * value returned would be freed before it is even
       * returned to the caller!!!
       */
      LW6SYS_FREE (*list);

      (*list) = new_list;
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING, "sys", _("calling pop on NULL list"));
    }

  return data;
}

/*
 * Executes a function on all list items.
 * The func_data parameter allows you to pass extra values to
 * the function, such as a file handler or any variable which
 * can not be inferred from list item values, and you of course
 * do not want to make global...
 */
void
lw6sys_list_map (LW6SYS_LIST * list,
		 void (*func) (void *func_data, void *data), void *func_data)
{
  if (list)
    {
      while (list)
	{
	  if (list->next_item)
	    {
	      func (func_data, list->data);
	    }
	  list = list->next_item;
	}
    }
  else
    {
      lw6sys_log (LW6SYS_LOG_WARNING, "sys", _("calling map on NULL list"));
    }
}
