/*
Liquid War 6 is a unique multiplayer wargame.
Copyright (C)  2005, 2006  Christian Mauduit <ufoot@ufoot.org>

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
Contact author        : ufoot@ufoot.org
*/

#include <stdlib.h>
#include <string.h>

#include "config.h"
#include "map.h"
#include "../cfg/cfg.h"
#include "map-internal.h"

#define STYLE_XML_FILE "style.xml"

void
lw6map_style_defaults (LW6MAP_STYLE * style)
{
  memset (style, 0, sizeof (LW6MAP_STYLE));

  style->keep_ratio = LW6MAP_STYLE_DEFAULT_KEEP_RATIO;
  style->zoom = LW6MAP_STYLE_DEFAULT_ZOOM;
  style->background_style =
    lw6sys_str_copy (LW6MAP_STYLE_DEFAULT_BACKGROUND_STYLE);
  style->hud_style = lw6sys_str_copy (LW6MAP_STYLE_DEFAULT_HUD_STYLE);
  style->menu_style = lw6sys_str_copy (LW6MAP_STYLE_DEFAULT_MENU_STYLE);
  style->view_style = lw6sys_str_copy (LW6MAP_STYLE_DEFAULT_VIEW_STYLE);
  lw6sys_color_a_to_f (&style->team_color_red,
		       LW6MAP_STYLE_DEFAULT_TEAM_COLOR_RED);
  lw6sys_color_a_to_f (&style->team_color_green,
		       LW6MAP_STYLE_DEFAULT_TEAM_COLOR_GREEN);
  lw6sys_color_a_to_f (&style->team_color_blue,
		       LW6MAP_STYLE_DEFAULT_TEAM_COLOR_BLUE);
  lw6sys_color_a_to_f (&style->team_color_yellow,
		       LW6MAP_STYLE_DEFAULT_TEAM_COLOR_YELLOW);
  lw6sys_color_a_to_f (&style->team_color_cyan,
		       LW6MAP_STYLE_DEFAULT_TEAM_COLOR_CYAN);
  lw6sys_color_a_to_f (&style->team_color_magenta,
		       LW6MAP_STYLE_DEFAULT_TEAM_COLOR_MAGENTA);
  lw6sys_color_a_to_f (&style->team_color_orange,
		       LW6MAP_STYLE_DEFAULT_TEAM_COLOR_ORANGE);
  lw6sys_color_a_to_f (&style->team_color_sky,
		       LW6MAP_STYLE_DEFAULT_TEAM_COLOR_SKY);
  lw6sys_color_a_to_f (&style->team_color_purple,
		       LW6MAP_STYLE_DEFAULT_TEAM_COLOR_PURPLE);
  lw6sys_color_a_to_f (&style->team_color_pink,
		       LW6MAP_STYLE_DEFAULT_TEAM_COLOR_PINK);
}

static void
read_callback (void *callback_data, char *element, char *key, char *value)
{
  LW6MAP_STYLE *style_data;

  style_data = (LW6MAP_STYLE *) callback_data;

  if (!strcmp (element, "int"))
    {
      // none yet
    }

  if (!strcmp (element, "bool"))
    {
      lw6cfg_read_xml_bool (key, value, "keep-ratio",
			    &style_data->keep_ratio);
    }

  if (!strcmp (element, "float"))
    {
      lw6cfg_read_xml_float (key, value, "zoom", &style_data->zoom);
    }

  if (!strcmp (element, "string"))
    {
      lw6cfg_read_xml_string (key, value, "background-style",
			      &style_data->background_style);
      lw6cfg_read_xml_string (key, value, "hud-style",
			      &style_data->hud_style);
      lw6cfg_read_xml_string (key, value, "menu-style",
			      &style_data->menu_style);
      lw6cfg_read_xml_string (key, value, "view-style",
			      &style_data->view_style);
    }

  if (!strcmp (element, "color"))
    {
      lw6cfg_read_xml_color (key, value, "team-color-red",
			     &style_data->team_color_red);
      lw6cfg_read_xml_color (key, value, "team-color-green",
			     &style_data->team_color_green);
      lw6cfg_read_xml_color (key, value, "team-color-blue",
			     &style_data->team_color_blue);
      lw6cfg_read_xml_color (key, value, "team-color-yellow",
			     &style_data->team_color_yellow);
      lw6cfg_read_xml_color (key, value, "team-color-cyan",
			     &style_data->team_color_cyan);
      lw6cfg_read_xml_color (key, value, "team-color-magenta",
			     &style_data->team_color_magenta);
      lw6cfg_read_xml_color (key, value, "team-color-orange",
			     &style_data->team_color_orange);
      lw6cfg_read_xml_color (key, value, "team-color-sky",
			     &style_data->team_color_sky);
      lw6cfg_read_xml_color (key, value, "team-color-purple",
			     &style_data->team_color_purple);
      lw6cfg_read_xml_color (key, value, "team-color-pink",
			     &style_data->team_color_pink);
    }
}

/*
 * Read the style associated to a map. Pointer to style must be valid,
 * it's modified in-place.
 */
int
lw6map_style_read (LW6MAP_STYLE * style, char *dirname)
{
  int ret = 0;
  char *buf = NULL;

  buf = lw6sys_str_concat (dirname, STYLE_XML_FILE);

  if (buf)
    {
      if (lw6sys_file_exists (buf))
	{
	  ret =
	    lw6cfg_read_key_value_xml_file (buf, read_callback,
					    (void *) style);
	}
      else
	{
	  lw6sys_log (LW6SYS_LOG_INFO, "map",
		      _("couldn't find \"%s\", using defaults"), buf);
	  ret = 1;
	}

      LW6SYS_FREE (buf);
    }

  if (!ret)
    {
      lw6sys_log (LW6SYS_LOG_WARNING, "map", _("unable to read map style"));
    }

  return ret;
}

/*
 * Note that this function won't work on an unitialized
 * structure, structure must be zeroed by some CALLOC
 * or something, else automatic freeing of pointers
 * will fail.
 */
void
lw6map_style_clear (LW6MAP_STYLE * style)
{
  if (style)
    {
      if (style->background_style)
	{
	  LW6SYS_FREE (style->background_style);
	}

      if (style->hud_style)
	{
	  LW6SYS_FREE (style->hud_style);
	}

      if (style->menu_style)
	{
	  LW6SYS_FREE (style->menu_style);
	}

      if (style->view_style)
	{
	  LW6SYS_FREE (style->view_style);
	}

      memset (style, 0, sizeof (LW6MAP_STYLE));
    }
}

/*
 * Like with clear, *dst must be either initialized or
 * totally zeroed, else function will fail (core dump)
 */
void
lw6map_style_copy (LW6MAP_STYLE * dst, LW6MAP_STYLE * src)
{
  lw6map_style_clear (dst);
  memcpy (dst, src, sizeof (LW6MAP_STYLE));

  if (src->background_style)
    {
      dst->background_style = lw6sys_str_copy (src->background_style);
    }
  if (src->hud_style)
    {
      dst->hud_style = lw6sys_str_copy (src->hud_style);
    }
  if (src->menu_style)
    {
      dst->menu_style = lw6sys_str_copy (src->menu_style);
    }
  if (src->view_style)
    {
      dst->view_style = lw6sys_str_copy (src->view_style);
    }
}

void
lw6map_style_get_team_color (LW6MAP_STYLE * style, LW6SYS_COLOR_F * color_f,
			     int id)
{
/*
 * Questionnable design, since colors are hold in
 * a "named" struct and not in a array[LW6OPT_MAX_NB_TEAMS]
 * we're stuck with this ugly switch. Hopefully it's not
 * callled so often.
 */
  switch (id)
    {
    case LW6MAP_TEAM_COLOR_RED:
      (*color_f) = style->team_color_red;
      break;
    case LW6MAP_TEAM_COLOR_GREEN:
      (*color_f) = style->team_color_green;
      break;
    case LW6MAP_TEAM_COLOR_BLUE:
      (*color_f) = style->team_color_blue;
      break;
    case LW6MAP_TEAM_COLOR_YELLOW:
      (*color_f) = style->team_color_yellow;
      break;
    case LW6MAP_TEAM_COLOR_CYAN:
      (*color_f) = style->team_color_cyan;
      break;
    case LW6MAP_TEAM_COLOR_MAGENTA:
      (*color_f) = style->team_color_magenta;
      break;
    case LW6MAP_TEAM_COLOR_ORANGE:
      (*color_f) = style->team_color_orange;
      break;
    case LW6MAP_TEAM_COLOR_SKY:
      (*color_f) = style->team_color_sky;
      break;
    case LW6MAP_TEAM_COLOR_PURPLE:
      (*color_f) = style->team_color_purple;
      break;
    case LW6MAP_TEAM_COLOR_PINK:
      (*color_f) = style->team_color_pink;
      break;
    default:
      lw6sys_log (LW6SYS_LOG_WARNING, "map",
		  _("out of range team color %d"), id);
      break;
    }
}
