/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#ifndef LIQUIDWAR6KER_H
#define LIQUIDWAR6KER_H

#include "../lw6-common.h"

#include "../sys/sys.h"
#include "../opt/opt.h"
#include "../map/map.h"

#define LW6KER_NB_DIRS 12

#define LW6KER_DIR_NNE 0
#define LW6KER_DIR_NE 1
#define LW6KER_DIR_ENE 2
#define LW6KER_DIR_ESE 3
#define LW6KER_DIR_SE 4
#define LW6KER_DIR_SSE 5
#define LW6KER_DIR_SSW 6
#define LW6KER_DIR_SW 7
#define LW6KER_DIR_WSW 8
#define LW6KER_DIR_WNW 9
#define LW6KER_DIR_NW 10
#define LW6KER_DIR_NNW 11

#define LW6KER_NB_PARITIES 2

#define LW6KER_PARITY_EVEN 0
#define LW6KER_PARITY_ODD 1

#include "ker-trigo.h"

typedef struct LW6KER_ZONE_STRUCT_STRUCT
{
  LW6SYS_XY pos;
  LW6SYS_UINT32 size;
  LW6SYS_INT32 link[LW6KER_NB_DIRS];
}
LW6KER_ZONE_STRUCT;

typedef struct LW6KER_SLOT_STRUCT_STRUCT
{
  LW6SYS_INT32 zone_id;
  LW6SYS_INT32 depth;
}
LW6KER_SLOT_STRUCT;

/*
 * To some extent, this structure is very similar to LW6MAP_MAP.
 * Let's explain why it exists separately: 
 * - this structure is not only used to store the information, but
 *   also to present it in a form which is adapted to (fast) treatments.
 *   This means it can be redundant, akward, whereas LW6MAP_MAP is meant
 *   to just convey informations. LW6KER_MAP_STRUCT is used when the
 *   game is running, to inform the core algorithm of tha map structure,
 *   whereas LW6MAP_MAP is used to intialize LW6KER_MAP_STRUCT, to send
 *   the maps structure to other players over the network, to initialize
 *   the graphics artifacts which will be used for rendering, and so on.
 * - having a loading/serialization (that is LW6MAP_MAP) separated from
 *   algorithmic wizardry related to shortest path insanities might be
 *   what could prevent LW6 from being as bloated as LW5.
 */
typedef struct LW6KER_MAP_STRUCT_STRUCT
{
  LW6SYS_WH shape;
  LW6SYS_INT32 nb_zones;
  LW6SYS_INT32 nb_slots;
  LW6SYS_INT32 nb_usable_slots;
  LW6SYS_INT32 room_for_armies;
  LW6SYS_INT32 max_depth;
  LW6SYS_INT32 max_zone_size;
  LW6KER_ZONE_STRUCT *zones;
  LW6KER_SLOT_STRUCT *slots;
}
LW6KER_MAP_STRUCT;

/*
 * Game struct contains all the data which never change during a game.
 * If it was feasible in C, we would change it to "read-only" when game
 * starts.
 */
typedef struct LW6KER_GAME_STRUCT_STRUCT
{
  int id;
  LW6KER_MAP_STRUCT map;
  LW6OPT_STATIC options;
}
LW6KER_GAME_STRUCT;

/*
 * Completes LW6KER_ZONE_STRUCT with per-team information.
 */
typedef struct LW6KER_ZONE_STATE_STRUCT
{
  LW6SYS_INT32 potential:24;
  LW6SYS_INT32 direction_to_cursor:8;
  LW6SYS_XY closest_cursor_pos;
}
LW6KER_ZONE_STATE;

typedef struct LW6KER_SLOT_STATE_STRUCT
{
  LW6SYS_INT32 fighter_id;
}
LW6KER_SLOT_STATE;

typedef struct LW6KER_FIGHTER_STRUCT
{
  LW6SYS_UINT32 team_id:4;
  LW6SYS_UINT32 layer:4;
  LW6SYS_UINT32 last_direction:8;
  LW6SYS_UINT32 health:16;
  LW6SYS_XY pos;
}
LW6KER_FIGHTER;

typedef struct LW6KER_CURSOR_STRUCT
{
  LW6SYS_XY pos;
  LW6SYS_XY last_applied_pos;
  LW6SYS_INT32 pot_offset;
}
LW6KER_CURSOR;

typedef struct LW6KER_CURSOR_ARRAY_STRUCT
{
  LW6SYS_INT32 nb_cursors;
  LW6KER_CURSOR cursors[LW6OPT_MAX_CURSORS_PER_TEAM];
}
LW6KER_CURSOR_ARRAY;

typedef struct LW6KER_TEAM_STRUCT
{
  LW6SYS_BOOL active;
  LW6KER_MAP_STRUCT *map_struct;
  LW6KER_ZONE_STATE *gradient;
  LW6SYS_INT32 cursor_ref_pot;
  LW6SYS_INT32 last_spread_dir;
  LW6KER_CURSOR_ARRAY cursor_array;
}
LW6KER_TEAM;

typedef struct LW6KER_ARMIES_STRUCT
{
  LW6KER_MAP_STRUCT *map_struct;
  /*
   * The maximum number or fighters in the armies, in fact
   * it's more a less the amount of free space in the map, minus
   * some standard percentage to avoid bloating a map completely
   * (playing with 100% space filled doesn't really make sense).
   */
  LW6SYS_INT32 max_fighters;
  /*
   * The number of active fighters, that is the ones that are
   * actually doing things on the map, moving, fighting. Other
   * ones are dormant fighters which will activate when a
   * network player connects for instance.
   */
  LW6SYS_INT32 active_fighters;
  /*
   * This is highly redundant for one could get this information
   * by simply reading the fighters themselves, however this is
   * inefficient, and having the exact count is usefull at every
   * game refreh, to display the information on the screen.
   */
  LW6SYS_INT32 fighters_per_team[LW6OPT_MAX_NB_TEAMS];
  /*
   * The actual data, a pointer to all the fighters in the map.
   * It's more convenient to access them this way, instead of
   * having to read a w*h array, which can be very big and 90%
   * empty. This optimization is not usefull on small and/or
   * crowded maps, but experience shows that:
   * a) only small maps can be crowded anyway
   * b) small maps are fast, only big maps require optimization.
   */
  LW6KER_FIGHTER *fighters;
}
LW6KER_ARMIES;

typedef struct LW6KER_LAYER_STRUCT
{
  LW6KER_MAP_STRUCT *map_struct;
  LW6SYS_INT32 nb_slots;	// redundant but convenient
  LW6KER_ARMIES *armies;
  LW6KER_SLOT_STATE *slots;
} LW6KER_LAYER;

typedef struct LW6KER_MAP_STATE_STRUCT
{
  LW6KER_MAP_STRUCT *map_struct;
  LW6SYS_WH shape;		// redundant but convenient
  LW6KER_ARMIES armies;
  LW6SYS_INT32 max_nb_teams;
  LW6KER_TEAM teams[LW6OPT_MAX_NB_TEAMS];
  LW6SYS_INT32 nb_layers;
  LW6KER_LAYER layers[LW6MAP_MAX_DEPTH];
}
LW6KER_MAP_STATE;

typedef struct LW6KER_BOT_STRUCT
{
  LW6SYS_BOOL active;
  LW6SYS_XY pos;
  LW6SYS_INT32 rounds_at_end_of_last_move;
  LW6SYS_BOOL moving;
  LW6SYS_INT32 move_nb_steps;
  LW6SYS_INT32 move_current_step;
  LW6SYS_XY move_pos_from;
  LW6SYS_XY move_pos_to;
} LW6KER_BOT;

/*
 * Game state contains all the data which is verstatile, stuff that
 * changes, this includes "where fighters are" of course but also
 * some parameters which might change in-game (such as an option
 * which can be changeable, an alliance between teams, and so on...)
 */
typedef struct LW6KER_GAME_STATE_STRUCT
{
  int id;
  LW6KER_GAME_STRUCT *game_struct;
  LW6KER_MAP_STATE map;
  LW6OPT_DYNAMIC options;
  LW6SYS_UINT32 moves;
  LW6SYS_UINT32 spreads;
  LW6SYS_UINT32 rounds;
  LW6KER_BOT bots[LW6OPT_MAX_NB_TEAMS][LW6OPT_MAX_CURSORS_PER_TEAM];
}
LW6KER_GAME_STATE;

/*
 * in armies.c
 */
extern LW6SYS_INT32 lw6ker_armies_add_fighter (LW6KER_ARMIES * armies,
					       LW6KER_FIGHTER fighter);
extern LW6SYS_BOOL lw6ker_armies_remove_fighter (LW6KER_ARMIES * armies);

/*
 * in cursor.c
 */
extern LW6SYS_BOOL lw6ker_cursor_set (LW6KER_CURSOR * cursor, LW6SYS_INT32 x,
				      LW6SYS_INT32 y, LW6SYS_INT32 pot_offset,
				      LW6KER_MAP_STRUCT * map_struct,
				      LW6SYS_INT32 max_cursor_pot_offset);
extern void lw6ker_cursor_clear (LW6KER_CURSOR * cursor);

/*
 * in cursorarray.c
 */
extern void lw6ker_cursor_array_clear (LW6KER_CURSOR_ARRAY * cursor_array);
extern LW6SYS_BOOL lw6ker_cursor_array_add (LW6KER_CURSOR_ARRAY *
					    cursor_array,
					    LW6KER_CURSOR * cursor);
extern LW6SYS_BOOL lw6ker_cursor_array_set (LW6KER_CURSOR_ARRAY *
					    cursor_array, LW6SYS_INT32 i,
					    LW6SYS_INT32 x, LW6SYS_INT32 y,
					    LW6SYS_INT32 pot_offset,
					    LW6KER_MAP_STRUCT * map_struct,
					    LW6SYS_INT32
					    max_cursor_pot_offset);
extern LW6SYS_BOOL lw6ker_cursor_array_force_nb_cursors (LW6KER_CURSOR_ARRAY *
							 cursor_array,
							 LW6SYS_INT32
							 nb_cursors);

/*
 * in gamestate.c
 */
extern LW6KER_GAME_STATE *lw6ker_game_state_new (LW6KER_GAME_STRUCT *
						 game_struct);
extern void lw6ker_game_state_free (LW6KER_GAME_STATE * game_state);
extern int lw6ker_game_state_memory_footprint (LW6KER_GAME_STATE *
					       game_state);
extern char *lw6ker_game_state_repr (LW6KER_GAME_STATE * game_state);
extern LW6SYS_BOOL lw6ker_game_state_copy (LW6KER_GAME_STATE * dst,
					   LW6KER_GAME_STATE * src);
extern LW6SYS_UINT32 lw6ker_game_state_checksum (LW6KER_GAME_STATE *
						 game_state);
extern LW6SYS_BOOL lw6ker_game_state_add_team (LW6KER_GAME_STATE * game_state,
					       LW6SYS_INT32 team_id,
					       LW6SYS_INT32 nb_cursors);
extern LW6SYS_BOOL lw6ker_game_state_remove_team (LW6KER_GAME_STATE *
						  game_state,
						  LW6SYS_INT32 team_id);
extern void lw6ker_game_state_do_round (LW6KER_GAME_STATE * game_state);
extern LW6SYS_UINT32 lw6ker_game_state_get_moves (LW6KER_GAME_STATE *
						  game_state);
extern LW6SYS_UINT32 lw6ker_game_state_get_spreads (LW6KER_GAME_STATE *
						    game_state);
extern LW6SYS_UINT32 lw6ker_game_state_get_rounds (LW6KER_GAME_STATE *
						   game_state);
extern LW6SYS_BOOL lw6ker_game_state_enable_bot (LW6KER_GAME_STATE *
						 game_state,
						 LW6SYS_INT32 team_id);
extern LW6SYS_BOOL lw6ker_game_state_disable_bot (LW6KER_GAME_STATE *
						  game_state,
						  LW6SYS_INT32 team_id);

/*
 * In gamestruct.c
 */
extern LW6KER_GAME_STRUCT *lw6ker_game_struct_new (LW6MAP_MAP * map);
extern void lw6ker_game_struct_free (LW6KER_GAME_STRUCT * game_struct);
extern int lw6ker_game_struct_memory_footprint (LW6KER_GAME_STRUCT *
						game_struct);
extern char *lw6ker_game_struct_repr (LW6KER_GAME_STRUCT * game_struct);
extern LW6SYS_UINT32 lw6ker_game_struct_checksum (LW6KER_GAME_STRUCT *
						  game_struct);

/*
 * In layer.c
 */

/*
 * In map.c
 */
extern LW6MAP_MAP *lw6ker_map_new_rectangle (int width, int height);
extern void lw6ker_map_free_rectangle (LW6MAP_MAP * map);

/*
 * in mapstate.c
 */
extern LW6SYS_INT32 lw6ker_map_state_get_free_team_id (LW6KER_MAP_STATE *
						       map_state);
extern void lw6ker_map_state_start_xy (LW6KER_MAP_STATE * map_state,
				       LW6SYS_XY * start_xy,
				       LW6SYS_INT32 team_id);
extern LW6SYS_INT32 lw6ker_map_state_get_nb_active_teams (LW6KER_MAP_STATE *
							  map_state);;
extern LW6SYS_INT32 lw6ker_map_state_populate_team (LW6KER_MAP_STATE *
						    map_state,
						    LW6SYS_INT32 team_id,
						    LW6SYS_INT32 nb_fighters,
						    LW6SYS_XY desired_center,
						    LW6SYS_INT32 nb_cursors,
						    LW6OPT_STATIC options);
extern LW6SYS_BOOL lw6ker_map_state_redistribute_team (LW6KER_MAP_STATE *
						       map_state,
						       LW6SYS_INT32
						       dst_team_id,
						       LW6SYS_INT32
						       src_team_id,
						       LW6SYS_INT32
						       nb_fighters,
						       LW6OPT_STATIC options);
extern LW6SYS_BOOL lw6ker_map_state_cancel_team (LW6KER_MAP_STATE * map_state,
						 LW6SYS_INT32 team_id);
extern LW6SYS_BOOL lw6ker_map_state_remove_fighter (LW6KER_MAP_STATE *
						    map_state,
						    LW6SYS_INT32 fighter_id);
extern LW6SYS_BOOL lw6ker_map_state_remove_fighters (LW6KER_MAP_STATE *
						     map_state,
						     LW6SYS_INT32
						     nb_fighters);
extern LW6SYS_BOOL lw6ker_map_state_remove_team_fighters (LW6KER_MAP_STATE *
							  map_state,
							  LW6SYS_INT32
							  team_id,
							  LW6SYS_INT32
							  nb_fighters);
static inline void
lw6ker_map_state_set_fighter_id (LW6KER_MAP_STATE * map_state,
				 LW6SYS_INT32 layer, LW6SYS_INT32 x,
				 LW6SYS_INT32 y, LW6SYS_INT32 fighter_id)
{
  map_state->layers[layer].slots[map_state->shape.w * y + x].fighter_id =
    fighter_id;
};
static inline LW6SYS_INT32
lw6ker_map_state_get_fighter_id (LW6KER_MAP_STATE * map_state,
				 LW6SYS_INT32 layer, LW6SYS_INT32 x,
				 LW6SYS_INT32 y)
{
  return (map_state->layers[layer].slots[map_state->shape.w * y + x].
	  fighter_id);
};

/*
 * Carefull with this one, there must really be a fighter
 * or it segfaults directly.
 */
static inline LW6KER_FIGHTER *
lw6ker_map_state_get_fighter (LW6KER_MAP_STATE * map_state,
			      LW6SYS_INT32 layer, LW6SYS_INT32 x,
			      LW6SYS_INT32 y)
{
  return (&
	  (map_state->armies.
	   fighters[map_state->layers[layer].
		    slots[map_state->shape.w * y + x].fighter_id]));
};
extern void lw6ker_map_state_print_debug (LW6KER_MAP_STATE * map_state);
extern LW6SYS_BOOL lw6ker_map_state_sanity_check (LW6KER_MAP_STATE *
						  map_state);
extern void lw6ker_map_state_spread_gradient (LW6KER_MAP_STATE * map_state,
					      LW6OPT_STATIC * options,
					      LW6SYS_INT32 nb_spreads);
extern void lw6ker_map_state_move_fighters (LW6KER_MAP_STATE * map_state,
					    LW6SYS_BOOL parity,
					    LW6OPT_STATIC * options,
					    LW6SYS_INT32 nb_moves);

/*
 * In mapstruct.c
 */
static inline void
lw6ker_map_struct_set_depth (LW6KER_MAP_STRUCT * map_struct, LW6SYS_INT32 x,
			     LW6SYS_INT32 y, LW6SYS_INT32 depth)
{
  map_struct->slots[map_struct->shape.w * y + x].depth = depth;
};
static inline LW6SYS_INT32
lw6ker_map_struct_get_depth (LW6KER_MAP_STRUCT * map_struct, LW6SYS_INT32 x,
			     LW6SYS_INT32 y)
{
  return (map_struct->slots[map_struct->shape.w * y + x].depth);
};
static inline void
lw6ker_map_struct_set_zone_id (LW6KER_MAP_STRUCT * map_struct, LW6SYS_INT32 x,
			       LW6SYS_INT32 y, LW6SYS_INT32 zone_id)
{
  map_struct->slots[map_struct->shape.w * y + x].zone_id = zone_id;
};
static inline LW6SYS_INT32
lw6ker_map_struct_get_zone_id (LW6KER_MAP_STRUCT * map_struct, LW6SYS_INT32 x,
			       LW6SYS_INT32 y)
{
  return (map_struct->slots[map_struct->shape.w * y + x].zone_id);
};
extern void lw6ker_map_struct_find_free_slot_near (LW6KER_MAP_STRUCT *
						   map_struct,
						   LW6SYS_XY * there,
						   LW6SYS_XY here);

/*
 * In team.c
 */
extern void lw6ker_team_activate (LW6KER_TEAM * team,
				  LW6SYS_INT32 nb_cursors, LW6SYS_XY pos);
extern void lw6ker_team_unactivate (LW6KER_TEAM * team);
extern void lw6ker_team_apply_cursors (LW6KER_TEAM * team,
				       LW6OPT_STATIC * options);
extern void lw6ker_team_spread_gradient (LW6KER_TEAM * team);
extern void lw6ker_team_normalize_pot (LW6KER_TEAM * team,
				       LW6OPT_STATIC * options);

/*
 * In trigo.c
 */
static inline LW6SYS_INT32
lw6ker_sin (LW6SYS_INT32 alpha)
{
  alpha = alpha & LW6KER_TRIGO_2PI_MASK;
  return LW6KER_TRIGO_SIN_TABLE[alpha];
}
static inline LW6SYS_INT32
lw6ker_cos (LW6SYS_INT32 alpha)
{
  alpha = (LW6KER_TRIGO_PI2 + alpha) & LW6KER_TRIGO_2PI_MASK;
  return LW6KER_TRIGO_SIN_TABLE[alpha];
}

/*
 * Various utils
 */
static inline LW6SYS_INT32
lw6ker_percent (LW6SYS_INT32 n, LW6SYS_INT32 percent)
{
  return (n * percent) / 100;
}
static inline LW6SYS_INT32
lw6ker_per1000 (LW6SYS_INT32 n, LW6SYS_INT32 per1000)
{
  return (n * per1000) / 1000;
}

#endif
