/*
  Liquid War 6 is a unique multiplayer wargame.
  Copyright (C)  2005, 2006  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

  Liquid War 6 homepage : http://www.gnu.org/software/liquidwar6/
  Contact author        : ufoot@ufoot.org
*/

#include <GL/glu.h>

#include "config.h"
#include "../../gfx.h"
#include "gl-utils.h"
#include "gl-utils-internal.h"

static void
get_text_wh (TTF_Font * font, char *text, int *width, int *height)
{
  if (TTF_SizeUTF8 (font, text, width, height) == -1)
    {
      lw6sys_log (LW6SYS_LOG_WARNING, "gfx-mod-gl",
		  _
		  ("unable to calculate font width and height for text \"%s\""),
		  text);
      width = 0;
    }
}

static int
get_text_width (TTF_Font * font, char *text)
{
  int width = 0;

  get_text_wh (font, text, &width, NULL);

  return width;
}

static int
get_text_height (TTF_Font * font, char *text)
{
  int height = 0;

  get_text_wh (font, text, NULL, &height);

  return height;
}

/*
 * Returns the size of the string if it was drawn
 */
int
mod_gl_utils_get_system_text_width (MOD_GL_UTILS_CONTEXT * context,
				    char *text)
{
  return get_text_width (context->font_data.system, text);
}

/*
 * Returns the size of the string if it was drawn
 */
int
mod_gl_utils_get_system_text_height (MOD_GL_UTILS_CONTEXT * context,
				     char *text)
{
  return get_text_height (context->font_data.system, text);
}

/*
 * Helper internal func
 */
static void
draw_text (MOD_GL_UTILS_CONTEXT * context,
	   TTF_Font * font, SDL_Color color, char *text, int x, int y)
{
  SDL_Surface *text_surface;
  GLuint text_texture;
  char *utf8;
  int ret = 0;

  utf8 = lw6sys_locale_to_utf8 (text);
  if (utf8 != NULL)
    {
      text_surface = TTF_RenderUTF8_Blended (font, utf8, color);
      if (text_surface != NULL)
	{
	  context->surface_counter.new_counter++;
	  text_texture = mod_gl_utils_surface2texture (context, text_surface);
	  if (text_texture)
	    {
	      glEnable (GL_TEXTURE_2D);
	      glColor3f (1.0f, 1.0f, 1.0f);
	      glBindTexture (GL_TEXTURE_2D, text_texture);

	      GLdouble x1 = (GLdouble) x;
	      GLdouble y1 = (GLdouble) y;
	      GLdouble x2 = (GLdouble) text_surface->w + x;
	      GLdouble y2 = (GLdouble) text_surface->h + y;
	      GLdouble texw = mod_gl_utils_texture_scale (text_surface->w);
	      GLdouble texh = mod_gl_utils_texture_scale (text_surface->h);

	      glBegin (GL_QUADS);
	      glTexCoord2d (0.0f, 0.0f);
	      glVertex3f (x1, y1, 0.0f);	// top left
	      glTexCoord2d (texw, 0.0f);
	      glVertex3f (x2, y1, 0.0f);	// top right
	      glTexCoord2d (texw, texh);
	      glVertex3f (x2, y2, 0.0f);	// bottom right
	      glTexCoord2d (0.0f, texh);
	      glVertex3f (x1, y2, 0.0f);	// bottom left
	      glEnd ();
	      glPopMatrix ();

	      ret = 1;

	      mod_gl_utils_schedule_delete_texture (context, text_texture);
	    }
	  mod_gl_utils_delete_surface (context, text_surface);
	}
      else
	{
	  lw6sys_log (LW6SYS_LOG_WARNING, "gfx-mod-gl",
		      _("TTF_RenderUTF8_Blended failed"));
	}
      LW6SYS_FREE (utf8);
    }

  if (!ret)
    {
      lw6sys_log (LW6SYS_LOG_WARNING, "gfx-mod-gl",
		  _("unable to draw text \"%s\""), text);
    }
}

static void
system_colors (SDL_Color * fg, SDL_Color * shade,
	       MOD_GL_UTILS_CONTEXT * context)
{
  (*fg) = mod_gl_utils_color_sys2sdl (&(context->const_data.system_font_fg));
  (*shade) =
    mod_gl_utils_color_sys2sdl (&(context->const_data.system_font_shade));
}

/*
 * Draws a font on the screen, at a given position
 */
void
mod_gl_utils_draw_system_text (MOD_GL_UTILS_CONTEXT * context, char *text,
			       int x, int y)
{
  SDL_Color fg, shade;

  system_colors (&fg, &shade, context);
  mod_gl_utils_set_render_mode_2d_blend (context);


  draw_text (context, context->font_data.system,
	     shade, text,
	     x + context->const_data.system_font_dw,
	     y + context->const_data.system_font_dh);
  draw_text (context, context->font_data.system, fg, text, x, y);
}

void
mod_gl_utils_draw_system_text_top_left (MOD_GL_UTILS_CONTEXT * context,
					char **text_list)
{
  int y;
  int w;
  int h;
  SDL_Color fg, shade;

  system_colors (&fg, &shade, context);
  mod_gl_utils_set_render_mode_2d_blend (context);

  y = 0;
  while (*text_list)
    {
      get_text_wh (context->font_data.system, *text_list, &w, &h);
      draw_text (context, context->font_data.system,
		 shade, *text_list, 0,
		 y + context->const_data.system_font_dh);
      draw_text (context, context->font_data.system, fg, *text_list, 0, y);
      y +=
	(context->const_data.system_font_hcoef * h) +
	context->const_data.system_font_dh;
      text_list++;
    }
}

void
mod_gl_utils_draw_system_text_top_right (MOD_GL_UTILS_CONTEXT * context,
					 char **text_list)
{
  int x;
  int y;
  int w;
  int h;
  SDL_Color fg, shade;

  system_colors (&fg, &shade, context);

  mod_gl_utils_set_render_mode_2d_blend (context);

  x = context->screen.width;
  y = 0;
  while (*text_list)
    {
      get_text_wh (context->font_data.system, *text_list, &w, &h);
      draw_text (context, context->font_data.system,
		 shade, *text_list, x - w,
		 y + context->const_data.system_font_dh);
      draw_text (context, context->font_data.system,
		 fg, *text_list,
		 x - w - context->const_data.system_font_dw, y);
      y +=
	(context->const_data.system_font_hcoef * h) +
	context->const_data.system_font_dh;
      text_list++;
    }
}

void
mod_gl_utils_draw_system_text_bottom_left (MOD_GL_UTILS_CONTEXT * context,
					   char **text_list)
{
  int y;
  int w;
  int h;
  float hcoef = 1.0f;
  char **tmp_list;
  char **list_end;
  SDL_Color fg, shade;

  system_colors (&fg, &shade, context);

  mod_gl_utils_set_render_mode_2d_blend (context);

  list_end = text_list;
  while (*list_end)
    {
      list_end++;
    }

  y = context->screen.height;
  tmp_list = list_end;
  while (tmp_list != text_list)
    {
      tmp_list--;

      get_text_wh (context->font_data.system, *tmp_list, &w, &h);
      y -= (hcoef * h) + context->const_data.system_font_dh;
      draw_text (context, context->font_data.system,
		 shade, *tmp_list,
		 context->const_data.system_font_dw,
		 y + context->const_data.system_font_dh);
      draw_text (context, context->font_data.system, fg, *tmp_list, 0, y);
      hcoef = context->const_data.system_font_hcoef;
    }
}

void
mod_gl_utils_draw_system_text_bottom_right (MOD_GL_UTILS_CONTEXT * context,
					    char **text_list)
{
  int x;
  int y;
  int w;
  int h;
  float hcoef = 1.0f;
  char **tmp_list;
  char **list_end;
  SDL_Color fg, shade;

  system_colors (&fg, &shade, context);

  mod_gl_utils_set_render_mode_2d_blend (context);

  list_end = text_list;
  while (*list_end)
    {
      list_end++;
    }

  x = context->screen.width;
  y = context->screen.height;
  tmp_list = list_end;
  while (tmp_list != text_list)
    {
      tmp_list--;

      get_text_wh (context->font_data.system, *tmp_list, &w, &h);
      y -= (hcoef * h) + context->const_data.system_font_dh;
      draw_text (context, context->font_data.system,
		 shade, *tmp_list,
		 x - w + context->const_data.system_font_dw,
		 y + context->const_data.system_font_dh);
      draw_text (context, context->font_data.system, fg, *tmp_list, x - w, y);
      hcoef = context->const_data.system_font_hcoef;
    }
}
