/*
Liquid War is a multiplayer wargame.
Copyright (C) 2005  Christian Mauduit <ufoot@ufoot.org>

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

Liquid War homepage : http://www.ufoot.org/liquidwar/v6
Contact author      : ufoot@ufoot.org
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <time.h>

#include "../liquidwar6common.h"
#include "../liquidwar6gfx.h"
#include "internal.h"

/*
 * Low-level SDL initialisation.
 */
_LW6GFX_CONTEXT *_lw6gfx_init() {
  _LW6GFX_CONTEXT *context=NULL;

  context=(_LW6GFX_CONTEXT *) LW6SYS_MALLOC(sizeof(_LW6GFX_CONTEXT));
  if (!context) {
    /*
     * No use to continue if this basic malloc fails...
     */
    exit(1);
  }
  memset(context,0,sizeof(_LW6GFX_CONTEXT));

  if (SDL_Init(SDL_INIT_TIMER|SDL_INIT_VIDEO|SDL_INIT_JOYSTICK)<0) {	
    lw6sys_log_error("gfx", _("SDL init error: \"%s\""), SDL_GetError());
    exit(1);
  } else {
    lw6sys_log_info("gfx", _("SDL Init"));
  }

  SDL_EnableUNICODE(1);
  
  if(TTF_Init()==-1) {
    lw6sys_log_error("gfx", _("SDL_ttf init error: \"%s\""), TTF_GetError());
    exit(1);
  }

  srandom(time(NULL));

  return context;
}

void *lw6gfx_init() {
  return _lw6gfx_init();
}

/*
 * Initialize display.
 */
int _lw6gfx_set_video_mode(_LW6GFX_CONTEXT *context,int width, int height, int fullscreen) {
  /* Information about the current video settings. */
  const SDL_VideoInfo* info = NULL;
  /* Color depth in bits of our window. */
  int bpp = 0;
  /* Flags we will pass into SDL_SetVideoMode. */
  int flags = 0;
  
  /* Let's get some video information. */
  info = SDL_GetVideoInfo( );
  
  if( !info ) {
    /* This should probably never happen. */
    lw6sys_log_error("gfx", _("SDL GetVideoInfo failed: \"%s\""), SDL_GetError());
    fflush(stderr);
    return 0;
  }
  
  bpp = info->vfmt->BitsPerPixel;
  
  /*
   * Now, we want to setup our requested
   * window attributes for our OpenGL window.
   * We want *at least* 5 bits of red, green
   * and blue. We also want at least a 16-bit
   * depth buffer.
   *
   * The last thing we do is request a double
   * buffered window. '1' turns on double
   * buffering, '0' turns it off.
   *
   * Note that we do not use SDL_DOUBLEBUF in
   * the flags to SDL_SetVideoMode. That does
   * not affect the GL attribute state, only
   * the standard 2D blitting setup.
   */
  SDL_GL_SetAttribute( SDL_GL_RED_SIZE, 5 );
  SDL_GL_SetAttribute( SDL_GL_GREEN_SIZE, 5 );
  SDL_GL_SetAttribute( SDL_GL_BLUE_SIZE, 5 );
  SDL_GL_SetAttribute( SDL_GL_DEPTH_SIZE, 16 );
  SDL_GL_SetAttribute( SDL_GL_DOUBLEBUFFER, 1 );

  flags = SDL_OPENGL | (fullscreen ? SDL_FULLSCREEN : SDL_RESIZABLE);

  /*
   * Set the video mode
   */
  if( SDL_SetVideoMode( width, height, bpp, flags ) == 0 ) {
    /* 
     * This could happen for a variety of reasons,
     * including DISPLAY not being set, the specified
     * resolution not being available, etc.
     */
    lw6sys_log_error("gfx", _("SDL SetVideoMode %dx%d failed: \"%s\""), width, height, SDL_GetError());
    fflush(stderr);
    return 0;
  } else {
    lw6sys_log_info("gfx", _("SDL SetVideoMode %dx%d"), width, height);
    context->screen.width=width;
    context->screen.height=height;
    context->screen.bpp=bpp;
    context->screen.fullscreen=fullscreen;
  }

  SDL_WM_SetCaption(PACKAGE_STRING, PACKAGE_NAME);

  lw6sys_log_info("gfx", _("%d BPP"), SDL_GetVideoSurface()->format->BitsPerPixel);
  lw6sys_log_info("gfx", _("OpenGL vendor \"%s\""),
		  glGetString(GL_VENDOR));
  lw6sys_log_info("gfx", _("OpenGL renderer \"%s\""),
		  glGetString(GL_RENDERER));
  lw6sys_log_info("gfx", _("OpenGL version \"%s\""),
		  glGetString(GL_VERSION));

  return 1;
}

int lw6gfx_set_video_mode(void *context, int width, int height, int fullscreen) {
  return _lw6gfx_set_video_mode((_LW6GFX_CONTEXT *) context, width, height, fullscreen);
}

/*
 * Called whenever window resize is asked for.
 */
int _lw6gfx_resize_video_mode(_LW6GFX_CONTEXT *context,int width, int height) {
  if (width!=context->screen.width || height!=context->screen.height) {
    int flags = 0;

    flags = SDL_OPENGL | (context->screen.fullscreen ? SDL_FULLSCREEN : SDL_RESIZABLE);

    if( SDL_SetVideoMode( width, height, context->screen.bpp, flags ) == 0 ) {
      lw6sys_log_warning("gfx",_("unable to resize screen %dx%d"),width,height);
    } else { 
      glViewport(0,0,width,height);
      context->screen.width=width;
      context->screen.height=height;
      lw6sys_log_info("gfx",_("resizing screen %dx%d"),width,height);
    }
  }
}

/*
 * Ends-up all SDL stuff.
 */
void _lw6gfx_quit(_LW6GFX_CONTEXT *context) {
  lw6sys_log_info("gfx", _("SDL Quit"));
  glFinish();
  TTF_Quit();
  SDL_Quit();

  LW6SYS_FREE(context);
}

void lw6gfx_quit(void *context) {
  _lw6gfx_quit((_LW6GFX_CONTEXT *) context);
}

