/*
Liquid War is a multiplayer wargame.
Copyright (C) 2005  Christian Mauduit <ufoot@ufoot.org>

This program is free software; you can redistribute it and/or
modify it under the terms of the GNU General Public License
as published by the Free Software Foundation; either version 2
of the License, or (at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

Liquid War homepage : http://www.ufoot.org/liquidwar
Contact author      : ufoot@ufoot.org
*/

#include <stdio.h>

#include <GL/glu.h>
#include "../liquidwar6common.h"
#include "../liquidwar6gfx.h"
#include "internal.h"

/*
 * A bunch of code here has been ripped from SDL examples,
 * mostly testgl.c
 */

/* 
 * Quick utility function for texture creation
 */
static int power_of_two(int input)
{
  int value = 1;

  while ( value < input ) {
    value <<= 1;
  }
  return value;
}

/*
 * Returns the number / the closest power of 2
 */
float _lw6gfx_texture_scale(int size) {
  return ((float) size)/((float) power_of_two(size));
}

/*
 * Transforms an SDL bitmap into a proper GL texture.
 */
static GLuint surface2texture_xywh(SDL_Surface *surface, int x, int y, int w, int h)
{
  GLuint texture;
  SDL_Surface *image;
  SDL_Rect dst_area;
  SDL_Rect src_area;
  Uint32 saved_flags;
  Uint8  saved_alpha;
  SDL_Color black={0,0,0};

  w = power_of_two(w);
  h = power_of_two(h);

  image = _lw6gfx_create_surface(w,h);
  //_lw6gfx_clear_surface(image,black);

  if ( image == NULL ) {
    return 0;
  }

  /* Save the alpha blending attributes */
  saved_flags = surface->flags&(SDL_SRCALPHA|SDL_RLEACCELOK);
  saved_alpha = surface->format->alpha;
  if ( (saved_flags & SDL_SRCALPHA) == SDL_SRCALPHA ) {
    SDL_SetAlpha(surface, 0, 0);
  }

  /* Copy the surface into the GL texture image */
  src_area.x = 0;
  src_area.y = 0;
  src_area.w = surface->w;
  src_area.h = surface->h;
  dst_area.x = x;
  dst_area.y = y;
  dst_area.w = surface->w-x;
  dst_area.h = surface->h-y;
  SDL_BlitSurface(surface, &src_area, image, &dst_area);

  /* Restore the alpha blending attributes */
  if ( (saved_flags & SDL_SRCALPHA) == SDL_SRCALPHA ) {
    SDL_SetAlpha(surface, saved_flags, saved_alpha);
  }

  /* Create an OpenGL texture for the image */
  glGenTextures(1, &texture);
  glBindTexture(GL_TEXTURE_2D, texture);
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
  glTexImage2D(GL_TEXTURE_2D,
	       0,
	       GL_RGBA,
	       w, h,
	       0,
	       GL_RGBA,
	       GL_UNSIGNED_BYTE,
	       image->pixels);
  SDL_FreeSurface(image); /* No longer needed */

  return texture;
}

/*
 * Transforms an SDL bitmap into a proper GL texture.
 */
static GLuint surface2texture_wh(SDL_Surface *surface, int w, int h)
{
  return surface2texture_xywh(surface,0,0,w,h);
}

/*
 * Transforms an SDL bitmap into a proper GL texture.
 */
static GLuint surface2texture(SDL_Surface *surface)
{
  int w, h;

  /* Use the surface width and height expanded to powers of 2 */
  w = power_of_two(surface->w);
  h = power_of_two(surface->h);

  return surface2texture_wh(surface,w,h);
}

/*
 * Converts a surface to a texture. Public wrapper function.
 */
GLuint _lw6gfx_surface2texture_xywh(_LW6GFX_CONTEXT *context, SDL_Surface *surface, int x, int y, int w, int h) {
  GLuint texture;

  texture=surface2texture_xywh(surface,x,y,w,h);
  if (!glIsTexture(texture)) {
    lw6sys_log_warning("gfx",_("unable to convert surface to texture"));
  }
  
  return (int) texture;
}

/*
 * Converts a surface to a texture. Public wrapper function.
 */
GLuint _lw6gfx_surface2texture_wh(_LW6GFX_CONTEXT *context, SDL_Surface *surface, int w, int h) {
  GLuint texture;

  texture=surface2texture_wh(surface,w,h);
  if (!glIsTexture(texture)) {
    lw6sys_log_warning("gfx",_("unable to convert surface to texture"));
  }
  
  return (int) texture;
}

/*
 * Converts a surface to a texture. Public wrapper function.
 */
GLuint _lw6gfx_surface2texture(_LW6GFX_CONTEXT *context, SDL_Surface *surface) {
  GLuint texture;

  texture=surface2texture(surface);
  if (!glIsTexture(texture)) {
    lw6sys_log_warning("gfx",_("unable to convert surface to texture"));
  }
  
  return (int) texture;
}

/*
 * Deletes a texture.
 * Note that calling this directly from just after a glEnd
 * when you're drawing textured stuff might cause ugly unexpected
 * results. Not enough time to find out how OpenGL works internally,
 * my workarround is to create textures, use them, and when glFinish
 * has been called, then we delete all textures at once. In short:
 * use _lw6gfx_schedule_delete_texture instead.
 */
void _lw6gfx_delete_texture(_LW6GFX_CONTEXT *context, GLuint texture) {
  if (glIsTexture((GLuint) texture)) {
    glDeleteTextures(1, (GLuint *) &texture);
  } else {
    lw6sys_log_warning("gfx","trying to delete texture, but it isn't a texture");
  }
}

/*
 * Schedules a texture to be deleted by _lw6gfx_flush_delete_texture
 */
void _lw6gfx_schedule_delete_texture(_LW6GFX_CONTEXT *context, GLuint texture) {
  _LW6GFX_TEXTURE_CHAIN *next;

  next=context->texture_data.to_delete;

  context->texture_data.to_delete=LW6SYS_MALLOC(sizeof(_LW6GFX_TEXTURE_CHAIN));
  if (context->texture_data.to_delete) {
    context->texture_data.to_delete->texture=texture;
    context->texture_data.to_delete->next=(void *) next;
  }
}

/*
 * Deletes all textures scheduled for delete.
 */
void _lw6gfx_delete_scheduled_textures(_LW6GFX_CONTEXT *context) {
  _LW6GFX_TEXTURE_CHAIN *to_delete;

  while (context->texture_data.to_delete!=NULL) {
    to_delete=context->texture_data.to_delete;
    context->texture_data.to_delete=(_LW6GFX_TEXTURE_CHAIN *) to_delete->next;
    _lw6gfx_delete_texture(context,to_delete->texture);
    LW6SYS_FREE(to_delete);
  }
}

