/*
  Liquid War is a multiplayer wargame.
  Copyright (C) 2005  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

  Liquid War homepage : http://www.ufoot.org/liquidwar
  Contact author      : ufoot@ufoot.org
*/

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <locale.h>

#include <GL/glu.h>
#include <SDL/SDL_image.h>
#include <expat.h>
#include "../liquidwar6common.h"
#include "../liquidwar6gfx.h"
#include "internal.h"

#define CONST_FILE_1 "gfx/const.xml"
#define CONST_FILE_2 "data/gfx/const.xml"
#define CONST_FILE_3 "../data/gfx/const.xml"
#define CONST_FILE_4 LW6_DATA_DIR "gfx/const.xml"
#define CONST_FILE_DEFAULT LW6_DATA_DIR "gfx/const.xml"

#define XML_BUF_SIZE 4096

static void read_color(char *xml_id, char *xml_value,
		       char *target_id, _LW6GFX_COLOR *color) {
  /*
   * Seems weird to use strcmp here to compare xml_id and
   * target_id, in fact the purpose is to put the test here
   * and simplify the caller's code. Simply call read_color,
   * and if xml_id and target_id are different, it will
   * simply do nothing. It avoids copy/pasting zillions of
   * strcmp calls whenever read_color is called... 
   */
  if (!strcmp(xml_id,target_id)) {
    if (xml_value[0]=='#') {
      char buf[3];

      /*
       * Many calls "if (color)" here, this enables parsing
       * the file for correctness even when we don't want
       * to actually store the value.
       */
      if (color) {
 	// default alpha = "solid" when not defined explicitly
	color->alpha=1.0f;
      }

      memset(buf,0,3);
      switch (strlen(xml_value+1)) {
      case 4:
	if (color) {
	  buf[0]=xml_value[4];
	  color->alpha=(strtol(buf,NULL,16)<<4)/256.0f;
	}
	// Important *NOT* to put a break here.
      case 3:
	if (color) {
	  buf[0]=xml_value[1];
	  color->rgb.r=strtol(buf,NULL,16)<<4;
	  buf[0]=xml_value[2];
	  color->rgb.g=strtol(buf,NULL,16)<<4;
	  buf[0]=xml_value[3];
	  color->rgb.b=strtol(buf,NULL,16)<<4;
	}
	break;
      case 8:
	if (color) {
	  buf[0]=xml_value[7];
	  buf[1]=xml_value[8];
	  color->alpha=strtol(buf,NULL,16)/256.0f;
	}
	// Important *NOT* to put a break here.
      case 6:
	if (color) {
	  buf[0]=xml_value[1];
	  buf[1]=xml_value[2];
	  color->rgb.r=strtol(buf,NULL,16);
	  buf[0]=xml_value[3];
	  buf[1]=xml_value[4];
	  color->rgb.g=strtol(buf,NULL,16);
	  buf[0]=xml_value[5];
	  buf[1]=xml_value[6];
	  color->rgb.b=strtol(buf,NULL,16);
	}
	break;
      default:
	lw6sys_log_warning("gfx",_("invalid color value \"%s\", color format must be \"#RGB\", \"#RGBA\", \"#RRGGBB\" or \"#RRGGBBAA\""), xml_value);	
      }
    } else {
      lw6sys_log_warning("gfx",_("invalid color value \"%s\", color value must start with \"#\""), xml_value);
    }
  }
}

static void read_int(char *xml_id, char *xml_value,
		     char *target_id, int *value) {
  if (!strcmp(xml_id,target_id)) {
    if (value) {
      (*value)=strtol(xml_value,NULL,10);
    }
  }  
}

static void read_float(char *xml_id, char *xml_value,
		       char *target_id, float *value) {
  if (!strcmp(xml_id,target_id)) {
    if (value) {
      char *old_locale;

      /*
       * Important to set locale to C when calling atof,
       * otherwise it returns garbage. Of course the XML
       * file needs to be the same on every computer,
       * and does not depend on locale.
       */
      old_locale=setlocale(LC_ALL,NULL);
      setlocale(LC_ALL,"C");
      /*
       * For some reason strtof doesn't work...
       */
      //(*value)=strtof(xml_value,NULL);
      (*value)=atof(xml_value);
      setlocale(LC_ALL,old_locale);
    }
  }  
}

static void XMLCALL
element_start(void *data, const char *el, const char **attr)
{
  _LW6GFX_CONST_DATA *const_data;  
  char *id=NULL;
  char *value=NULL;
  char **at;

  const_data=(_LW6GFX_CONST_DATA *) data;

  for (at=(char **) attr; (*at)!=NULL; at+=2) {
    if (strcmp("id",(*at))==0) {
      id=*(at+1);
    }
    if (strcmp("value",(*at))==0) {
      value=*(at+1);
    }
  }

  if (id && value && !strcmp(el,"int")) {
    read_int(id,value,"menu_max_displayed_items",&const_data->menu_max_displayed_items);
    read_int(id,value,"menucyl_slices1",&const_data->menucyl_slices1);
    read_int(id,value,"menucyl_slices_min",&const_data->menucyl_slices_min);
    read_int(id,value,"menucyl_stacks",&const_data->menucyl_stacks);
    read_int(id,value,"menucyl_oscil_period",&const_data->menucyl_oscil_period);
    read_int(id,value,"system_font_dw",&const_data->system_font_dw);
    read_int(id,value,"system_font_dh",&const_data->system_font_dh);
    read_int(id,value,"background_nb_bubbles",&const_data->background_nb_bubbles);
  }

  if (id && value && !strcmp(el,"float")) {
    read_float(id,value,"persp_fovy",&const_data->persp_fovy);
    read_float(id,value,"persp_znear",&const_data->persp_znear);
    read_float(id,value,"persp_zfar",&const_data->persp_zfar);
    read_float(id,value,"menucyl_radius1",&const_data->menucyl_radius1);
    read_float(id,value,"menucyl_between1",&const_data->menucyl_between1);
    read_float(id,value,"menucyl_height",&const_data->menucyl_height);
    read_float(id,value,"menucyl_oscil_range1",&const_data->menucyl_oscil_range1);
    read_float(id,value,"system_font_hcoef",&const_data->system_font_hcoef);
    read_float(id,value,"background_yspeed",&const_data->background_yspeed);
    read_float(id,value,"background_bubble_yspeed",&const_data->background_bubble_yspeed);
    read_float(id,value,"background_bubble_size_min",&const_data->background_bubble_size_min);
    read_float(id,value,"background_bubble_size_max",&const_data->background_bubble_size_max);
  }

  if (id && value && !strcmp(el,"color")) {
    read_color(id,value,"menu_fg",&const_data->menu_fg);
    read_color(id,value,"menu_bg",&const_data->menu_bg);
    read_color(id,value,"menu_fg_selected",&const_data->menu_fg_selected);
    read_color(id,value,"menu_bg_selected",&const_data->menu_bg_selected);
    read_color(id,value,"system_font_fg",&const_data->system_font_fg);
    read_color(id,value,"system_font_shade",&const_data->system_font_shade);
  }
}

static void XMLCALL
element_end(void *data, const char *el)
{

}

/*
 * Loads constants.
 */
int _lw6gfx_load_consts(_LW6GFX_CONTEXT *context) {
  int ret=0;
  XML_Parser parser;  
  char buf[XML_BUF_SIZE];
  FILE *f;
  char *const_file=NULL;

  if (lw6sys_file_exists(CONST_FILE_1)) {
    const_file=CONST_FILE_1;
  } else if (lw6sys_file_exists(CONST_FILE_2)) {
    const_file=CONST_FILE_2;
  } else if (lw6sys_file_exists(CONST_FILE_3)) {
    const_file=CONST_FILE_3;
  } else if (lw6sys_file_exists(CONST_FILE_4)) {
    const_file=CONST_FILE_4;
  }
  if (!const_file) {
    const_file=CONST_FILE_DEFAULT;
  }

  lw6sys_log_info("gfx", _("reading \"%s\""), const_file);

  f=fopen(const_file,"r");

  if (f!=NULL) {
    parser = XML_ParserCreate(NULL);
    if (parser) {
      XML_SetElementHandler(parser, element_start, element_end);
      XML_SetUserData(parser, (void *) &context->const_data);
      ret=1;
      
      for (;;) {
	int done;
	int len;
	len = fread(buf, 1, XML_BUF_SIZE, f);
	if (ferror(f)) {
	  lw6sys_log_warning("gfx",_("error reading XML file \"%s\""),const_file);
	  ret=0;
	  break;
	}
	done = feof(f);
	
	if (XML_Parse(parser, buf, len, done) == XML_STATUS_ERROR) {
	  lw6sys_log_warning("gfx",_("parse error reading \"%s\" at line %d: \"%s\""),
			     const_file,
			     XML_GetCurrentLineNumber(parser),
			     XML_ErrorString(XML_GetErrorCode(parser)));
	  ret=0;
	  break;
	}
	
	if (done) {
	  break;
	}
      }

      XML_ParserFree(parser);
    } else {
      lw6sys_log_warning("gfx",_("unable to create parser"));
    }
  } else {
    lw6sys_log_warning("gfx",_("unable to open file \"%s\""),const_file);
  }

  if (!ret) {
    exit(1);
  }

  return ret;  
}

int lw6gfx_load_consts(void *context) {
  return _lw6gfx_load_consts((_LW6GFX_CONTEXT *) context);
}

/*
 * Unload constants, free memory
 */
void _lw6gfx_unload_consts(_LW6GFX_CONTEXT *context) {
  memset(&context->const_data,0,sizeof(_LW6GFX_CONST_DATA));   
}

void lw6gfx_unload_consts(void *context) {
  _lw6gfx_unload_consts((_LW6GFX_CONTEXT *) context);
}



