/*
  Liquid War is a multiplayer wargame.
  Copyright (C) 2005  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

  Liquid War homepage : http://www.ufoot.org/liquidwar
  Contact author      : ufoot@ufoot.org
*/

#include <stdlib.h>
#include <string.h>

#include <libguile.h>

#include "liquidwar6common.h"
#include "liquidwar6.h"

/*
 * For a GNU gettext-like behavior of scheme code.
 */
static SCM scm_gettext(SCM string) {
  char *c_string;
  int length;
  SCM ret;

  SCM_ASSERT (SCM_STRINGP(string), 
	      string, 
	      SCM_ARG1, 
	      "c-gettext");
  
  length=SCM_STRING_LENGTH(string);
  c_string=(char *) LW6SYS_MALLOC((length+1)*sizeof(char));
  if (c_string) {
    memcpy((void *) c_string, (void *) SCM_STRING_CHARS(string), length*sizeof(char));
    c_string[length]='\0';
    ret=scm_makfrom0str(gettext(c_string));
    LW6SYS_FREE(c_string);
  } else {
    ret=SCM_UNDEFINED;
  }

  return ret;
}

/*
 * In liquidwar6sys
 */ 

/*
 * In log.c
 */

static SCM scm_lw6sys_log_debug(SCM message) {
  char *c_message;
  int length;

  SCM_ASSERT (SCM_STRINGP(message), 
	      message, 
	      SCM_ARG1, 
	      "c-lw6sys-log-debug");
  
  length=SCM_STRING_LENGTH(message);
  c_message=(char *) LW6SYS_MALLOC((length+1)*sizeof(char));
  if (c_message) {
    memcpy((void *) c_message, (void *) SCM_STRING_CHARS(message), length*sizeof(char));
    c_message[length]='\0';
    lw6sys_log_debug("script","%s",c_message);
    LW6SYS_FREE(c_message);
  }

  return SCM_UNDEFINED;
}

static SCM scm_lw6sys_log_info(SCM message) {
  char *c_message;
  int length;
  
  SCM_ASSERT (SCM_STRINGP(message), 
	      message, 
	      SCM_ARG1, 
	      "c-lw6sys-log-info");
  
  length=SCM_STRING_LENGTH(message);
  c_message=(char *) LW6SYS_MALLOC((length+1)*sizeof(char));
  if (c_message) {
    memcpy((void *) c_message, (void *) SCM_STRING_CHARS(message), length*sizeof(char));
    c_message[length]='\0';
    lw6sys_log_info("script","%s",c_message);
    LW6SYS_FREE(c_message);
  }

  return SCM_UNDEFINED;
}

static SCM scm_lw6sys_log_warning(SCM message) {
  char *c_message;
  int length;
  
  SCM_ASSERT (SCM_STRINGP(message), 
	      message, 
	      SCM_ARG1, 
	      "c-lw6sys-log-warning");
  
  length=SCM_STRING_LENGTH(message);
  c_message=(char *) LW6SYS_MALLOC((length+1)*sizeof(char));
  if (c_message) {
    memcpy((void *) c_message, (void *) SCM_STRING_CHARS(message), length*sizeof(char));
    c_message[length]='\0';
    lw6sys_log_warning("script","%s",c_message);
    LW6SYS_FREE(c_message);
  }

  return SCM_UNDEFINED;
}

static SCM scm_lw6sys_log_error(SCM message) {
  char *c_message;
  int length;
  
  SCM_ASSERT (SCM_STRINGP(message), 
	      message, 
	      SCM_ARG1, 
	      "c-lw6sys-log-error");
  
  length=SCM_STRING_LENGTH(message);
  c_message=(char *) LW6SYS_MALLOC((length+1)*sizeof(char));
  if (c_message) {
    memcpy((void *) c_message, (void *) SCM_STRING_CHARS(message), length*sizeof(char));
    c_message[length]='\0';
    lw6sys_log_error("script","%s",c_message);
    LW6SYS_FREE(c_message);
  }

  return SCM_UNDEFINED;
}

/*
 * In liquidwar6gfx
 */ 

/*
 * In background.c
 */
static SCM scm_lw6gfx_display_background() {
  if (lw6_global_gfx_context!=NULL) {
    lw6gfx_display_background(lw6_global_gfx_context);
  }

  return SCM_UNDEFINED;  
}

/*
 * In const.c
 */
static SCM scm_lw6gfx_load_consts() {
  SCM ret;

  if (lw6_global_gfx_context!=NULL) {
    ret=lw6gfx_load_consts(lw6_global_gfx_context) ? SCM_BOOL_T : SCM_BOOL_F;
  } else {
    ret=SCM_BOOL_F;
  }
  
  return ret;
}

static SCM scm_lw6gfx_unload_consts() {
  if (lw6_global_gfx_context!=NULL) {
    lw6gfx_unload_consts(lw6_global_gfx_context);
  }
  
  return SCM_UNDEFINED;
}

/*
 * In data.c
 */
static SCM scm_lw6gfx_load_images() {
  SCM ret;

  if (lw6_global_gfx_context!=NULL) {
    ret=lw6gfx_load_images(lw6_global_gfx_context) ? SCM_BOOL_T : SCM_BOOL_F;
  } else {
    ret=SCM_BOOL_F;
  }
  
  return ret;
}

static SCM scm_lw6gfx_unload_images() {
  if (lw6_global_gfx_context!=NULL) {
    lw6gfx_unload_images(lw6_global_gfx_context);
  }
  
  return SCM_UNDEFINED;
}

static SCM scm_lw6gfx_load_textures() {
  SCM ret;

  if (lw6_global_gfx_context!=NULL) {
    ret=lw6gfx_load_textures(lw6_global_gfx_context) ? SCM_BOOL_T : SCM_BOOL_F;
  } else {
    ret=SCM_BOOL_F;
  }
  
  return ret;
}

static SCM scm_lw6gfx_unload_textures() {
  if (lw6_global_gfx_context!=NULL) {
    lw6gfx_unload_textures(lw6_global_gfx_context);
  }
  
  return SCM_UNDEFINED;
}

static SCM scm_lw6gfx_load_fonts() {
  SCM ret;

  if (lw6_global_gfx_context!=NULL) {
    ret=lw6gfx_load_fonts(lw6_global_gfx_context) ? SCM_BOOL_T : SCM_BOOL_F;
  } else {
    ret=SCM_BOOL_F;
  }
  
  return ret;
}

static SCM scm_lw6gfx_unload_fonts() {
  if (lw6_global_gfx_context!=NULL) {
    lw6gfx_unload_fonts(lw6_global_gfx_context);
  }
  
  return SCM_UNDEFINED;
}

/*
 * In event.c
 */
static SCM scm_lw6gfx_poll_keypress() {
  SCM ret;

  if (lw6_global_gfx_context!=NULL) {
    LW6GFX_KEYPRESS keypress;

    if (lw6gfx_poll_keypress(lw6_global_gfx_context,&keypress)) {
      ret=scm_list_3(scm_cons(scm_makfrom0str("sym"),scm_int2num(keypress.sym)),
		     scm_cons(scm_makfrom0str("unicode"),scm_int2num(keypress.unicode)),
		     scm_cons(scm_makfrom0str("label"),scm_makfrom0str(keypress.label)));
    } else {
      ret=SCM_BOOL_F;
    }
  } else {
    ret=SCM_BOOL_F;
  }
  
  return ret;
}

static SCM scm_lw6gfx_keysym_esc() {
  SCM ret;

  if (lw6_global_gfx_context!=NULL) {
    ret=scm_int2num(lw6gfx_keysym_esc(lw6_global_gfx_context));
  } else {
    ret=SCM_UNDEFINED;
  }

  return ret;
}

static SCM scm_lw6gfx_keysym_up() {
  SCM ret;

  if (lw6_global_gfx_context!=NULL) {
    ret=scm_int2num(lw6gfx_keysym_up(lw6_global_gfx_context));
  } else {
    ret=SCM_UNDEFINED;
  }

  return ret;
}

static SCM scm_lw6gfx_keysym_down() {
  SCM ret;

  if (lw6_global_gfx_context!=NULL) {
    ret=scm_int2num(lw6gfx_keysym_down(lw6_global_gfx_context));
  } else {
    ret=SCM_UNDEFINED;
  }

  return ret;
}

static SCM scm_lw6gfx_keysym_left() {
  SCM ret;

  if (lw6_global_gfx_context!=NULL) {
    ret=scm_int2num(lw6gfx_keysym_left(lw6_global_gfx_context));
  } else {
    ret=SCM_UNDEFINED;
  }

  return ret;
}

static SCM scm_lw6gfx_keysym_right() {
  SCM ret;

  if (lw6_global_gfx_context!=NULL) {
    ret=scm_int2num(lw6gfx_keysym_right(lw6_global_gfx_context));
  } else {
    ret=SCM_UNDEFINED;
  }

  return ret;
}

static SCM scm_lw6gfx_keysym_enter() {
  SCM ret;

  if (lw6_global_gfx_context!=NULL) {
    ret=scm_int2num(lw6gfx_keysym_enter(lw6_global_gfx_context));
  } else {
    ret=SCM_UNDEFINED;
  }

  return ret;
}

static SCM scm_lw6gfx_keysym_del() {
  SCM ret;

  if (lw6_global_gfx_context!=NULL) {
    ret=scm_int2num(lw6gfx_keysym_del(lw6_global_gfx_context));
  } else {
    ret=SCM_UNDEFINED;
  }

  return ret;
}

static SCM scm_lw6gfx_keysym_backspace() {
  SCM ret;

  if (lw6_global_gfx_context!=NULL) {
    ret=scm_int2num(lw6gfx_keysym_backspace(lw6_global_gfx_context));
  } else {
    ret=SCM_UNDEFINED;
  }

  return ret;
}

static SCM scm_lw6gfx_keysym_help() {
  SCM ret;

  if (lw6_global_gfx_context!=NULL) {
    ret=scm_int2num(lw6gfx_keysym_help(lw6_global_gfx_context));
  } else {
    ret=SCM_UNDEFINED;
  }

  return ret;
}

static SCM scm_lw6gfx_keysym_quit() {
  SCM ret;

  if (lw6_global_gfx_context!=NULL) {
    ret=scm_int2num(lw6gfx_keysym_quit(lw6_global_gfx_context));
  } else {
    ret=SCM_UNDEFINED;
  }

  return ret;
}

static SCM scm_lw6gfx_poll_quit() {
  SCM ret;

  if (lw6_global_gfx_context!=NULL) {
    ret=lw6gfx_poll_quit(lw6_global_gfx_context) ? SCM_BOOL_T : SCM_BOOL_F;
  } else {
    ret=SCM_BOOL_F;
  }

  return ret;
}

/*
 * In info.c
 */
static SCM scm_lw6gfx_display_sysinfo(SCM fps, SCM rps, SCM bytes_in, SCM bytes_out) {
  if (lw6_global_gfx_context!=NULL) {
    int c_fps;
    int c_rps;
    int c_bytes_in;
    int c_bytes_out;

    SCM_ASSERT (SCM_INUMP(fps), 
		fps, 
		SCM_ARG1, 
		"c-lw6sys-display-sysinfo");
    SCM_ASSERT (SCM_INUMP(rps), 
	        rps, 
		SCM_ARG2, 
		"c-lw6sys-display-sysinfo");
    SCM_ASSERT (SCM_INUMP(bytes_in), 
		bytes_in, 
		SCM_ARG3, 
		"c-lw6sys-display-sysinfo");
    SCM_ASSERT (SCM_INUMP(bytes_out), 
		bytes_out, 
		SCM_ARG4, 
		"c-lw6sys-display-sysinfo");

    c_fps=SCM_INUM(fps);
    c_rps=SCM_INUM(rps);
    c_bytes_in=SCM_INUM(bytes_in);
    c_bytes_out=SCM_INUM(bytes_out);

    _lw6gfx_display_sysinfo(lw6_global_gfx_context,c_fps,c_rps,c_bytes_in,c_bytes_out);
  }
 
  return SCM_UNDEFINED;
}

static SCM scm_lw6gfx_display_log(SCM log) {
  if (lw6_global_gfx_context!=NULL) {
    int log_size;
    char **c_log;
    SCM_ASSERT (SCM_CONSP(log), 
		log, 
		SCM_ARG1, 
		"c-lw6sys-display-log");
    log_size=SCM_INUM(scm_length(log));
    c_log=LW6SYS_MALLOC((log_size+1)*sizeof(char *));
    if (c_log) {
      int i;

      for (i=0;i<log_size;++i) {
	SCM str;
	int str_length;

	str=scm_list_ref(log,scm_int2num(i));
	str_length=SCM_STRING_LENGTH(str);
	c_log[i]=LW6SYS_MALLOC((str_length+1)*sizeof(char));
	if (c_log[i]) {
	  memcpy((void *) c_log[i], (void *) SCM_STRING_CHARS(str), str_length*sizeof(char));
	  c_log[i][str_length]='\0';
	}
      }
      c_log[log_size]=NULL;

      _lw6gfx_display_log(lw6_global_gfx_context,c_log);

      for (i=0;i<log_size;++i) {
	if (c_log[i]) {
	  LW6SYS_FREE(c_log[i]);
	}      
      }

      LW6SYS_FREE(c_log);
    }
  }
 
  return SCM_UNDEFINED;
}

/*
 * In menu.c
 */
static SCM scm_lw6gfx_display_menu(SCM menu) {
  if (lw6_global_gfx_context!=NULL) {
    LW6GFX_MENU c_menu;
    SCM items;

    memset(&c_menu,0,sizeof(LW6GFX_MENU));

    SCM_ASSERT (SCM_CONSP(menu), 
		menu, 
		SCM_ARG1, 
		"c-lw6gfx-display-menu");

    //scm_display(menu,scm_current_output_port ());

    c_menu.type=SCM_INUM(scm_assoc_ref(menu,scm_makfrom0str("type")));
    c_menu.selected_item=SCM_INUM(scm_assoc_ref(menu,scm_makfrom0str("selected-item")));
    items=scm_assoc_ref(menu,scm_makfrom0str("items"));
    c_menu.nb_items=SCM_INUM(scm_length(items));

    c_menu.items=LW6SYS_MALLOC(c_menu.nb_items*sizeof(LW6GFX_MENUITEM));
    if (c_menu.items) {
      int i;
      
      for (i=0;i<c_menu.nb_items;++i) {
	SCM item;      
	int label_length;
	SCM label;

	item=scm_list_ref(items,scm_int2num(i));
	  
	c_menu.items[i].type=SCM_INUM(scm_assoc_ref(item,scm_makfrom0str("type")));
	c_menu.items[i].enabled=SCM_INUM(scm_assoc_ref(item,scm_makfrom0str("enabled")));
	c_menu.items[i].value=SCM_INUM(scm_assoc_ref(item,scm_makfrom0str("value")));
	c_menu.items[i].selected_char=SCM_INUM(scm_assoc_ref(item,scm_makfrom0str("selected-char")));
	label=scm_assoc_ref(item,scm_makfrom0str("label"));

	label_length=SCM_STRING_LENGTH(label);
	c_menu.items[i].label=(char *) LW6SYS_MALLOC((label_length+1)*sizeof(char));
	if (c_menu.items[i].label) {
	  memcpy((void *) c_menu.items[i].label, (void *) SCM_STRING_CHARS(label), label_length*sizeof(char));
	  c_menu.items[i].label[label_length]='\0';
	}
	
	//scm_display(item,scm_current_output_port ());
      }
      
      lw6gfx_display_menu(lw6_global_gfx_context,&c_menu);	

      for (i=0;i<c_menu.nb_items;++i) {
	if (c_menu.items[i].label) {
	  LW6SYS_FREE(c_menu.items[i].label);
	}
      }

      LW6SYS_FREE(c_menu.items);
    }
  }

  return SCM_UNDEFINED;
}

/*
 * In render.c
 */
static SCM scm_lw6gfx_prepare_buffer() {
  lw6gfx_prepare_buffer(lw6_global_gfx_context);

  return SCM_UNDEFINED;  
}

static SCM scm_lw6gfx_swap_buffers() {
  if (lw6_global_gfx_context!=NULL) {
    lw6gfx_swap_buffers(lw6_global_gfx_context);
  }

  return SCM_UNDEFINED;  
}

/*
 * In setup.c
 */
static SCM scm_lw6gfx_init() {
  SCM ret;
  
  lw6_global_gfx_context=lw6gfx_init();
  
  ret = lw6_global_gfx_context? SCM_BOOL_T : SCM_BOOL_F;

  return ret;
}

static SCM scm_lw6gfx_set_video_mode(SCM width, SCM height, SCM fullscreen) {
  SCM ret;

  if (lw6_global_gfx_context!=NULL) {
    int c_width;
    int c_height;
    int c_fullscreen;
  
    SCM_ASSERT (SCM_INUMP(width), 
		width, 
		SCM_ARG1, 
		"c-lw6gfx-set-video-mode");
    
    SCM_ASSERT (SCM_INUMP(height), 
		height, 
		SCM_ARG2, 
		"c-lw6gfx-set-video-mode");
    
    c_width=SCM_INUM(width);
    c_height=SCM_INUM(height);
    c_fullscreen=SCM_NFALSEP(fullscreen);
  
    ret=lw6gfx_set_video_mode(lw6_global_gfx_context,c_width,c_height,c_fullscreen) ? SCM_BOOL_T : SCM_BOOL_F;
  } else {
    ret=SCM_BOOL_F;
  }

  return ret;
}

static SCM scm_lw6gfx_quit() {
  if (lw6_global_gfx_context!=NULL) {
    lw6gfx_quit(lw6_global_gfx_context);
  }

  return SCM_UNDEFINED;
}

/*
 * In timer.c
 */
static SCM scm_lw6gfx_delay(SCM ms) {
  int c_ms;

  if (lw6_global_gfx_context!=NULL) {
    SCM_ASSERT (SCM_INUMP(ms), 
		ms, 
		SCM_ARG1, 
		"c-lw6gfx-delay");
    
    c_ms=SCM_INUM(ms);
    
    lw6gfx_delay(lw6_global_gfx_context,c_ms);
  }

  return SCM_UNDEFINED;
}

static SCM scm_lw6gfx_get_ticks(void *context) {
  SCM ret;

  if (lw6_global_gfx_context!=NULL) {
    ret=scm_int2num(lw6gfx_get_ticks(context));
  }

  return ret;
}

/*
 * The function which registers everything.
 */
int lw6_register_scm() {
  int ret=1;

  scm_c_define_gsubr("c-gettext", 
		     1, 0, 0, 
		     (SCM (*)()) scm_gettext); 

  scm_c_define_gsubr("_", 
		     1, 0, 0, 
		     (SCM (*)()) scm_gettext); 

  /*
   * In liquidwar6sys
   */

  /*
   * In log.c
   */
  scm_c_define_gsubr("c-lw6sys-log-debug", 
		     1, 0, 0, 
		     (SCM (*)()) scm_lw6sys_log_debug); 

  scm_c_define_gsubr("c-lw6sys-log-info", 
		     1, 0, 0, 
		     (SCM (*)()) scm_lw6sys_log_info); 

  scm_c_define_gsubr("c-lw6sys-log-warning", 
		     1, 0, 0, 
		     (SCM (*)()) scm_lw6sys_log_warning); 

  scm_c_define_gsubr("c-lw6sys-log-error", 
		     1, 0, 0, 
		     (SCM (*)()) scm_lw6sys_log_error); 

  /*
   * In liquidwar6gfx
   */

  /*
   * In background.c
   */
  scm_c_define_gsubr("c-lw6gfx-display-background",
		     0, 0, 0,
		     (SCM (*)()) scm_lw6gfx_display_background); 

  /*
   * In const.c
   */
  scm_c_define_gsubr("c-lw6gfx-load-consts",
		     0, 0, 0,
		     (SCM (*)()) scm_lw6gfx_load_consts); 

  scm_c_define_gsubr("c-lw6gfx-unload-consts",
		     0, 0, 0,
		     (SCM (*)()) scm_lw6gfx_unload_consts); 

  /*
   * In data.c
   */
  scm_c_define_gsubr("c-lw6gfx-load-images",
		     0, 0, 0,
		     (SCM (*)()) scm_lw6gfx_load_images); 

  scm_c_define_gsubr("c-lw6gfx-unload-images",
		     0, 0, 0,
		     (SCM (*)()) scm_lw6gfx_unload_images); 

  scm_c_define_gsubr("c-lw6gfx-load-textures",
		     0, 0, 0,
		     (SCM (*)()) scm_lw6gfx_load_textures); 

  scm_c_define_gsubr("c-lw6gfx-unload-textures",
		     0, 0, 0,
		     (SCM (*)()) scm_lw6gfx_unload_textures); 

  scm_c_define_gsubr("c-lw6gfx-load-fonts",
		     0, 0, 0,
		     (SCM (*)()) scm_lw6gfx_load_fonts); 

  scm_c_define_gsubr("c-lw6gfx-unload-fonts",
		     0, 0, 0,
		     (SCM (*)()) scm_lw6gfx_unload_fonts); 

  /*
   * In event.c
   */
  scm_c_define_gsubr("c-lw6gfx-poll-keypress",
		     0, 0, 0,
		     (SCM (*)()) scm_lw6gfx_poll_keypress); 

  scm_c_define_gsubr("c-lw6gfx-keysym-esc",
		     0, 0, 0,
		     (SCM (*)()) scm_lw6gfx_keysym_esc); 

  scm_c_define_gsubr("c-lw6gfx-keysym-up",
		     0, 0, 0,
		     (SCM (*)()) scm_lw6gfx_keysym_up); 

  scm_c_define_gsubr("c-lw6gfx-keysym-down",
		     0, 0, 0,
		     (SCM (*)()) scm_lw6gfx_keysym_down); 

  scm_c_define_gsubr("c-lw6gfx-keysym-left",
		     0, 0, 0,
		     (SCM (*)()) scm_lw6gfx_keysym_left); 

  scm_c_define_gsubr("c-lw6gfx-keysym-right",
		     0, 0, 0,
		     (SCM (*)()) scm_lw6gfx_keysym_right); 

  scm_c_define_gsubr("c-lw6gfx-keysym-enter",
		     0, 0, 0,
		     (SCM (*)()) scm_lw6gfx_keysym_enter); 

  scm_c_define_gsubr("c-lw6gfx-keysym-del",
		     0, 0, 0,
		     (SCM (*)()) scm_lw6gfx_keysym_del); 

  scm_c_define_gsubr("c-lw6gfx-keysym-backspace",
		     0, 0, 0,
		     (SCM (*)()) scm_lw6gfx_keysym_backspace); 

  scm_c_define_gsubr("c-lw6gfx-keysym-help",
		     0, 0, 0,
		     (SCM (*)()) scm_lw6gfx_keysym_help); 

  scm_c_define_gsubr("c-lw6gfx-keysym-quit",
		     0, 0, 0,
		     (SCM (*)()) scm_lw6gfx_keysym_quit); 

  scm_c_define_gsubr("c-lw6gfx-poll-quit",
		     0, 0, 0,
		     (SCM (*)()) scm_lw6gfx_poll_quit); 

  /*
   * In info.c
   */
  scm_c_define_gsubr("c-lw6gfx-display-sysinfo",
		     4, 0, 0,
		     (SCM (*)()) scm_lw6gfx_display_sysinfo); 

  scm_c_define_gsubr("c-lw6gfx-display-log",
		     1, 0, 0,
		     (SCM (*)()) scm_lw6gfx_display_log); 

  /*
   * In menu.c
   */
  scm_c_define_gsubr("c-lw6gfx-display-menu",
		     1, 0, 0,
		     (SCM (*)()) scm_lw6gfx_display_menu); 

  /* 
   * In render.c
   */
  scm_c_define_gsubr("c-lw6gfx-prepare-buffer",
		     0, 0, 0,
		     (SCM (*)()) scm_lw6gfx_prepare_buffer); 

  scm_c_define_gsubr("c-lw6gfx-swap-buffers",
		     0, 0, 0,
		     (SCM (*)()) scm_lw6gfx_swap_buffers); 

  /*
   * In setup.c
   */
  scm_c_define_gsubr("c-lw6gfx-init", 
		     0, 0, 0, 
		     (SCM (*)()) scm_lw6gfx_init); 

  scm_c_define_gsubr("c-lw6gfx-set-video-mode", 
		     3, 0, 0, 
		     (SCM (*)()) scm_lw6gfx_set_video_mode); 

  scm_c_define_gsubr("c-lw6gfx-quit", 
		     0, 0, 0, 
		     (SCM (*)()) scm_lw6gfx_quit); 

  /*
   * In timer.c
   */
  scm_c_define_gsubr("c-lw6gfx-delay", 
		     1, 0, 0, 
		     (SCM (*)()) scm_lw6gfx_delay); 

  scm_c_define_gsubr("c-lw6gfx-get-ticks", 
		     0, 0, 0, 
		     (SCM (*)()) scm_lw6gfx_get_ticks); 

  return ret;
}
