/*
  Liquid War is a multiplayer wargame.
  Copyright (C) 2005  Christian Mauduit <ufoot@ufoot.org>

  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA  02110-1301, USA.

  Liquid War homepage : http://www.ufoot.org/liquidwar
  Contact author      : ufoot@ufoot.org
*/

#include <stdio.h>
#include <stdlib.h>
#include <math.h>

#include <GL/glu.h>
#include "../liquidwar6common.h"
#include "../liquidwar6gfx.h"
#include "internal.h"

#define TEST_MENU_SIZE 8

/*
 * OpenGL wizardry, to prepare view parameters.
 */
static void prepare_view(_LW6GFX_CONTEXT *context) {
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR);
  glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR);
  glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_S,GL_CLAMP);
  glTexParameteri(GL_TEXTURE_2D,GL_TEXTURE_WRAP_T,GL_CLAMP);
}

/*
 * Display a menu item.
 */
static void draw_cylinder(_LW6GFX_CONTEXT *context, GLuint texture, int i, int n, float ratio) {
  GLUquadricObj *cyl;

  cyl=gluNewQuadric();
  if (cyl!=NULL) {
    float radius;
    float height;
    int slices;
    float y;
    float dy;

    gluQuadricTexture(cyl,GL_TRUE);    

    radius=context->const_data.menucyl_radius1/n;
    height=context->const_data.menucyl_height*((float) context->screen.width)/((float) context->screen.height);
    y=-context->const_data.menucyl_between1*(i-(n-1)/2.0f)/n;
    dy=(context->const_data.menucyl_oscil_range1/n)*sin(2.0f*M_PI*(((((float) lw6gfx_get_ticks(context)))/((float) context->const_data.menucyl_oscil_period))+(((float) i)/((float) n))));
    slices=context->const_data.menucyl_slices1/n;
    if (slices<context->const_data.menucyl_slices_min) {
      slices=context->const_data.menucyl_slices_min;
    }

    glMatrixMode(GL_MODELVIEW);
    glPushMatrix();
    glLoadIdentity();	
    glTranslatef(-height/2.0f,0.0f,-1.0f);
    glRotatef(90.0f,0.0f,1.0f,0.0f);
    glTranslatef(0.0f,y+dy,0.0f);

    gluCylinder(cyl,radius,radius,height,slices,context->const_data.menucyl_stacks);

    glMatrixMode(GL_MODELVIEW);
    glPopMatrix();

    gluDeleteQuadric(cyl);
  } else {
    lw6sys_log_warning("gfx",_("unable to create quadric"));
  }
}

static SDL_Color get_fg_color(_LW6GFX_CONTEXT *context, int select) {
  SDL_Color color;

  if (select) {
    color=context->const_data.menu_fg_selected.rgb;
  } else {
    color=context->const_data.menu_fg.rgb;
  }

  return color;
}

static SDL_Color get_bg_color(_LW6GFX_CONTEXT *context, int select) {
  SDL_Color color;

  if (select) {
    color=context->const_data.menu_bg_selected.rgb;
  } else {
    color=context->const_data.menu_bg.rgb;
  }

  return color;
}

/*
 * Draw a basic text menu item (button).
 */
static void draw_button(_LW6GFX_CONTEXT *context, char *text, int select, int i, int n) {
  SDL_Surface *text_surface;
  SDL_Surface *outline_surface;
  GLuint text_texture;
  char *utf8;
  SDL_Color bg_color;
  SDL_Color fg_color;
  int ret=0;
  SDL_Rect text_rect;
  SDL_Rect outline_rect;

  utf8=lw6sys_locale_to_utf8(text);
  if (utf8!=NULL) {
    fg_color=get_fg_color(context,select);
    bg_color=get_bg_color(context,select);
    text_surface=TTF_RenderUTF8_Shaded(context->font_data.menu, utf8, fg_color, bg_color);
    if (text_surface!=NULL) {
      outline_surface=_lw6gfx_create_surface(text_surface->w+2*_LW6GFX_FONT_MENU_SIZE,_LW6GFX_MENU_TEXTURE_H);
      if (outline_surface!=NULL) {
	text_rect.x=0;
	text_rect.y=0;
	text_rect.w=text_surface->w;
	text_rect.h=text_surface->h;
	outline_rect.x=(outline_surface->w-text_surface->w)/2;
	outline_rect.y=(outline_surface->h-text_surface->h)/2;
	outline_rect.w=outline_surface->w;
	outline_rect.h=outline_surface->h;

	_lw6gfx_clear_surface(outline_surface,bg_color);
	SDL_BlitSurface(text_surface,&text_rect,outline_surface,&outline_rect);

	text_texture=_lw6gfx_surface2texture_xywh(context,outline_surface,(_LW6GFX_MENU_TEXTURE_W-outline_surface->w)/2,(_LW6GFX_MENU_TEXTURE_H-outline_surface->h)/2,_LW6GFX_MENU_TEXTURE_W,_LW6GFX_MENU_TEXTURE_H);    
	if (text_texture) {      
	  glEnable(GL_TEXTURE_2D); 
	  glColor3f(1.0f, 1.0f, 0.0f);
	  glBindTexture(GL_TEXTURE_2D, text_texture);
	  
	  glMatrixMode(GL_TEXTURE);
	  glPushMatrix();
	  glLoadIdentity();
	  glScalef(1.0f,-1.0f,1.0f);
	  glTranslatef(0.0f,-0.25f,0.0f);
	  glRotatef(-90.0f,0.0f,0.0f,1.0f);
	  
	  draw_cylinder(context,text_texture,i,n,((float) _LW6GFX_MENU_TEXTURE_W)/((float) outline_surface->w));
	  
	  glMatrixMode(GL_TEXTURE);
	  glPopMatrix();

	  ret=1;
	  
	  _lw6gfx_schedule_delete_texture(context,text_texture);
	}
	_lw6gfx_delete_surface(outline_surface);
      }		
      _lw6gfx_delete_surface(text_surface);
    }
    LW6SYS_FREE(utf8);
  }
  
  if (!ret) {
    lw6sys_log_warning("gfx",_("unable to draw button \"%s\""),text);
  }
}

/*
 * Display a menu.
 */
void _lw6gfx_display_menu(_LW6GFX_CONTEXT *context, LW6GFX_MENU *menu) {
  int i;
  int selection_offset=0;
  int displayed_items;
  LW6GFX_MENUITEM *menuitem;

  _lw6gfx_set_render_mode_3d_menu(context);

  prepare_view(context);
  
  if (menu->nb_items>context->const_data.menu_max_displayed_items) {
    /*
     * Here there are more items in the menu than the max
     * displayable number. 
     */
    displayed_items=context->const_data.menu_max_displayed_items;

    // We center the menu on thethe selected item with an offset.
    selection_offset=menu->selected_item-displayed_items/2;

    // Checking for offset limits.
    if (selection_offset<0) {
      selection_offset=0;
    }
    if (selection_offset+displayed_items>menu->nb_items) {
      selection_offset=menu->nb_items-displayed_items;
    }
  } else {
    displayed_items=menu->nb_items;
  }

  for (i=0;i<displayed_items;++i) {
    menuitem=&(menu->items[i+selection_offset]);
    switch (menuitem->type) {
    case LW6GFX_MENUITEM_NONE:
      break;
    case LW6GFX_MENUITEM_BUTTON:
      draw_button(context,
		  menuitem->label,
		  i+selection_offset==menu->selected_item,
		  i,
		  displayed_items);
      break;
    case LW6GFX_MENUITEM_EDIT:
      break;
    case LW6GFX_MENUITEM_SLIDER:
      break;
    default:
      lw6sys_log_warning("gfx",_("unknown menuitem type %d"), menuitem->type);
    }
  }
}

void lw6gfx_display_menu(void *context, LW6GFX_MENU *menu) {
  _lw6gfx_display_menu((_LW6GFX_CONTEXT *) context, menu);
}


