/*
	belle-sip - SIP (RFC3261) library.
    Copyright (C) 2010  Belledonne Communications SARL

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "belle_sip_internal.h"
#include "dns.h"

#include <stdlib.h>
#ifdef __APPLE__
#include "TargetConditionals.h"
#endif

#define DNS_EAGAIN  EAGAIN




typedef struct belle_sip_simple_resolver_context belle_sip_simple_resolver_context_t;
#define BELLE_SIP_SIMPLE_RESOLVER_CONTEXT(obj) BELLE_SIP_CAST(obj,belle_sip_simple_resolver_context_t)

typedef struct belle_sip_combined_resolver_context belle_sip_combined_resolver_context_t;
#define BELLE_SIP_COMBINED_RESOLVER_CONTEXT(obj) BELLE_SIP_CAST(obj,belle_sip_combined_resolver_context_t)

typedef struct belle_sip_dual_resolver_context belle_sip_dual_resolver_context_t;
#define BELLE_SIP_DUAL_RESOLVER_CONTEXT(obj) BELLE_SIP_CAST(obj,belle_sip_dual_resolver_context_t)


struct belle_sip_dns_srv{
	belle_sip_object_t base;
	unsigned short priority;
	unsigned short weight;
	unsigned short port;
	char *target;
	belle_sip_combined_resolver_context_t *root_resolver;/* used internally to combine SRV and A queries*/
	belle_sip_resolver_context_t *a_resolver; /* used internally to combine SRV and A queries*/
	struct addrinfo *a_results; /* used internally to combine SRV and A queries*/
	unsigned char a_done;
};

static void belle_sip_dns_srv_destroy(belle_sip_dns_srv_t *obj){
	if (obj->target) {
		belle_sip_free(obj->target);
		obj->target=NULL;
	}
	if (obj->a_resolver){
		belle_sip_resolver_context_cancel(obj->a_resolver);
		obj->a_resolver=NULL;
	}
	if (obj->a_results){
		freeaddrinfo(obj->a_results);
		obj->a_results=NULL;
	}
}

belle_sip_dns_srv_t *belle_sip_dns_srv_create(struct dns_srv *srv){
	belle_sip_dns_srv_t *obj=belle_sip_object_new(belle_sip_dns_srv_t);
	obj->priority=srv->priority;
	obj->weight=srv->weight;
	obj->port=srv->port;
	obj->target=belle_sip_strdup(srv->target);
	return obj;
}

const char *belle_sip_dns_srv_get_target(const belle_sip_dns_srv_t *obj){
	return obj->target;
}

unsigned short belle_sip_dns_srv_get_priority(const belle_sip_dns_srv_t *obj){
	return obj->priority;
}

unsigned short belle_sip_dns_srv_get_weight(const belle_sip_dns_srv_t *obj){
	return obj->weight;
}

unsigned short belle_sip_dns_srv_get_port(const belle_sip_dns_srv_t *obj){
	return obj->port;
}

BELLE_SIP_DECLARE_NO_IMPLEMENTED_INTERFACES(belle_sip_dns_srv_t);
BELLE_SIP_INSTANCIATE_VPTR(belle_sip_dns_srv_t, belle_sip_object_t,belle_sip_dns_srv_destroy, NULL, NULL,TRUE);


struct belle_sip_resolver_context{
	belle_sip_source_t source;
	belle_sip_stack_t *stack;
	uint8_t done;
	uint8_t pad[3];
};

struct belle_sip_simple_resolver_context{
	belle_sip_resolver_context_t base;
	belle_sip_resolver_callback_t cb;
	belle_sip_resolver_srv_callback_t srv_cb;
	void *cb_data;
	void *srv_cb_data;
	struct dns_resolv_conf *resconf;
	struct dns_hosts *hosts;
	struct dns_resolver *R;
	enum dns_type type;
	char *name;
	int port;
	struct addrinfo *ai_list;
	belle_sip_list_t *srv_list; /*list of belle_sip_dns_srv_t*/
	int family;
	int flags;
	uint64_t start_time;
};

struct belle_sip_combined_resolver_context{
	belle_sip_resolver_context_t base;
	belle_sip_resolver_callback_t cb;
	void *cb_data;
	char *name;
	int port;
	int family;
	struct addrinfo *final_results;
	belle_sip_list_t *srv_results;
	belle_sip_resolver_context_t *srv_ctx;
	belle_sip_resolver_context_t *a_fallback_ctx;
};

struct belle_sip_dual_resolver_context{
	belle_sip_resolver_context_t base;
	belle_sip_resolver_callback_t cb;
	void *cb_data;
	char *name;
	belle_sip_resolver_context_t *a_ctx;
	belle_sip_resolver_context_t *aaaa_ctx;
	struct addrinfo *a_results;
	struct addrinfo *aaaa_results;
	unsigned char a_done;
	unsigned char aaaa_done;
};

void belle_sip_resolver_context_init(belle_sip_resolver_context_t *obj, belle_sip_stack_t *stack){
	obj->stack=stack;
}

static struct dns_resolv_conf *resconf(belle_sip_simple_resolver_context_t *ctx) {
	const char *path;
	int error;

	if (ctx->resconf)
		return ctx->resconf;

	if (!(ctx->resconf = dns_resconf_open(&error))) {
		belle_sip_error("%s dns_resconf_open error: %s", __FUNCTION__, dns_strerror(error));
		return NULL;
	}
	
	path=belle_sip_stack_get_dns_resolv_conf_file(ctx->base.stack);
	if (!path){
#ifdef _WIN32
		error = dns_resconf_loadwin(ctx->resconf);
		if (error) {
			belle_sip_error("%s dns_resconf_loadwin error", __FUNCTION__);
		}
#elif ANDROID
		error = dns_resconf_loadandroid(ctx->resconf);
		if (error) {
			belle_sip_error("%s dns_resconf_loadandroid error", __FUNCTION__);
		}
#elif HAVE_RESINIT
/*#elif HAVE_RESINIT && TARGET_OS_IPHONE*/
		error = dns_resconf_loadfromresolv(ctx->resconf);
		if (error) {
			belle_sip_error("%s dns_resconf_loadfromresolv error", __FUNCTION__);
		}
#else
		path = "/etc/resolv.conf";
		error = dns_resconf_loadpath(ctx->resconf, path);
		if (error) {
			belle_sip_error("%s dns_resconf_loadpath error [%s]: %s", __FUNCTION__, path, dns_strerror(error));
			return NULL;
		}

		path = "/etc/nsswitch.conf";
		error = dns_nssconf_loadpath(ctx->resconf, path);
		if (error) {
			belle_sip_message("%s dns_nssconf_loadpath error [%s]: %s", __FUNCTION__, path, dns_strerror(error));
		}
#endif
	}else{
		error = dns_resconf_loadpath(ctx->resconf, path);
		if (error) {
			belle_sip_error("%s dns_resconf_loadpath() of custom file error [%s]: %s", __FUNCTION__, path, dns_strerror(error));
			return NULL;
		}
	}
	
	if (error==0){
		char ip[64];
		char serv[10];
		int using_ipv6=FALSE;
		int i;
		
		belle_sip_message("Resolver is using DNS server(s):");
		for(i=0;i<sizeof(ctx->resconf->nameserver)/sizeof(ctx->resconf->nameserver[0]);++i){
			struct sockaddr *ns_addr=(struct sockaddr*)&ctx->resconf->nameserver[i];
			if (ns_addr->sa_family==AF_UNSPEC) break;
			getnameinfo(ns_addr,sizeof(struct sockaddr_storage),ip,sizeof(ip),serv,sizeof(serv),NI_NUMERICHOST|NI_NUMERICSERV);
			belle_sip_message("\t%s",ip);
			if (ns_addr->sa_family==AF_INET6) using_ipv6=TRUE;
		}
		ctx->resconf->iface.ss_family=using_ipv6 ? AF_INET6 : AF_INET;
	}else{
		belle_sip_error("Error loading dns server addresses.");
	}

	return ctx->resconf;
}

static struct dns_hosts *hosts(belle_sip_simple_resolver_context_t *ctx) {
	int error;

	if (ctx->hosts)
		return ctx->hosts;

	if (!(ctx->hosts = dns_hosts_local(&error))) {
		belle_sip_error("%s dns_hosts_local error: %s", __FUNCTION__, dns_strerror(error));
		return NULL;
	}

	if (ctx->base.stack->dns_user_hosts_file) {
		error = dns_hosts_loadpath(ctx->hosts, ctx->base.stack->dns_user_hosts_file);
		if (error) {
			belle_sip_error("%s dns_hosts_loadfile(\"%s\"): %s", __FUNCTION__,ctx->base.stack->dns_user_hosts_file,dns_strerror(error));
		}
	}

	return ctx->hosts;
}

struct dns_cache *cache(belle_sip_simple_resolver_context_t *ctx) {
	return NULL;
}

static struct addrinfo * ai_list_append(struct addrinfo *ai_list, struct addrinfo *ai_to_append) {
	struct addrinfo *ai_current = ai_list;
	if (ai_to_append == NULL) return ai_list;
	if (ai_list == NULL) return ai_to_append;
	while (ai_current->ai_next != NULL) {
		ai_current = ai_current->ai_next;
	}
	ai_current->ai_next = ai_to_append;
	return ai_list;
}

static int srv_compare_prio(const void *psrv1, const void *psrv2){
	belle_sip_dns_srv_t *srv1=(belle_sip_dns_srv_t*)psrv1;
	belle_sip_dns_srv_t *srv2=(belle_sip_dns_srv_t*)psrv2;
	if (srv1->priority < srv2->priority) return -1;
	if (srv1->priority == srv2->priority) return 0;
	return 1;
}

static void notify_results(belle_sip_simple_resolver_context_t *ctx){
	ctx->base.done=TRUE;
	if ((ctx->type == DNS_T_A) || (ctx->type == DNS_T_AAAA)) {
		ctx->cb(ctx->cb_data, ctx->name, ctx->ai_list);
	} else if (ctx->type == DNS_T_SRV) {
		ctx->srv_cb(ctx->srv_cb_data, ctx->name, ctx->srv_list);
	}
}

static void append_dns_result(belle_sip_simple_resolver_context_t *ctx, struct sockaddr *addr, socklen_t addrlen){
	char host[NI_MAXHOST + 1];
	int gai_err;
	int family=ctx->family;
	
	if ((gai_err=getnameinfo(addr, addrlen, host, sizeof(host), NULL, 0, NI_NUMERICHOST)) != 0){
		belle_sip_error("append_dns_result(): getnameinfo() failed: %s",gai_strerror(gai_err));
		return;
	}
	if (ctx->flags & AI_V4MAPPED) family=AF_INET6;
	ctx->ai_list = ai_list_append(ctx->ai_list, belle_sip_ip_address_to_addrinfo(family, host, ctx->port));
	belle_sip_message("%s resolved to %s", ctx->name, host);
}

static int resolver_process_data(belle_sip_simple_resolver_context_t *ctx, unsigned int revents) {
	struct dns_packet *ans;
	struct dns_rr_i *I;
	struct dns_rr_i dns_rr_it;
	int error;
	unsigned char simulated_timeout=0;
	int timeout=belle_sip_stack_get_dns_timeout(ctx->base.stack);

	/*Setting timeout to 0 can be used to simulate DNS timeout*/
	if ((revents!=0) && timeout==0){
		belle_sip_warning("Simulating DNS timeout");
		simulated_timeout=1;
	}
	
	if (simulated_timeout || ((revents & BELLE_SIP_EVENT_TIMEOUT) && (belle_sip_time_ms()-ctx->start_time>=timeout))) {
		belle_sip_error("%s timed-out", __FUNCTION__);
		notify_results(ctx);
		return BELLE_SIP_STOP;
	}
	/*belle_sip_message("resolver_process_data(): revents=%i",revents);*/
	error = dns_res_check(ctx->R);
	if (!error) {
		struct dns_rr rr;
		union dns_any any;
		enum dns_section section = DNS_S_AN;

		ans = dns_res_fetch(ctx->R, &error);
		memset(&dns_rr_it, 0, sizeof dns_rr_it);
		I = dns_rr_i_init(&dns_rr_it, ans);

		while (dns_rr_grep(&rr, 1, I, ans, &error)) {
			if (rr.section == section) {
				if ((error = dns_any_parse(dns_any_init(&any, sizeof(any)), &rr, ans))) {
					belle_sip_error("%s dns_any_parse error: %s", __FUNCTION__, dns_strerror(error));
					break;
				}
				if ((ctx->type == DNS_T_AAAA) && (rr.class == DNS_C_IN) && (rr.type == DNS_T_AAAA)) {
					struct dns_aaaa *aaaa = &any.aaaa;
					struct sockaddr_in6 sin6;
					memset(&sin6, 0, sizeof(sin6));
					memcpy(&sin6.sin6_addr, &aaaa->addr, sizeof(sin6.sin6_addr));
					sin6.sin6_family = AF_INET6;
					sin6.sin6_port = ctx->port;
					append_dns_result(ctx,(struct sockaddr*)&sin6,sizeof(sin6));
				} else if ((ctx->type == DNS_T_A) && (rr.class == DNS_C_IN) && (rr.type == DNS_T_A)) {
					struct dns_a *a = &any.a;
					struct sockaddr_in sin;
					memset(&sin, 0, sizeof(sin));
					memcpy(&sin.sin_addr, &a->addr, sizeof(sin.sin_addr));
					sin.sin_family = AF_INET;
					sin.sin_port = ctx->port;
					append_dns_result(ctx,(struct sockaddr*)&sin,sizeof(sin));
				} else if ((ctx->type == DNS_T_SRV) && (rr.class == DNS_C_IN) && (rr.type == DNS_T_SRV)) {
					char host[NI_MAXHOST + 1];
					struct dns_srv *srv = &any.srv;
					belle_sip_dns_srv_t * b_srv=belle_sip_dns_srv_create(srv);
					snprintf(host, sizeof(host), "[target:%s port:%d prio:%d weight:%d]", srv->target, srv->port, srv->priority, srv->weight);
					ctx->srv_list = belle_sip_list_insert_sorted(ctx->srv_list, belle_sip_object_ref(b_srv), srv_compare_prio);
					belle_sip_message("SRV %s resolved to %s", ctx->name, host);
				}
			}
		}
		free(ans);
		notify_results(ctx);
		return BELLE_SIP_STOP;
	}
	if (error != DNS_EAGAIN) {
		belle_sip_error("%s dns_res_check() error: %s (%d)", __FUNCTION__, dns_strerror(error), error);
		notify_results(ctx);
		return BELLE_SIP_STOP;
	}else{
		belle_sip_message("%s dns_res_check() in progress",__FUNCTION__);
	}
	return BELLE_SIP_CONTINUE;
}

static int _resolver_send_query(belle_sip_simple_resolver_context_t *ctx) {
	int error;

	if (!ctx->base.stack->resolver_send_error) {
		error = dns_res_submit(ctx->R, ctx->name, ctx->type, DNS_C_IN);
		if (error)
			belle_sip_error("%s dns_res_submit error [%s]: %s", __FUNCTION__, ctx->name, dns_strerror(error));
	} else {
		/* Error simulation */
		error = ctx->base.stack->resolver_send_error;
		belle_sip_error("%s dns_res_submit error [%s]: simulated error %d", __FUNCTION__, ctx->name, error);
	}
	if (error < 0) {
		return -1;
	}

	if (resolver_process_data(ctx, 0) == BELLE_SIP_CONTINUE) {
		ctx->start_time=belle_sip_time_ms();
		belle_sip_message("DNS resolution awaiting response, queued to main loop");
		/*only init source if res inprogress*/
		/*the timeout set to the source is 1 s, this is to allow dns.c to send request retransmissions*/
		belle_sip_socket_source_init((belle_sip_source_t*)ctx, (belle_sip_source_func_t)resolver_process_data, ctx, dns_res_pollfd(ctx->R), BELLE_SIP_EVENT_READ | BELLE_SIP_EVENT_TIMEOUT, 1000);
	}
	return 0;
}

static int resolver_process_data_delayed(belle_sip_simple_resolver_context_t *ctx, unsigned int revents) {
	int err=_resolver_send_query(ctx);
	if (err==0) return BELLE_SIP_CONTINUE;
	
	return BELLE_SIP_STOP;
}

static int _resolver_start_query(belle_sip_simple_resolver_context_t *ctx) {
	struct dns_hints *(*hints)() = &dns_hints_local;
	struct dns_options *opts;
#ifndef HAVE_C99
	struct dns_options opts_st;
#endif
	int error;
	struct dns_resolv_conf *conf;

	if (!ctx->name) return -1;

	conf=resconf(ctx);
	if (conf){
		conf->options.recurse = 0;
		conf->options.timeout=2;
		conf->options.attempts=5;
	}else
		return -1;
	if (!hosts(ctx))
		return -1;

	memset(&opts_st, 0, sizeof opts_st);
	opts = &opts_st;

	if (!(ctx->R = dns_res_open(ctx->resconf, ctx->hosts, dns_hints_mortal(hints(ctx->resconf, &error)), cache(ctx), opts, &error))) {
		belle_sip_error("%s dns_res_open error [%s]: %s", __FUNCTION__, ctx->name, dns_strerror(error));
		return -1;
	}
	error=0;
	if (ctx->base.stack->resolver_tx_delay > 0) {
		belle_sip_socket_source_init((belle_sip_source_t*)ctx, (belle_sip_source_func_t)resolver_process_data_delayed, ctx, -1, BELLE_SIP_EVENT_TIMEOUT, ctx->base.stack->resolver_tx_delay + 1000);
		belle_sip_message("%s DNS resolution delayed by %d ms", __FUNCTION__, ctx->base.stack->resolver_tx_delay);
	} else {
		error=_resolver_send_query(ctx);
	}
	if (error==0 && !ctx->base.done) belle_sip_main_loop_add_source(ctx->base.stack->ml,(belle_sip_source_t*)ctx);
	return error;
}

static belle_sip_simple_resolver_context_t * resolver_start_query(belle_sip_simple_resolver_context_t *ctx){
	int error=_resolver_start_query(ctx);
	if (error==0){
		if (!ctx->base.done){
			/* the resolution could not be done synchronously, return the context*/
			return ctx;
		}
		/*else resolution could be done synchronously*/
	}else{
		/*An error occured. We must notify the app.*/
		notify_results(ctx);
	}
	/*If it is failed or result could be found immediately, the context is now useless, we don't have to return it.*/
	belle_sip_object_unref(ctx);
	return NULL;
}


int belle_sip_addrinfo_to_ip(const struct addrinfo *ai, char *ip, size_t ip_size, int *port){
	char serv[16];
	int err=getnameinfo(ai->ai_addr,ai->ai_addrlen,ip,ip_size,serv,sizeof(serv),NI_NUMERICHOST|NI_NUMERICSERV);
	if (err!=0){
		belle_sip_error("getnameinfo() error: %s",gai_strerror(err));
		strncpy(ip,"<bug!!>",ip_size);
	}
	if (port) *port=atoi(serv);
	return 0;
}

struct addrinfo * belle_sip_ip_address_to_addrinfo(int family, const char *ipaddress, int port){
	struct addrinfo *res=NULL;
	struct addrinfo hints={0};
	char serv[10];
	int err;

	snprintf(serv,sizeof(serv),"%i",port);
	hints.ai_family=family;
	hints.ai_flags=AI_NUMERICSERV|AI_NUMERICHOST;
	hints.ai_socktype=SOCK_STREAM; //not used but it's needed to specify it because otherwise getaddrinfo returns one struct addrinfo per socktype.
	
	if (family==AF_INET6 && strchr(ipaddress,'.')!=NULL) {
		hints.ai_flags|=AI_V4MAPPED;
	}
	err=getaddrinfo(ipaddress,serv,&hints,&res);
	if (err!=0){
		return NULL;
	}
	return res;
}

static void belle_sip_combined_resolver_context_destroy(belle_sip_combined_resolver_context_t *obj){
	if (obj->name != NULL) {
		belle_sip_free(obj->name);
		obj->name = NULL;
	}
	if (obj->srv_ctx){
		belle_sip_object_unref(obj->srv_ctx);
		obj->srv_ctx=NULL;
	}
	if (obj->a_fallback_ctx){
		belle_sip_object_unref(obj->a_fallback_ctx);
		obj->a_fallback_ctx=NULL;
	}
}

static void belle_sip_simple_resolver_context_destroy(belle_sip_simple_resolver_context_t *ctx){
	/* Do not free elements of ctx->ai_list with freeaddrinfo(). Let the caller do it, otherwise
	   it will not be able to use them after the resolver has been destroyed. */
	if (ctx->name != NULL) {
		belle_sip_free(ctx->name);
		ctx->name = NULL;
	}
	if (ctx->R != NULL) {
		dns_res_close(ctx->R);
		ctx->R = NULL;
	}
	if (ctx->hosts != NULL) {
		dns_hosts_close(ctx->hosts);
		ctx->hosts = NULL;
	}
	if (ctx->resconf != NULL) {
		free(ctx->resconf);
		ctx->resconf = NULL;
	}
}

static void belle_sip_dual_resolver_context_destroy(belle_sip_dual_resolver_context_t *obj){
	if (obj->a_ctx){
		belle_sip_object_unref(obj->a_ctx);
		obj->a_ctx=NULL;
	}
	if (obj->aaaa_ctx){
		belle_sip_object_unref(obj->aaaa_ctx);
		obj->aaaa_ctx=NULL;
	}
	if (obj->a_results){
		freeaddrinfo(obj->a_results);
		obj->a_results=NULL;
	}
	if (obj->aaaa_results){
		freeaddrinfo(obj->aaaa_results);
		obj->aaaa_results=NULL;
	}
	if (obj->name){
		belle_sip_free(obj->name);
		obj->name=NULL;
	}
}

static void simple_resolver_cancel(belle_sip_resolver_context_t *obj){
	belle_sip_main_loop_remove_source(obj->stack->ml,(belle_sip_source_t*)obj);
	belle_sip_object_unref(obj);
}

static void combined_resolver_cancel(belle_sip_resolver_context_t *p){
	belle_sip_combined_resolver_context_t *obj=(belle_sip_combined_resolver_context_t*)p;
	
	if (obj->srv_ctx){
		belle_sip_resolver_context_cancel(obj->srv_ctx);
		obj->srv_ctx=NULL;
	}
	if (obj->a_fallback_ctx){
		belle_sip_resolver_context_cancel(obj->a_fallback_ctx);
		obj->a_fallback_ctx=NULL;
	}
	belle_sip_list_free_with_data(obj->srv_results,belle_sip_object_unref);
	obj->srv_results=NULL;
	belle_sip_object_unref(obj);
}

static void dual_resolver_cancel(belle_sip_resolver_context_t *p){
	belle_sip_dual_resolver_context_t *obj=(belle_sip_dual_resolver_context_t*)p;
	if (obj->a_ctx){
		belle_sip_resolver_context_cancel(obj->a_ctx);
		obj->a_ctx=NULL;
	}
	if (obj->aaaa_ctx){
		belle_sip_resolver_context_cancel(obj->aaaa_ctx);
		obj->aaaa_ctx=NULL;
	}
	belle_sip_object_unref(p);
}

BELLE_SIP_DECLARE_NO_IMPLEMENTED_INTERFACES(belle_sip_resolver_context_t);
BELLE_SIP_INSTANCIATE_CUSTOM_VPTR_BEGIN(belle_sip_resolver_context_t)
	{
		BELLE_SIP_VPTR_INIT(belle_sip_resolver_context_t,belle_sip_source_t,TRUE),
		(belle_sip_object_destroy_t) NULL,
		NULL,
		NULL
	},
	NULL
BELLE_SIP_INSTANCIATE_CUSTOM_VPTR_END


BELLE_SIP_DECLARE_NO_IMPLEMENTED_INTERFACES(belle_sip_simple_resolver_context_t);
BELLE_SIP_INSTANCIATE_CUSTOM_VPTR_BEGIN(belle_sip_simple_resolver_context_t)
	{
		{
			BELLE_SIP_VPTR_INIT(belle_sip_simple_resolver_context_t,belle_sip_resolver_context_t,TRUE),
			(belle_sip_object_destroy_t) belle_sip_simple_resolver_context_destroy,
			NULL,
			NULL
		},
		simple_resolver_cancel
	}
BELLE_SIP_INSTANCIATE_CUSTOM_VPTR_END

BELLE_SIP_DECLARE_NO_IMPLEMENTED_INTERFACES(belle_sip_dual_resolver_context_t);
BELLE_SIP_INSTANCIATE_CUSTOM_VPTR_BEGIN(belle_sip_dual_resolver_context_t)
	{
		{
			BELLE_SIP_VPTR_INIT(belle_sip_dual_resolver_context_t,belle_sip_resolver_context_t,TRUE),
			(belle_sip_object_destroy_t) belle_sip_dual_resolver_context_destroy,
			NULL,
			NULL
		},
		dual_resolver_cancel
	}
BELLE_SIP_INSTANCIATE_CUSTOM_VPTR_END

BELLE_SIP_DECLARE_NO_IMPLEMENTED_INTERFACES(belle_sip_combined_resolver_context_t);
BELLE_SIP_INSTANCIATE_CUSTOM_VPTR_BEGIN(belle_sip_combined_resolver_context_t)
	{
		{
			BELLE_SIP_VPTR_INIT(belle_sip_combined_resolver_context_t,belle_sip_resolver_context_t,TRUE),
			(belle_sip_object_destroy_t) belle_sip_combined_resolver_context_destroy,
			NULL,
			NULL
		},
		combined_resolver_cancel
	}
BELLE_SIP_INSTANCIATE_CUSTOM_VPTR_END

static char * srv_prefix_from_transport(const char *transport) {
	char *prefix = "";

	if (strcasecmp(transport, "udp") == 0) {
		prefix = "_sip._udp.";
	} else if (strcasecmp(transport, "tcp") == 0) {
		prefix = "_sip._tcp.";
	} else if (strcasecmp(transport, "tls") == 0) {
		prefix = "_sips._tcp.";
	} else {
		prefix = "_sip._udp.";
	}
	return prefix;
}

static void combined_notify_results(belle_sip_combined_resolver_context_t *obj){
	obj->base.done=TRUE;
	obj->cb(obj->cb_data,obj->name,obj->final_results);
	obj->final_results=NULL;
	/*clean everything (cancel() does it very well)*/
	combined_resolver_cancel((belle_sip_resolver_context_t*)obj);
}

static void process_a_fallback_result(void *data, const char *name, struct addrinfo *ai_list){
	belle_sip_combined_resolver_context_t *ctx=(belle_sip_combined_resolver_context_t *)data;
	ctx->final_results=ai_list;
	combined_notify_results(ctx);
}

static void combined_resolver_context_check_finished(belle_sip_combined_resolver_context_t *obj){
	belle_sip_list_t *elem;
	struct addrinfo *final=NULL;
	unsigned char finished=TRUE;
	
	for(elem=obj->srv_results;elem!=NULL;elem=elem->next){
		belle_sip_dns_srv_t *srv=(belle_sip_dns_srv_t*)elem->data;
		if (!srv->a_done) {
			finished=FALSE;
			break;
		}
	}
	if (finished){
		belle_sip_message("All A/AAAA results for combined resolution have arrived.");
		for(elem=obj->srv_results;elem!=NULL;elem=elem->next){
			belle_sip_dns_srv_t *srv=(belle_sip_dns_srv_t*)elem->data;
			final=ai_list_append(final,srv->a_results);
			srv->a_results=NULL;
		}
		belle_sip_list_free_with_data(obj->srv_results,belle_sip_object_unref);
		obj->srv_results=NULL;
		obj->final_results=final;
		combined_notify_results(obj);
	}
}

static void process_a_from_srv(void *data, const char *name, struct addrinfo *ai_list){
	belle_sip_dns_srv_t *srv=(belle_sip_dns_srv_t*)data;
	srv->a_results=ai_list;
	srv->a_done=TRUE;
	belle_sip_message("A query finished for srv result [%s]",srv->target);
	combined_resolver_context_check_finished(srv->root_resolver);
}

static void srv_resolve_a(belle_sip_combined_resolver_context_t *obj, belle_sip_dns_srv_t *srv){
	belle_sip_message("Starting A/AAAA query for srv result [%s]",srv->target);
	srv->root_resolver=obj;
	srv->a_resolver=belle_sip_stack_resolve_a(obj->base.stack,srv->target,srv->port,obj->family,process_a_from_srv,srv);
	if (srv->a_resolver){
		belle_sip_object_ref(srv->a_resolver);
	}
}

static void process_srv_results(void *data, const char *name, belle_sip_list_t *srv_results){
	belle_sip_combined_resolver_context_t *ctx=(belle_sip_combined_resolver_context_t *)data;
	/*take a ref here, because the A resolution might succeed synchronously and terminate the context before exiting this function*/
	belle_sip_object_ref(ctx);
	if (srv_results){
		belle_sip_list_t *elem;
		ctx->srv_results=srv_results;
		for(elem=srv_results;elem!=NULL;elem=elem->next){
			belle_sip_dns_srv_t *srv=(belle_sip_dns_srv_t*)elem->data;
			srv_resolve_a(ctx,srv);
		}
	}else{
		/*no SRV results, perform A query */
		belle_sip_message("No SRV result for [%s], trying A/AAAA.",name);
		ctx->a_fallback_ctx=belle_sip_stack_resolve_a(ctx->base.stack,ctx->name,ctx->port,ctx->family,process_a_fallback_result,ctx);
		if (ctx->a_fallback_ctx) belle_sip_object_ref(ctx->a_fallback_ctx);
	}
	belle_sip_object_unref(ctx);
}

/**
 * Perform combined SRV + A / AAAA resolution.
**/
belle_sip_resolver_context_t * belle_sip_stack_resolve(belle_sip_stack_t *stack, const char *transport, const char *name, int port, int family, belle_sip_resolver_callback_t cb, void *data) {
	struct addrinfo *res = belle_sip_ip_address_to_addrinfo(family, name, port);
	if (res == NULL) {
		/* First perform asynchronous DNS SRV query */
		belle_sip_combined_resolver_context_t *ctx = belle_sip_object_new(belle_sip_combined_resolver_context_t);
		belle_sip_resolver_context_init((belle_sip_resolver_context_t*)ctx,stack);
		belle_sip_object_ref(ctx);/*we don't want the object to be destroyed until the end of this function*/
		ctx->cb=cb;
		ctx->cb_data = data;
		ctx->name = belle_sip_strdup(name);
		ctx->port=port;
		if (family == 0) family = AF_UNSPEC;
		ctx->family = family;
		/*take a ref for the entire duration of the DNS procedure, it will be released when it is finished*/
		belle_sip_object_ref(ctx);
		ctx->srv_ctx=belle_sip_stack_resolve_srv(stack,transport,name,process_srv_results,ctx);
		if (ctx->srv_ctx) belle_sip_object_ref(ctx->srv_ctx);
		if (ctx->base.done){
			belle_sip_object_unref(ctx);
			return NULL;
		}
		belle_sip_object_unref(ctx);
		return BELLE_SIP_RESOLVER_CONTEXT(ctx);
	} else {
		/* There is no resolve to be done */
		cb(data, name, res);
		return NULL;
	}
}

static belle_sip_resolver_context_t * belle_sip_stack_resolve_single(belle_sip_stack_t *stack, const char *name, int port, int family, int flags, belle_sip_resolver_callback_t cb , void *data){
	/* Then perform asynchronous DNS A or AAAA query */
	belle_sip_simple_resolver_context_t *ctx = belle_sip_object_new(belle_sip_simple_resolver_context_t);
	belle_sip_resolver_context_init((belle_sip_resolver_context_t*)ctx,stack);
	
	ctx->cb_data = data;
	ctx->cb = cb;
	ctx->name = belle_sip_strdup(name);
	ctx->port = port;
	ctx->flags = flags;
	if (family == 0) family = AF_UNSPEC;
	ctx->family = family;
	ctx->type = (ctx->family == AF_INET6) ? DNS_T_AAAA : DNS_T_A;
	return (belle_sip_resolver_context_t*)resolver_start_query(ctx);
}

static void notify_dual_results(belle_sip_dual_resolver_context_t *ctx){
	if (ctx->a_done && ctx->aaaa_done){
		struct addrinfo *results=ctx->aaaa_results;
		results=ai_list_append(results,ctx->a_results);
		ctx->a_results=NULL;
		ctx->aaaa_results=NULL;
		ctx->cb(ctx->cb_data,ctx->name,results);
		belle_sip_object_unref(ctx);
	}
}

static void on_ipv4_results(void *data, const char *name, struct addrinfo *ai_list){
	belle_sip_dual_resolver_context_t *ctx=(belle_sip_dual_resolver_context_t *)data;
	ctx->a_done=TRUE;
	ctx->a_results=ai_list;
	notify_dual_results(ctx);
}

static void on_ipv6_results(void *data, const char *name, struct addrinfo *ai_list){
	belle_sip_dual_resolver_context_t *ctx=(belle_sip_dual_resolver_context_t *)data;
	ctx->aaaa_done=TRUE;
	ctx->aaaa_results=ai_list;
	notify_dual_results(ctx);
}

static belle_sip_resolver_context_t * belle_sip_stack_resolve_dual(belle_sip_stack_t *stack, const char *name, int port, belle_sip_resolver_callback_t cb , void *data){
	/* Then perform asynchronous DNS A or AAAA query */
	belle_sip_dual_resolver_context_t *ctx = belle_sip_object_new(belle_sip_dual_resolver_context_t);
	belle_sip_resolver_context_init((belle_sip_resolver_context_t*)ctx,stack);
	
	ctx->cb_data = data;
	ctx->cb = cb;
	ctx->name = belle_sip_strdup(name);
	belle_sip_object_ref(ctx);
	ctx->a_ctx=belle_sip_stack_resolve_single(stack,name,port,AF_INET, AI_V4MAPPED, on_ipv4_results,ctx);
	if (ctx->a_ctx) belle_sip_object_ref(ctx->a_ctx);
	ctx->aaaa_ctx=belle_sip_stack_resolve_single(stack, name, port, AF_INET6, 0, on_ipv6_results, ctx);
	if (ctx->aaaa_ctx) belle_sip_object_ref(ctx->aaaa_ctx);
	if (ctx->aaaa_ctx==NULL && ctx->a_ctx==NULL){
		/*all results found synchronously*/
		belle_sip_object_unref(ctx);
		ctx=NULL;
	}
	return BELLE_SIP_RESOLVER_CONTEXT(ctx);
}

belle_sip_resolver_context_t * belle_sip_stack_resolve_a(belle_sip_stack_t *stack, const char *name, int port, int family, belle_sip_resolver_callback_t cb , void *data) {
	struct addrinfo *res = belle_sip_ip_address_to_addrinfo(family, name, port);
	if (res == NULL) {
		switch(family){
			case AF_UNSPEC:
				family=AF_INET6;
			case AF_INET6:
				return belle_sip_stack_resolve_dual(stack,name,port,cb,data);
				break;
			case AF_INET:
				return belle_sip_stack_resolve_single(stack,name,port,AF_INET,0,cb,data);
				break;
			default:
				belle_sip_error("belle_sip_stack_resolve_a(): unsupported address family [%i]",family);
		}
	} else {
		/* There is no resolve to be done */
		cb(data, name, res);
	}
	return NULL;
}

belle_sip_resolver_context_t * belle_sip_stack_resolve_srv(belle_sip_stack_t *stack, const char *transport, const char *name, belle_sip_resolver_srv_callback_t cb, void *data) {
	belle_sip_simple_resolver_context_t *ctx = belle_sip_object_new(belle_sip_simple_resolver_context_t);
	belle_sip_resolver_context_init((belle_sip_resolver_context_t*)ctx,stack);
	ctx->srv_cb_data = data;
	ctx->srv_cb = cb;
	ctx->name = belle_sip_concat(srv_prefix_from_transport(transport), name, NULL);
	ctx->type = DNS_T_SRV;
	return (belle_sip_resolver_context_t*)resolver_start_query(ctx);
}

void belle_sip_resolver_context_cancel(belle_sip_resolver_context_t *obj){
	BELLE_SIP_OBJECT_VPTR(obj,belle_sip_resolver_context_t)->cancel(obj);
}


void belle_sip_get_src_addr_for(const struct sockaddr *dest, socklen_t destlen, struct sockaddr *src, socklen_t *srclen, int local_port){
	int af_type=dest->sa_family;
	int sock=socket(af_type,SOCK_DGRAM,IPPROTO_UDP);
	
	if (sock==(belle_sip_socket_t)-1){
		belle_sip_fatal("Could not create socket: %s",belle_sip_get_socket_error_string());
		goto fail;
	}
	if (connect(sock,dest,destlen)==-1){
		belle_sip_error("belle_sip_get_src_addr_for: connect() failed: %s",belle_sip_get_socket_error_string());
		goto fail;
	}
	if (getsockname(sock,src,srclen)==-1){
		belle_sip_error("belle_sip_get_src_addr_for: getsockname() failed: %s",belle_sip_get_socket_error_string());
		goto fail;
	}
	
	if (af_type==AF_INET6){
		struct sockaddr_in6 *sin6=(struct sockaddr_in6*)src;
		sin6->sin6_port=htons(local_port);
	}else{
		struct sockaddr_in *sin=(struct sockaddr_in*)src;
		sin->sin_port=htons(local_port);
	}
	
	close_socket(sock);
	return;
fail:
	{
		struct addrinfo hints={0},*res=NULL;
		int err;
		hints.ai_family=af_type;
		err=getaddrinfo(af_type==AF_INET ? "0.0.0.0" : "::0","0",&hints,&res);
		if (err!=0) belle_sip_fatal("belle_sip_get_src_addr_for(): getaddrinfo failed: %s",belle_sip_get_socket_error_string_from_code(err));
		memcpy(src,res->ai_addr,MIN((size_t)*srclen,res->ai_addrlen));
		*srclen=res->ai_addrlen;
		freeaddrinfo(res);
	}
	if (sock!=(belle_sip_socket_t)-1) close_socket(sock);
}

#ifndef IN6_GET_ADDR_V4MAPPED
#define IN6_GET_ADDR_V4MAPPED(sin6_addr)	*(unsigned int*)((unsigned char*)(sin6_addr)+12)
#endif


void belle_sip_address_remove_v4_mapping(const struct sockaddr *v6, struct sockaddr *result, socklen_t *result_len){
	if (v6->sa_family==AF_INET6){
		struct sockaddr_in6 *in6=(struct sockaddr_in6*)v6;
		
		if (IN6_IS_ADDR_V4MAPPED(&in6->sin6_addr)){
			struct sockaddr_in *in=(struct sockaddr_in*)result;
			result->sa_family=AF_INET;
			in->sin_addr.s_addr = IN6_GET_ADDR_V4MAPPED(&in6->sin6_addr);
			in->sin_port=in6->sin6_port;
			*result_len=sizeof(struct sockaddr_in);
		}else{
			if (v6!=result) memcpy(result,v6,sizeof(struct sockaddr_in6));
			*result_len=sizeof(struct sockaddr_in6);
		}
		
	}else{
		*result_len=sizeof(struct sockaddr_in);
		if (v6!=result) memcpy(result,v6,sizeof(struct sockaddr_in));
	}
}



