/*
 * Link Monitor Applet
 * Copyright (C) 2004-2007 Jean-Yves Lefort <jylefort@brutele.be>
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program; if not, write to the Free Software Foundation, Inc.,
 * 51 Franklin Street, Fifth Floor, Boston, MA 02110-1301 USA.
 */

#include "config.h"
#include <stdlib.h>
#include <stdarg.h>
#include <time.h>
#include <errno.h>
#include <glib/gi18n.h>
#include <gnome.h>
#include <glade/glade.h>
#include <eel/eel.h>
#include "lm-util.h"

/*** types *******************************************************************/

typedef struct
{
  GtkContainer	*container;
  const char	*callback_prefix;
} ContainerCreateInterfaceConnectInfo;

typedef struct
{
  gpointer	instance;
  unsigned long	id;
} SignalHandler;

/*** functions ***************************************************************/

static GladeXML *lm_glade_xml_new (const char *filename,
				   const char *root,
				   const char *domain);
static GtkWidget *lm_glade_xml_get_widget (GladeXML *xml,
					   const char *widget_name);

static void lm_container_create_interface_connect_cb (const char *handler_name,
						      GObject *object,
						      const char *signal_name,
						      const char *signal_data,
						      GObject *connect_object,
						      gboolean after,
						      gpointer user_data);

static GtkWidget *lm_menu_item_new (const char *stock_id, const char *mnemonic);

static void lm_error_dialog_real (GtkWindow *parent,
				  gboolean blocking,
				  const char *primary,
				  const char *secondary);

static void lm_g_object_connect_weak_notify_cb (gpointer data, GObject *former_object);

static gboolean lm_callback_dispatch_cb (gpointer data);

/*** implementation **********************************************************/

/**
 * lm_g_object_slist_free:
 * @list: a #GSList of #GObject instances
 *
 * Equivalent of eel_g_object_list_free() for a singly-linked list.
 **/
void
lm_g_object_slist_free (GSList *list)
{
  eel_g_slist_free_deep_custom(list, (GFunc) g_object_unref, NULL);
}

GSList *
lm_g_object_slist_copy (GSList *list)
{
  g_slist_foreach(list, (GFunc) g_object_ref, NULL);

  return g_slist_copy(list);
}

static GladeXML *
lm_glade_xml_new (const char *filename, const char *root, const char *domain)
{
  GladeXML *xml;

  g_return_val_if_fail(filename != NULL, NULL);

  xml = glade_xml_new(filename, root, domain);
  if (! xml)
    lm_fatal_error_dialog(NULL, _("Unable to load interface \"%s\". Please check your link monitor installation."), filename);

  return xml;
}

static GtkWidget *
lm_glade_xml_get_widget (GladeXML *xml, const char *widget_name)
{
  GtkWidget *widget;

  g_return_val_if_fail(GLADE_IS_XML(xml), NULL);
  g_return_val_if_fail(widget_name != NULL, NULL);

  widget = glade_xml_get_widget(xml, widget_name);
  if (! widget)
    lm_fatal_error_dialog(NULL, _("Widget \"%s\" not found in interface \"%s\". Please check your link monitor installation."), widget_name, xml->filename);

  return widget;
}

void
lm_container_create_interface (GtkContainer *container,
			       const char *filename,
			       const char *child_name,
			       const char *callback_prefix,
			       ...)
{
  GladeXML *xml;
  GtkWidget *child;
  ContainerCreateInterfaceConnectInfo info;
  va_list args;
  const char *widget_name;

  g_return_if_fail(GTK_IS_CONTAINER(container));
  g_return_if_fail(filename != NULL);
  g_return_if_fail(child_name != NULL);
  g_return_if_fail(callback_prefix != NULL);

  xml = lm_glade_xml_new(filename, child_name, NULL);
  child = lm_glade_xml_get_widget(xml, child_name);

  if (GTK_IS_DIALOG(container))
    gtk_box_pack_start(GTK_BOX(GTK_DIALOG(container)->vbox), child, TRUE, TRUE, 0);
  else
    gtk_container_add(container, child);

  info.container = container;
  info.callback_prefix = callback_prefix;
  glade_xml_signal_autoconnect_full(xml, lm_container_create_interface_connect_cb, &info);

  va_start(args, callback_prefix);
  while ((widget_name = va_arg(args, const char *)))
    {
      GtkWidget **widget;

      widget = va_arg(args, GtkWidget **);
      g_return_if_fail(widget != NULL);

      *widget = lm_glade_xml_get_widget(xml, widget_name);
    }
  va_end(args);

  g_object_unref(xml);
}

static void
lm_container_create_interface_connect_cb (const char *handler_name,
					  GObject *object,
					  const char *signal_name,
					  const char *signal_data,
					  GObject *connect_object,
					  gboolean after,
					  gpointer user_data)
{
  static GModule *module = NULL;
  ContainerCreateInterfaceConnectInfo *info = user_data;
  char *cb_name;
  GCallback cb;
  GConnectFlags flags;

  if (! module)
    {
      module = g_module_open(NULL, 0);
      if (! module)
	lm_fatal_error_dialog(NULL, _("Unable to open self as a module (%s)."), g_module_error());
    }

  cb_name = g_strconcat(info->callback_prefix, handler_name, NULL);
  if (! g_module_symbol(module, cb_name, (gpointer) &cb))
    lm_fatal_error_dialog(NULL, _("Signal handler \"%s\" not found. Please check your link monitor installation."), cb_name);
  g_free(cb_name);

  flags = G_CONNECT_SWAPPED;
  if (after)
    flags |= G_CONNECT_AFTER;

  g_signal_connect_data(object, signal_name, cb, info->container, NULL, flags);
}

GdkPixbuf *
lm_pixbuf_new (const char *filename)
{
  GdkPixbuf *pixbuf;
  GError *err = NULL;

  g_return_val_if_fail(filename != NULL, NULL);

  pixbuf = gdk_pixbuf_new_from_file(filename, &err);
  if (! pixbuf)
    {
      lm_fatal_error_dialog(NULL, _("Unable to load image \"%s\" (%s). Please check your link monitor installation."), filename, err->message);
      g_error_free(err);
    }

  return pixbuf;
}

void
lm_thread_create (GThreadFunc func, gpointer data)
{
  GError *err = NULL;

  g_return_if_fail(func != NULL);

  if (! g_thread_create(func, data, FALSE, &err))
    {
      lm_fatal_error_dialog(NULL, _("Unable to create a thread: %s."), err->message);
      g_error_free(err);
    }
}

time_t
lm_time (void)
{
  time_t t;

  t = time(NULL);
  if (t < 0)
    {
      g_warning(_("unable to get current time: %s"), g_strerror(errno));
      t = 0;
    }

  return t;
}

/*
 * Subtract 2 timeval structs: out = out - in. Out is assumed to be >=
 * in (based on tvsub() from FreeBSD's ping.c).
 *
 * Copyright (c) 1989, 1993
 *	The Regents of the University of California.  All rights reserved.
 *
 * This code is derived from software contributed to Berkeley by
 * Mike Muuss.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 4. Neither the name of the University nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE REGENTS AND CONTRIBUTORS ``AS IS'' AND
 * ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
 * IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
 * ARE DISCLAIMED.  IN NO EVENT SHALL THE REGENTS OR CONTRIBUTORS BE LIABLE
 * FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
 * DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
 * OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
 * HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
 * LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
 * SUCH DAMAGE.
 */
void
lm_tvsub (GTimeVal *out, const GTimeVal *in)
{
  g_return_if_fail(out != NULL);
  g_return_if_fail(in != NULL);

  if ((out->tv_usec -= in->tv_usec) < 0)
    {
      --out->tv_sec;
      out->tv_usec += 1000000;
    }
  out->tv_sec -= in->tv_sec;
}

void
lm_display_help (const char *link_id)
{
  GError *err = NULL;

  if (! gnome_help_display("link-monitor-applet.xml", link_id, &err))
    {
      lm_error_dialog(NULL, _("Unable to display help"), "%s", err->message);
      g_error_free(err);
    }
}

/**
 * lm_menu_shell_append:
 * @shell: the #GtkMenuShell to append to
 * @stock_id: the stock ID of the item, or NULL
 * @mnemonic: the mnemonic of the item, or NULL
 *
 * Creates a new menu item, shows it and appends it to @shell.
 *
 * If both @stock_id and @mnemonic are provided, a #GtkImageMenuItem
 * will be created using the text of @mnemonic and the icon of
 * @stock_id.
 *
 * If only @stock_id is provided, a #GtkImageMenuitem will be created
 * using the text and icon of @stock_id.
 *
 * If only @mnemonic is provided, a #GtkMenuItem will be created using
 * the text of @mnemonic.
 *
 * If @stock_id and @mnemonic are both NULL, a #GtkSeparatorMenuItem
 * will be created.
 *
 * Return value: the new menu item
 **/
GtkWidget *
lm_menu_shell_append (GtkMenuShell *shell,
		      const char *stock_id,
		      const char *mnemonic)
{
  GtkWidget *item;

  g_return_val_if_fail(GTK_IS_MENU_SHELL(shell), NULL);

  item = lm_menu_item_new(stock_id, mnemonic);
  gtk_menu_shell_append(shell, item);
  gtk_widget_show(item);

  return item;
}

static GtkWidget *
lm_menu_item_new (const char *stock_id, const char *mnemonic)
{
  GtkWidget *item;

  if (stock_id && mnemonic)
    {
      GtkWidget *image;

      item = gtk_image_menu_item_new_with_mnemonic(mnemonic);

      image = gtk_image_new_from_stock(stock_id, GTK_ICON_SIZE_MENU);
      gtk_image_menu_item_set_image(GTK_IMAGE_MENU_ITEM(item), image);
      gtk_widget_show(image);
    }
  else if (stock_id)
    item = gtk_image_menu_item_new_from_stock(stock_id, NULL);
  else if (mnemonic)
    item = gtk_menu_item_new_with_mnemonic(mnemonic);
  else
    item = gtk_separator_menu_item_new();

  return item;
}

static void
lm_error_dialog_real (GtkWindow *parent,
		      gboolean blocking,
		      const char *primary,
		      const char *secondary)
{
  GtkWidget *dialog;

  g_return_if_fail(primary != NULL);
  g_return_if_fail(secondary != NULL);

  dialog = gtk_message_dialog_new(parent,
				  GTK_DIALOG_DESTROY_WITH_PARENT,
				  GTK_MESSAGE_ERROR,
				  GTK_BUTTONS_NONE,
				  "%s",
				  primary);

  gtk_message_dialog_format_secondary_text(GTK_MESSAGE_DIALOG(dialog), "%s", secondary);
  gtk_window_set_title(GTK_WINDOW(dialog), ""); /* HIG */

  gtk_dialog_add_button(GTK_DIALOG(dialog), GTK_STOCK_OK, GTK_RESPONSE_OK);
  gtk_dialog_set_default_response(GTK_DIALOG(dialog), GTK_RESPONSE_OK);

  if (blocking)
    {
      gtk_dialog_run(GTK_DIALOG(dialog));
      gtk_widget_destroy(dialog);
    }
  else
    {
      g_signal_connect_swapped(dialog,
			       "response",
			       G_CALLBACK(gtk_widget_destroy),
			       dialog);
      gtk_widget_show(dialog);
    }
}

void
lm_error_dialog (GtkWindow *parent,
		 const char *primary,
		 const char *format,
		 ...)
{
  va_list args;
  char *secondary;

  g_return_if_fail(primary != NULL);
  g_return_if_fail(format != NULL);

  va_start(args, format);
  secondary = g_strdup_vprintf(format, args);
  va_end(args);

  lm_error_dialog_real(parent, FALSE, primary, secondary);
  g_free(secondary);
}

void
lm_fatal_error_dialog (GtkWindow *parent, const char *format, ...)
{
  va_list args;
  char *secondary;

  g_assert(format != NULL);

  va_start(args, format);
  secondary = g_strdup_vprintf(format, args);
  va_end(args);

  lm_error_dialog_real(parent, TRUE, _("A fatal error has occurred in the link monitor"), secondary);
  g_free(secondary);

  exit(1);
}

void
lm_source_clear (unsigned int *tag)
{
  g_return_if_fail(tag != NULL);

  if (*tag)
    {
      g_source_remove(*tag);
      *tag = 0;
    }
}

void
lm_gtk_object_ref_and_sink (GtkObject *object)
{
  g_return_if_fail(GTK_IS_OBJECT(object));

  g_object_ref(object);
  gtk_object_sink(object);
}

/**
 * lm_g_object_connect:
 * @object: the object to associate the handlers with
 * @instance: the instance to connect to
 * @signal_spec: the spec for the first signal
 * @...: #GCallback for the first signal, followed by data for the
 *       first signal, followed optionally by more signal spec/callback/data
 *       triples, followed by NULL
 *
 * Connects to one or more signals of @instance, associating the
 * handlers with @object. The handlers will be disconnected whenever
 * @object is finalized.
 *
 * Note: this function is not thread-safe. If @object and @instance
 * are finalized concurrently, the behaviour is undefined.
 *
 * The signals specs must be in the same format than those passed to
 * g_object_connect(), except that object-signal,
 * swapped-object-signal, object-signal-after and
 * swapped-object-signal-after are not accepted.
 *
 * Return value: @object
 **/
gpointer
lm_g_object_connect (gpointer object,
		     gpointer instance,
		     const char *signal_spec,
		     ...)
{
  va_list args;

  g_return_val_if_fail(G_IS_OBJECT(object), NULL);
  g_return_val_if_fail(G_IS_OBJECT(instance), NULL);

  va_start(args, signal_spec);
  while (signal_spec)
    {
      GCallback callback = va_arg(args, GCallback);
      gpointer data = va_arg(args, gpointer);
      SignalHandler *handler;

      handler = g_new(SignalHandler, 1);
      handler->instance = instance;

      if (g_str_has_prefix(signal_spec, "signal::"))
	handler->id = g_signal_connect(instance, signal_spec + 8, callback, data);
      else if (g_str_has_prefix(signal_spec, "swapped_signal")
	       || g_str_has_prefix(signal_spec, "swapped-signal"))
	handler->id = g_signal_connect_swapped(instance, signal_spec + 16, callback, data);
      else if (g_str_has_prefix(signal_spec, "signal_after::")
	       || g_str_has_prefix(signal_spec, "signal-after::"))
	handler->id = g_signal_connect_after(instance, signal_spec + 14, callback, data);
      else if (g_str_has_prefix(signal_spec, "swapped_signal_after::")
	       || g_str_has_prefix(signal_spec, "swapped-signal-after::"))
	handler->id = g_signal_connect_data(instance, signal_spec + 22, callback, data, NULL, G_CONNECT_AFTER | G_CONNECT_SWAPPED);
      else
	g_critical("invalid signal specification \"%s\"", signal_spec);

      eel_add_weak_pointer(&handler->instance);
      g_object_weak_ref(object, lm_g_object_connect_weak_notify_cb, handler);

      signal_spec = va_arg(args, const char *);
    }
  va_end(args);

  return object;
}

static void
lm_g_object_connect_weak_notify_cb (gpointer data, GObject *former_object)
{
  SignalHandler *handler = data;

  if (handler->instance)
    {
      g_signal_handler_disconnect(handler->instance, handler->id);
      eel_remove_weak_pointer(&handler->instance);
    }
  g_free(handler);
}

void
lm_callback_init (LMCallback *callback, LMCallbackFunc func, gpointer data)
{
  g_return_if_fail(callback != NULL);
  g_return_if_fail(func != NULL);

  callback->idle_id = 0;
  callback->func = func;
  callback->data = data;
}

void
lm_callback_queue (LMCallback *callback)
{
  g_return_if_fail(callback != NULL);
  g_return_if_fail(callback->func != NULL);

  if (! callback->idle_id)
    callback->idle_id = g_idle_add(lm_callback_dispatch_cb, callback);
}

static gboolean
lm_callback_dispatch_cb (gpointer data)
{
  LMCallback *callback = data;

  GDK_THREADS_ENTER();

  callback->func(callback->data);

  GDK_THREADS_LEAVE();

  callback->idle_id = 0;
  return FALSE;			/* remove source */
}

void
lm_callback_clear (LMCallback *callback)
{
  g_return_if_fail(callback != NULL);

  lm_source_clear(&callback->idle_id);
}
