/*
 * $Id: lm-preferences.c,v 1.10 2004/08/21 21:15:14 jylefort Exp $
 *
 * Copyright (c) 2004 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include <gtk/gtk.h>
#include "lm-applet.h"
#include "lm-util.h"
#include "lm-host-view.h"

/*** cpp *********************************************************************/

/* glib-i18n.h does not provide a stub for ngettext */
#ifndef ENABLE_NLS
#ifndef ngettext		/* but future versions might define it */
#define ngettext(Msgid1, Msgid2, N) \
  ((N) == 1 ? (const char *) (Msgid1) : (const char *) (Msgid2))
#endif /* ngettext */
#endif /* ENABLE_NLS */

/*** types *******************************************************************/
    
typedef struct
{
  GtkWidget	*dialog;
  GtkWidget	*scrolled;
  GtkWidget	*list;
  GtkWidget	*selected_label;
  GtkWidget	*remove;
  GtkWidget	*up;
  GtkWidget	*down;
  GtkWidget	*delay_label;
  GtkWidget	*delay_spin;
  GtkWidget	*timeout_label;
  GtkWidget	*timeout_spin;
  GtkWidget	*scale_label;
  GtkWidget	*scale_spin;
} PreferencesDialog;

/*** variables ***************************************************************/

static PreferencesDialog preferences = { NULL };

/*** functions ***************************************************************/

static void lm_preferences_selection_changed_h (GtkTreeSelection *selection,
						gpointer user_data);

static void lm_preferences_update_selected_label (void);
static void lm_preferences_update_sensitivity (void);

/*** implementation **********************************************************/

void
lm_preferences_display (void)
{
  GtkSizeGroup *size_group;
  GtkTreeSelection *selection;

  if (preferences.dialog)
    {
      gtk_window_present(GTK_WINDOW(preferences.dialog));
      return;
    }

  lm_create_interface("preferences",
		      "dialog", &preferences.dialog,
		      "scrolled", &preferences.scrolled,
		      "selected_label", &preferences.selected_label,
		      "remove", &preferences.remove,
		      "up", &preferences.up,
		      "down", &preferences.down,
		      "delay_label", &preferences.delay_label,
		      "delay_spin", &preferences.delay_spin,
		      "timeout_label", &preferences.timeout_label,
		      "timeout_spin", &preferences.timeout_spin,
		      "scale_label", &preferences.scale_label,
		      "scale_spin", &preferences.scale_spin,
		      NULL);

  g_object_add_weak_pointer(G_OBJECT(preferences.dialog), (gpointer *) &preferences.dialog);

  preferences.list = lm_host_view_new();
  gtk_container_add(GTK_CONTAINER(preferences.scrolled), preferences.list);
  gtk_widget_show(preferences.list);

  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(preferences.list));
  g_signal_connect(G_OBJECT(selection), "changed", G_CALLBACK(lm_preferences_selection_changed_h), NULL);

  g_signal_connect(G_OBJECT(preferences.list), "notify::can-go-up", G_CALLBACK(lm_preferences_update_sensitivity), NULL);
  g_signal_connect(G_OBJECT(preferences.list), "notify::can-go-down", G_CALLBACK(lm_preferences_update_sensitivity), NULL);

  size_group = gtk_size_group_new(GTK_SIZE_GROUP_HORIZONTAL);
  gtk_size_group_add_widget(size_group, preferences.delay_label);
  gtk_size_group_add_widget(size_group, preferences.timeout_label);
  gtk_size_group_add_widget(size_group, preferences.scale_label);
  g_object_unref(size_group);

  /* set the options */

  gtk_window_set_default_size(GTK_WINDOW(preferences.dialog),
			      lm_applet_get_preferences_width(lm_applet),
			      lm_applet_get_preferences_height(lm_applet));

  gtk_spin_button_set_value(GTK_SPIN_BUTTON(preferences.delay_spin), lm_applet_get_delay(lm_applet));
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(preferences.timeout_spin), lm_applet_get_timeout(lm_applet));
  gtk_spin_button_set_value(GTK_SPIN_BUTTON(preferences.scale_spin), lm_applet_get_scale(lm_applet));

  lm_preferences_update_selected_label();
  lm_preferences_update_sensitivity();

  gtk_widget_show(preferences.dialog);
}

static void
lm_preferences_selection_changed_h (GtkTreeSelection *selection,
				    gpointer user_data)
{
  lm_preferences_update_selected_label();
}

static void
lm_preferences_update_selected_label (void)
{
  GtkTreeSelection *selection;
  int n_rows;

  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(preferences.list));
  n_rows = gtk_tree_selection_count_selected_rows(selection);

  if (n_rows == 0)
    gtk_label_set_text(GTK_LABEL(preferences.selected_label), _("No host selected."));
  else
    {
      char *str;

      str = g_strdup_printf(ngettext("%i host selected.",
				     "%i hosts selected.",
				     n_rows), n_rows);
      gtk_label_set_text(GTK_LABEL(preferences.selected_label), str);
      g_free(str);
    }
}

static void
lm_preferences_update_sensitivity (void)
{
  GtkTreeSelection *selection;
  int n_rows;

  selection = gtk_tree_view_get_selection(GTK_TREE_VIEW(preferences.list));
  n_rows = gtk_tree_selection_count_selected_rows(selection);

  gtk_widget_set_sensitive(preferences.remove, n_rows > 0);
  gtk_widget_set_sensitive(preferences.up, lm_host_view_get_can_go_up(LM_HOST_VIEW(preferences.list)));
  gtk_widget_set_sensitive(preferences.down, lm_host_view_get_can_go_down(LM_HOST_VIEW(preferences.list)));
}

/* libglade callbacks */

void
lm_preferences_add_clicked_h (GtkButton *button, gpointer user_data)
{
  lm_host_view_activate_add(LM_HOST_VIEW(preferences.list));
}

void
lm_preferences_remove_clicked_h (GtkButton *button, gpointer user_data)
{
  lm_host_view_activate_remove(LM_HOST_VIEW(preferences.list));
}

void
lm_preferences_up_clicked_h (GtkButton *button, gpointer user_data)
{
  lm_host_view_activate_move(LM_HOST_VIEW(preferences.list), -1);
}

void
lm_preferences_down_clicked_h (GtkButton *button, gpointer user_data)
{
  lm_host_view_activate_move(LM_HOST_VIEW(preferences.list), +1);
}

void
lm_preferences_delay_spin_value_changed_h (GtkSpinButton *spinbutton,
					   gpointer user_data)
{
  lm_applet_set_delay(lm_applet, gtk_spin_button_get_value_as_int(spinbutton));
}

void
lm_preferences_timeout_spin_value_changed_h (GtkSpinButton *spinbutton,
					     gpointer user_data)
{
  lm_applet_set_timeout(lm_applet, gtk_spin_button_get_value_as_int(spinbutton));
}

void
lm_preferences_scale_spin_value_changed_h (GtkSpinButton *spinbutton,
					   gpointer user_data)
{
  lm_applet_set_scale(lm_applet, gtk_spin_button_get_value_as_int(spinbutton));
}

gboolean
lm_preferences_configure_event_h (GtkWidget *widget,
				  GdkEventConfigure *event,
				  gpointer user_data)
{
  lm_applet_set_preferences_width(lm_applet, event->width);
  lm_applet_set_preferences_height(lm_applet, event->height);

  return FALSE;			/* propagate event */
}

void
lm_preferences_response_h (GtkDialog *dialog,
			   int response,
			   gpointer user_data)
{
  switch (response)
    {
    case GTK_RESPONSE_HELP:
      lm_display_help("preferences");
      break;

    case GTK_RESPONSE_CLOSE:
      gtk_widget_destroy(preferences.dialog);
      break;
    }
}

void
lm_preferences_entry_activate_h (gpointer user_data, GtkEntry *entry)
{
  GtkWidget *next = user_data;

  gtk_widget_grab_focus(next);
}
