/*
 * $Id: lm-util.c,v 1.8 2004/08/15 00:38:37 jylefort Exp $
 *
 * Copyright (c) 1989, 1993 The Regents of the University of California
 * Copyright (c) 2004 Jean-Yves Lefort
 *
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. All advertising materials mentioning features or use of this software
 *    must display the following acknowledgement:
 *	This product includes software developed by the University of
 *	California, Berkeley and its contributors.
 * 4. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include <stdio.h>		/* required by stdlib.h on Darwin */
#include <stdlib.h>
#include <stdarg.h>
#ifdef TM_IN_SYS_TIME
#include <sys/time.h>
#else
#include <time.h>
#endif /* TM_IN_SYS_TIME */
#include <errno.h>
#include <gnome.h>
#include <glade/glade.h>
#include "lm-util.h"
#include "lm-dialog.h"

/*** implementation **********************************************************/

/*
 * Free a singly linked list of heap pointers.
 */
void
lm_slist_free (GSList *list)
{
  GSList *l;

  LM_LIST_FOREACH(l, list)
    g_free(l->data);

  g_slist_free(list);
}

void
lm_create_interface (const char *name, ...)
{
  char *filename;
  char *pathname;
  GladeXML *xml;
  va_list args;
  const char *widget_name;

  g_return_if_fail(name != NULL);

  filename = g_strconcat(name, ".glade", NULL);
  pathname = g_build_filename(UIDIR, filename, NULL);
  g_free(filename);

  xml = glade_xml_new(pathname, NULL, NULL);
  g_free(pathname);

  g_return_if_fail(xml != NULL);
  glade_xml_signal_autoconnect(xml);

  va_start(args, name);
  while ((widget_name = va_arg(args, const char *)))
    {
      GtkWidget **widget;

      widget = va_arg(args, GtkWidget **);
      g_return_if_fail(widget != NULL);

      *widget = glade_xml_get_widget(xml, widget_name);
      if (! *widget)
	g_critical(_("widget %s not found in interface %s"), widget_name, name);
    }
  va_end(args);
  
  g_object_unref(xml);
}

GdkPixbuf *
lm_pixbuf_new (const char *filename)
{
  char *pathname;
  GdkPixbuf *pixbuf;
  GError *err = NULL;

  g_return_val_if_fail(filename != NULL, NULL);

  pathname = g_build_filename(UIDIR, filename, NULL);
  pixbuf = gdk_pixbuf_new_from_file(pathname, &err);
  g_free(pathname);

  if (! pixbuf)
    {
      g_warning(_("error loading image: %s"), err->message);
      g_error_free(err);
    }

  return pixbuf;
}

void
lm_thread_create (GThreadFunc func, gpointer data)
{
  GError *err = NULL;

  g_return_if_fail(func != NULL);
  
  if (! g_thread_create(func, data, FALSE, &err))
    {
      lm_error_dialog(_("A fatal error has occurred in Link Monitor"),
		      _("Unable to create a thread: %s."),
		      err->message);
      g_error_free(err);
      exit(1);
    }
}

#define STRDUP_FTIME_BUFSIZE_INCREMENT	1024
#define STRDUP_FTIME_MAX_BUFSIZE	0xFFFF

time_t
lm_time (void)
{
  time_t t;

  t = time(NULL);
  if (t < 0)
    {
      g_warning(_("unable to get current time: %s"), g_strerror(errno));
      t = 0;
    }

  return t;
}

char *
lm_strdup_ftime (const char *format, const struct tm *timeptr)
{
  char *buf;
  size_t bufsize = STRDUP_FTIME_BUFSIZE_INCREMENT;

  g_return_val_if_fail(format != NULL, NULL);
  g_return_val_if_fail(timeptr != NULL, NULL);

  buf = g_malloc(bufsize);
  while (strftime(buf, bufsize, format, timeptr) == 0)
    {
      bufsize += STRDUP_FTIME_MAX_BUFSIZE;
      g_assert(bufsize <= STRDUP_FTIME_MAX_BUFSIZE);
      buf = g_realloc(buf, bufsize);
    }

  return buf;
}

/*
 * Subtract 2 timeval structs: out = out - in. Out is assumed to be >=
 * in (based on tvsub() from FreeBSD's ping.c).
 */
void
lm_tvsub (GTimeVal *out, const GTimeVal *in)
{
  g_return_if_fail(out != NULL);
  g_return_if_fail(in != NULL);

  if ((out->tv_usec -= in->tv_usec) < 0)
    {
      --out->tv_sec;
      out->tv_usec += 1000000;
    }
  out->tv_sec -= in->tv_sec;
}

void
lm_display_help (const char *link_id)
{
  GError *err = NULL;

  if (! gnome_help_display("link-monitor-applet.xml", link_id, &err))
    {
      lm_error_dialog(_("Unable to display help"), "%s", err->message);
      g_error_free(err);
    }
}
