/*
 * $Id: lm-host.c,v 1.7 2004/07/22 21:12:12 jylefort Exp $
 *
 * Copyright (c) 2004 Jean-Yves Lefort
 * All rights reserved.
 *
 * Redistribution and use in source and binary forms, with or without
 * modification, are permitted provided that the following conditions
 * are met:
 * 1. Redistributions of source code must retain the above copyright
 *    notice, this list of conditions and the following disclaimer.
 * 2. Redistributions in binary form must reproduce the above copyright
 *    notice, this list of conditions and the following disclaimer in the
 *    documentation and/or other materials provided with the distribution.
 * 3. Neither the name of Jean-Yves Lefort nor the names of its contributors
 *    may be used to endorse or promote products derived from this software
 *    without specific prior written permission.
 *
 * THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
 * CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES,
 * INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES OF
 * MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE
 * DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT OWNER OR CONTRIBUTORS
 * BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL,
 * EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
 * TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE,
 * DATA, OR PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON
 * ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY,
 * OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
 * OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
 * POSSIBILITY OF SUCH DAMAGE.
 */

#include "config.h"
#include <stdio.h>		/* required by stdlib.h on Darwin */
#include <stdlib.h>		/* required by sys/socket.h on Darwin */
#include <string.h>
#include <stdarg.h>
#include <sys/types.h>
#include <sys/socket.h>
#include <netdb.h>
#include <glib/gi18n-lib.h>
#include <glib-object.h>
#include "lm-host.h"
#include "lm-icmp.h"
#include "lm-sockets.h"
#include "lm-util.h"

/*** types *******************************************************************/

enum {
  PROP_0,
  PROP_NAME,
  PROP_IP,
  PROP_SEQ,
  PROP_DELAY,
  PROP_TIMEOUT,
  PROP_ALIVE,
  PROP_ROUNDTRIP_TIME,
  PROP_ERROR,
  PROP_STATUS
};
    
struct _LMHostPrivate
{
  char			*name;
  struct addrinfo	*addrinfo;
  char			*ip;
  unsigned int		seq;
  unsigned int		delay;
  unsigned int		timeout;
  gboolean		enabled;
  gboolean		alive;
  unsigned int		send_timeout_id;
  unsigned int		dead_timeout_id;
  double		roundtrip_time;
  char			*error;
  time_t		last_sent;
  time_t		last_received;
  char			*status;
};

/*** variables ***************************************************************/

static GObjectClass *parent_class = NULL;

/*** functions ***************************************************************/

static void lm_host_class_init (LMHostClass *class);
static void lm_host_init (LMHost *host);

static GObject *lm_host_constructor (GType type,
				     guint n_construct_properties,
				     GObjectConstructParam *construct_params);
static gpointer lm_host_constructor_thread_cb (gpointer data);

static void lm_host_finalize (GObject *object);

static void lm_host_set_property (GObject *object,
				  guint prop_id,
				  const GValue *value,
				  GParamSpec *pspec);
static void lm_host_get_property (GObject *object,
				  unsigned int prop_id,
				  GValue *value,
				  GParamSpec *pspec);

static void lm_host_set_alive (LMHost *host, gboolean alive);
static void lm_host_set_status (LMHost *host, const char *format, ...);

static void lm_host_send_echo_request (LMHost *host);

static void lm_host_install_send_timeout (LMHost *host);
static gboolean lm_host_send_timeout_cb (gpointer data);
static gboolean lm_host_dead_timeout_cb (gpointer data);

/*** implementation **********************************************************/

GType
lm_host_get_type (void)
{
  static GType host_type = 0;
  
  if (! host_type)
    {
      static const GTypeInfo host_info = {
	sizeof(LMHostClass),
	NULL,
	NULL,
	(GClassInitFunc) lm_host_class_init,
	NULL,
	NULL,
	sizeof(LMHost),
	0,
	(GInstanceInitFunc) lm_host_init,
      };
      
      host_type = g_type_register_static(G_TYPE_OBJECT,
					 "LMHost",
					 &host_info,
					 0);
    }
  
  return host_type;
}

static void
lm_host_class_init (LMHostClass *class)
{
  GObjectClass *object_class = G_OBJECT_CLASS(class);

  parent_class = g_type_class_peek_parent(class);

  object_class->constructor = lm_host_constructor;
  object_class->finalize = lm_host_finalize;
  object_class->set_property = lm_host_set_property;
  object_class->get_property = lm_host_get_property;

  g_object_class_install_property(object_class,
                                  PROP_NAME,
                                  g_param_spec_string("name",
                                                      _("Name"),
                                                      _("The hostname or IP address as given by the user"),
                                                      NULL,
                                                      G_PARAM_WRITABLE | G_PARAM_READABLE | G_PARAM_CONSTRUCT_ONLY));
  g_object_class_install_property(object_class,
                                  PROP_IP,
                                  g_param_spec_string("ip",
                                                      _("IP"),
                                                      _("The string representation of the IP address"),
                                                      NULL,
                                                      G_PARAM_READABLE));
  g_object_class_install_property(object_class,
                                  PROP_SEQ,
                                  g_param_spec_uint("seq",
						    _("Sequence number"),
						    _("The sequence number assigned to the host"),
						    0,
						    G_MAXUINT,
						    0,
						    G_PARAM_WRITABLE | G_PARAM_READABLE | G_PARAM_CONSTRUCT_ONLY));
  g_object_class_install_property(object_class,
                                  PROP_DELAY,
                                  g_param_spec_uint("delay",
						    _("Delay"),
						    _("The amount of time to wait between sending each echo request"),
						    LM_HOST_MIN_DELAY,
						    G_MAXUINT,
						    LM_HOST_MIN_DELAY,
						    G_PARAM_WRITABLE | G_PARAM_READABLE | G_PARAM_CONSTRUCT));
  g_object_class_install_property(object_class,
                                  PROP_TIMEOUT,
                                  g_param_spec_uint("timeout",
						    _("Timeout"),
						    _("The delay after which the host is considered dead if no reply was received"),
						    LM_HOST_MIN_TIMEOUT,
						    G_MAXUINT,
						    LM_HOST_MIN_TIMEOUT,
						    G_PARAM_WRITABLE | G_PARAM_READABLE | G_PARAM_CONSTRUCT));
  g_object_class_install_property(object_class,
                                  PROP_ALIVE,
                                  g_param_spec_boolean("alive",
						       _("Alive"),
						       _("Whether the host is alive or not"),
						       NULL,
						       G_PARAM_READABLE));
  g_object_class_install_property(object_class,
                                  PROP_ROUNDTRIP_TIME,
                                  g_param_spec_double("roundtrip-time",
						      _("Round-trip time"),
						      _("The last round-trip time"),
						      0,
						      G_MAXDOUBLE,
						      0,
						      G_PARAM_WRITABLE | G_PARAM_READABLE));
  g_object_class_install_property(object_class,
                                  PROP_ERROR,
                                  g_param_spec_string("error",
                                                      _("Error"),
                                                      _("The last error"),
                                                      NULL,
                                                      G_PARAM_READABLE | G_PARAM_WRITABLE));
  g_object_class_install_property(object_class,
                                  PROP_STATUS,
                                  g_param_spec_string("status",
                                                      _("Status"),
                                                      _("The status of the host"),
                                                      NULL,
                                                      G_PARAM_READABLE));
}

static void
lm_host_init (LMHost *host)
{
  host->priv = g_new0(LMHostPrivate, 1);
}

static GObject *
lm_host_constructor (GType type,
		     guint n_construct_properties,
		     GObjectConstructParam *construct_params)
{
  GObject *object;
  LMHost *host;
  
  object = G_OBJECT_CLASS(parent_class)->constructor(type, n_construct_properties, construct_params);
  host = LM_HOST(object);

  g_object_ref(host);
  lm_thread_create(lm_host_constructor_thread_cb, host);

  return object;
}

static gpointer
lm_host_constructor_thread_cb (gpointer data)
{
  LMHost *host = data;
  struct addrinfo hints;
  int status;

  /*
   * For design simplicity we don't use our own LMHost mutex; rather,
   * we run every non-blocking operation with the GDK lock held.
   *
   * The only LMHost member accessed outside of the lock is
   * host->priv->addrinfo; we are sure it can not be accessed by
   * multiple threads concurrently.
   */

  GDK_THREADS_ENTER();

  lm_host_set_status(host, _("resolving"));

  memset(&hints, 0, sizeof(hints));
#ifdef WITH_IPV6
  hints.ai_family = PF_UNSPEC;
#else
  hints.ai_family = PF_INET;
#endif /* WITH_IPV6 */
  hints.ai_socktype = SOCK_RAW;

  /*
   * A note on gdk_flush(): as adviced in the GDK threads
   * documentation, we only call gdk_flush() from a thread other than
   * our main thread.
   */
  
  gdk_flush();
  GDK_THREADS_LEAVE();

  status = getaddrinfo(host->priv->name, NULL, &hints, &host->priv->addrinfo);

  GDK_THREADS_ENTER();

  if (status == 0)
    {
      char ip[NI_MAXHOST];
      const LMSocket *s;

      gdk_flush();
      GDK_THREADS_LEAVE();

      status = getnameinfo(host->priv->addrinfo->ai_addr,
			   host->priv->addrinfo->ai_addrlen,
			   ip,
			   sizeof(ip),
			   NULL,
			   0,
			   NI_NUMERICHOST);

      GDK_THREADS_ENTER();

      if (status == 0)
	host->priv->ip = g_strdup(ip);
	
      s = lm_sockets_get_by_domain(host->priv->addrinfo->ai_family);
      if (s)
	{
	  if (! s->error)
	    {
	      host->priv->enabled = TRUE;
	      lm_host_set_status(host, _("starting"));
	    }
	  else
	    lm_host_set_status(host, _("socket could not be initialized: %s"), s->error);
	}
      else
	lm_host_set_status(host, _("unsupported address family"));
    }
  else
    lm_host_set_status(host, _("unable to resolve: %s"), gai_strerror(status));

  if (host->priv->enabled)
    {
      lm_host_send_echo_request(host);		/* send first packet... */
      lm_host_install_send_timeout(host);	/* ...and install send loop */
    }
  g_object_unref(host);

  gdk_flush();
  GDK_THREADS_LEAVE();

  return NULL;
}

static void
lm_host_finalize (GObject *object)
{
  LMHost *host = LM_HOST(object);

  if (host->priv->send_timeout_id)
    g_source_remove(host->priv->send_timeout_id);
  if (host->priv->dead_timeout_id)
    g_source_remove(host->priv->dead_timeout_id);
  g_free(host->priv->name);
  if (host->priv->addrinfo)
    freeaddrinfo(host->priv->addrinfo);
  g_free(host->priv->ip);
  g_free(host->priv->error);
  g_free(host->priv->status);
  g_free(host->priv);

  G_OBJECT_CLASS(parent_class)->finalize(object);
}

static void
lm_host_set_property (GObject *object,
		      guint prop_id,
		      const GValue *value,
		      GParamSpec *pspec)
{
  LMHost *host = LM_HOST(object);

  switch (prop_id)
    {
    case PROP_NAME:
      g_return_if_fail(host->priv->name == NULL);
      host->priv->name = g_value_dup_string(value);
      break;

    case PROP_SEQ:
      host->priv->seq = g_value_get_uint(value);
      break;

    case PROP_DELAY:
      host->priv->delay = g_value_get_uint(value);
      if (host->priv->enabled)
	lm_host_install_send_timeout(host);
      break;

    case PROP_TIMEOUT:
      host->priv->timeout = g_value_get_uint(value);
      break;

    case PROP_ROUNDTRIP_TIME:
      host->priv->last_received = lm_time();
      if (host->priv->dead_timeout_id)
	{
	  g_source_remove(host->priv->dead_timeout_id);
	  host->priv->dead_timeout_id = 0;
	}
      host->priv->roundtrip_time = g_value_get_double(value);
      lm_host_set_alive(host, TRUE);
      lm_host_set_status(host, _("%.3f ms"), host->priv->roundtrip_time);
      break;

    case PROP_ERROR:
      if (host->priv->dead_timeout_id)
	{
	  g_source_remove(host->priv->dead_timeout_id);
	  host->priv->dead_timeout_id = 0;
	}
      g_free(host->priv->error);
      host->priv->error = g_value_dup_string(value);
      lm_host_set_alive(host, FALSE);
      lm_host_set_status(host, "%s", host->priv->error);
      break;
      
    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
      break;
    }
}

void
lm_host_set_delay (LMHost *host, unsigned int delay)
{
  g_return_if_fail(LM_IS_HOST(host));

  g_object_set(G_OBJECT(host), "delay", delay, NULL);
}

void
lm_host_set_timeout (LMHost *host, unsigned int timeout)
{
  g_return_if_fail(LM_IS_HOST(host));

  g_object_set(G_OBJECT(host), "timeout", timeout, NULL);
}

static void
lm_host_set_alive (LMHost *host, gboolean alive)
{
  g_return_if_fail(LM_IS_HOST(host));

  host->priv->alive = alive;
  g_object_notify(G_OBJECT(host), "alive");
}

void
lm_host_set_roundtrip_time (LMHost *host, double roundtrip_time)
{
  g_return_if_fail(LM_IS_HOST(host));

  g_object_set(G_OBJECT(host), "roundtrip-time", roundtrip_time, NULL);
}

void
lm_host_set_error (LMHost *host, const char *error)
{
  g_return_if_fail(LM_IS_HOST(host));
  g_return_if_fail(error != NULL);

  g_object_set(G_OBJECT(host), "error", error, NULL);
}

static void
lm_host_set_status (LMHost *host, const char *format, ...)
{
  va_list args;

  g_return_if_fail(LM_IS_HOST(host));
  g_return_if_fail(format != NULL);

  g_free(host->priv->status);

  va_start(args, format);
  host->priv->status = g_strdup_vprintf(format, args);
  va_end(args);

  g_object_notify(G_OBJECT(host), "status");
}

static void
lm_host_get_property (GObject *object,
		      unsigned int prop_id,
		      GValue *value,
		      GParamSpec *pspec)
{
  LMHost *host = LM_HOST(object);

  switch (prop_id)
    {
    case PROP_NAME:
      g_value_set_string(value, lm_host_get_name(host));
      break;

    case PROP_IP:
      g_value_set_string(value, lm_host_get_ip(host));
      break;

    case PROP_DELAY:
      g_value_set_uint(value, lm_host_get_delay(host));
      break;

    case PROP_TIMEOUT:
      g_value_set_uint(value, lm_host_get_timeout(host));
      break;

    case PROP_SEQ:
      g_value_set_uint(value, lm_host_get_seq(host));
      break;

    case PROP_ALIVE:
      g_value_set_boolean(value, lm_host_get_alive(host));
      break;

    case PROP_ROUNDTRIP_TIME:
      g_value_set_double(value, lm_host_get_roundtrip_time(host));
      break;

    case PROP_ERROR:
      g_value_set_string(value, lm_host_get_error(host));
      break;

    case PROP_STATUS:
      g_value_set_string(value, lm_host_get_status(host));
      break;

    default:
      G_OBJECT_WARN_INVALID_PROPERTY_ID(object, prop_id, pspec);
      break;
    }
}

const char *
lm_host_get_name (LMHost *host)
{
  g_return_val_if_fail(LM_IS_HOST(host), NULL);

  return host->priv->name;
}

const char *
lm_host_get_ip (LMHost *host)
{
  g_return_val_if_fail(LM_IS_HOST(host), NULL);

  return host->priv->ip;
}

unsigned int
lm_host_get_delay (LMHost *host)
{
  g_return_val_if_fail(LM_IS_HOST(host), 0);

  return host->priv->delay;
}

unsigned int
lm_host_get_timeout (LMHost *host)
{
  g_return_val_if_fail(LM_IS_HOST(host), 0);

  return host->priv->timeout;
}

unsigned int
lm_host_get_seq (LMHost *host)
{
  g_return_val_if_fail(LM_IS_HOST(host), 0);

  return host->priv->seq;
}

gboolean
lm_host_get_alive (LMHost *host)
{
  g_return_val_if_fail(LM_IS_HOST(host), FALSE);

  return host->priv->alive;
}

double
lm_host_get_roundtrip_time (LMHost *host)
{
  g_return_val_if_fail(LM_IS_HOST(host), 0);

  return host->priv->roundtrip_time;
}

const char *
lm_host_get_error (LMHost *host)
{
  g_return_val_if_fail(LM_IS_HOST(host), NULL);

  return host->priv->error;
}

const char *
lm_host_get_status (LMHost *host)
{
  g_return_val_if_fail(LM_IS_HOST(host), NULL);

  return host->priv->status;
}

static void
lm_host_send_echo_request (LMHost *host)
{
  GError *err = NULL;

  g_return_if_fail(LM_IS_HOST(host));
  g_return_if_fail(host->priv->enabled == TRUE);

  if (lm_icmp_send_echo_request(host->priv->addrinfo, host->priv->seq, &err))
    {
      host->priv->last_sent = lm_time();
      if (! host->priv->dead_timeout_id)
	host->priv->dead_timeout_id = g_timeout_add(host->priv->timeout, lm_host_dead_timeout_cb, host);
    }
  else
    {
      lm_host_set_alive(host, FALSE);
      if (host->priv->last_sent)
	{
	  struct tm *timeptr;
	  char *date;
	  char *time;
	  
	  timeptr = localtime(&host->priv->last_sent);
	  date = lm_strdup_ftime("%x", timeptr);
	  time = lm_strdup_ftime("%X", timeptr);
	  lm_host_set_status(host, _("unable to send echo request: %s (last request successfully sent on %s at %s)"), err->message, date, time);
	  g_free(date);
	  g_free(time);
	}
      else
	lm_host_set_status(host, _("unable to send echo request: %s"), err->message);
      g_error_free(err);
    }
}

static void
lm_host_install_send_timeout (LMHost *host)
{
  g_return_if_fail(LM_IS_HOST(host));
  g_return_if_fail(host->priv->enabled == TRUE);

  if (host->priv->send_timeout_id)
    {
      g_source_remove(host->priv->send_timeout_id);
      host->priv->send_timeout_id = 0;
    }
  if (host->priv->enabled)
    host->priv->send_timeout_id = g_timeout_add(host->priv->delay, lm_host_send_timeout_cb, host);
}

static gboolean
lm_host_send_timeout_cb (gpointer data)
{
  LMHost *host = data;

  GDK_THREADS_ENTER();

  lm_host_send_echo_request(host);

  GDK_THREADS_LEAVE();

  return TRUE;			/* keep source */
}

static gboolean
lm_host_dead_timeout_cb (gpointer data)
{
  LMHost *host = data;

  GDK_THREADS_ENTER();

  lm_host_set_alive(host, FALSE);
  if (host->priv->last_received)
    {
      struct tm *timeptr;
      char *date;
      char *time;

      timeptr = localtime(&host->priv->last_received);
      date = lm_strdup_ftime("%x", timeptr);
      time = lm_strdup_ftime("%X", timeptr);
      lm_host_set_status(host, _("last reply received on %s at %s"), date, time);
      g_free(date);
      g_free(time);
    }
  else
    lm_host_set_status(host, _("no reply"));

  host->priv->dead_timeout_id = 0;

  GDK_THREADS_LEAVE();

  return FALSE;			/* remove source */
}

LMHost *
lm_host_new (const char *name,
	     unsigned int seq,
	     unsigned int delay,
	     unsigned int timeout)
{
  return g_object_new(LM_TYPE_HOST,
		      "name", name,
		      "seq", seq,
		      "delay", delay,
		      "timeout", timeout,
		      NULL);
}
