/*  Copyright (C) 2013, 2014 Ben Asselstine

  This program is free software; you can redistribute it and/or modify
  it under the terms of the GNU General Public License as published by
  the Free Software Foundation; either version 3 of the License, or
  (at your option) any later version.

  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.

  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
  02110-1301, USA.
*/
#include <config.h>
#include <stdlib.h>
#include <unistd.h>
#include "licensing_priv.h"
#include "gpl.h"
#include "gettext-more.h"
#include "xvasprintf.h"
#include "read-file.h"
#include "error.h"
#include "util.h"
#include "url-downloader.h"

static struct argp_option argp_options[] = 
{
    {"full", 'f', NULL, 0, N_("show the full license text")},
    {"v1", '1', NULL, 0, N_("show version 1 of the gpl")},
    {"v2", '2', NULL, 0, N_("show version 2 of the gpl")},
    {"v3", '3', NULL, 0, N_("show version 3 of the gpl")},
    {"jerkwad", 'j', NULL, 0, N_("remove the or-any-later-version clause")},
    {"list-license-notices", 'l', NULL, OPTION_HIDDEN, N_("show licenses and exit")},
    {0}
};

static error_t 
parse_opt (int key, char *arg, struct argp_state *state)
{
  struct lu_gpl_options_t *opt = NULL;
  if (state)
    opt = (struct lu_gpl_options_t*) state->input;
  switch (key)
    {
    case 'l':
        {
          int i = 0;
          while (gpl.notices[i].keyword != NULL)
            {
              fprintf (stdout, "%s\n", gpl.notices[i].keyword);
              i++;
            }
          exit (0);
        }
      break;
    case 'j':
      opt->future_versions = 0;
      break;
    case 'f':
      opt->full = 1;
      break;
    case '1':
      opt->version = 1;
      break;
    case '2':
      opt->version = 2;
      break;
    case '3':
      opt->version = 3;
      break;
    case ARGP_KEY_INIT:
      opt->full = 0;
      opt->version = 3;
      opt->future_versions = 1;
      opt->fsf_address = 0;
      state->child_inputs[0] = &opt->fsf_address;
      break;
    case ARGP_KEY_FINI:
      if (opt->future_versions == 0 && opt->full)
        {
          argp_failure (state, 0, 0, 
                        N_("--jerkwad cannot be used with --full"));
          argp_state_help (state, stderr, ARGP_HELP_STD_ERR);
        }
      else if (opt->fsf_address && opt->full)
        {
          argp_failure (state, 0, 0, 
                        N_("cannot use an address option with --full"));
          argp_state_help (state, stderr, ARGP_HELP_STD_ERR);
        }
      break;
    default:
      return ARGP_ERR_UNKNOWN;
    }
  return 0;
}

static const struct argp_child
parsers[] =
{
    {&fsf_addresses_argp},
    { 0 },
};

#undef GPL_DOC
#define GPL_DOC N_("Show the GNU General Public License notice.")

static struct argp argp = { argp_options, parse_opt, "",  GPL_DOC, parsers};

int 
lu_gpl_parse_argp (struct lu_state_t *state, int argc, char **argv)
{
  int err = 0;
  struct lu_gpl_options_t opts;
  opts.state = state;

  err = argp_parse (&argp, argc, argv, state->argp_flags,  0, &opts);
  if (!err)
    return lu_gpl (state, &opts);
  else
    return err;
}

int
show_lu_gpl(struct lu_state_t *state, struct lu_gpl_options_t *options)
{
  char *file;
  if (options->version == 1)
    file = strdup ("gpl-1.0");
  else if (options->version == 2)
    file = strdup ("gpl-2.0");
  else
    file = strdup ("gpl");
  char *url = xasprintf ("%s/licenses/%s%s.%s", GNU_SITE, file,
                         "", "txt");
  free (file);
  char *data = NULL;
  int err = download (state, url, &data);
  free (url);

  if (options->full)
    luprintf (state, "%s\n", data);
  else
    {
      char *chunk = NULL;
      switch (options->version)
        {
        case 1:
          chunk = get_lines (data, "    This program is free software;", 13);
          break;
        case 2:
          chunk = get_lines (data, "    This program is free software;", 13);
          break;
        case 3:
          chunk = get_lines (data, "    This program is free software:", 12);
          break;
        }

      if (!options->future_versions)
        {
          switch (options->version)
            {
            case 1:
              err = text_replace (chunk, "either version 1, or (at your option)\n    any later version.", "version 1 of the License.");
              break;
            case 2:
              err = text_replace (chunk, "either version 2 of the License, or\n    (at your option) any later version.", "version 2 of the License.");
              break;
            case 3:
              err = text_replace (chunk, "either version 3 of the License, or\n    (at your option) any later version.", "version 3 of the License.");
              break;
            }
        }

      if (options->fsf_address)
        replace_fsf_address (&chunk, options->fsf_address, "", 4);

      luprintf (state, "%s\n", chunk);
      free (chunk);
    }
  free (data);
  return err;
}

int 
lu_gpl (struct lu_state_t *state, struct lu_gpl_options_t *options)
{
  int err = 0;
  err = show_lu_gpl(state, options);
  return err;
}

struct lu_command_t gpl = 
{
  .name         = N_("gpl"),
  .doc          = GPL_DOC,
  .flags        = IS_A_LICENSE | SAVE_IN_HISTORY | SHOW_IN_HELP,
  .argp         = &argp,
  .parser       = lu_gpl_parse_argp,
  .latest_idx   = 3,
  .mentions =
    {
      "the GPL",
      "General Public License",
      "GNU Public License",
      "gplv2",
      "gplv3",
      " GPL",
      NULL
    },
  .notices      =
    {
        {
          .keyword   = "gplv3+,full-license",
          .cmd       = "gpl -3 --full",
          .words     = "ersion 3",
          .recommend = 1
        },
        {
          .keyword   = "gplv2+,full-license",
          .cmd       = "gpl -2 --full",
          .words     = "ersion 2",
          .recommend = 1
        },
        {
          .keyword   = "gplv1+,full-license",
          .cmd       = "gpl -1 --full",
          .words     = "ersion 1",
          .recommend = 1
        },
        {
          .keyword   = "gplv3+",
          .cmd       = " gpl --v3 --link",
          .words     = "ersion 3",
          .recommend = 1
        },
        {
          .keyword   = "gplv3,no-later",
          .cmd       = "gpl --v3 --jerkwad --link",
          .words     = "ersion 3,http",
          .recommend = 0
        },
        {
          .keyword   = "gplv2+",
          .cmd       = "gpl --v2 --link",
          .words     = "ersion 2,http",
          .recommend = 1
        },
        {
          .keyword   = "gplv2,no-later",
          .cmd       = "gpl --v2 --jerkwad --link",
          .words     = "ersion 2,http",
          .recommend = 0
        },
        {
          .keyword   = "gplv1+",
          .cmd       = "gpl --v1 --link",
          .words     = "ersion 1,http",
          .recommend = 1
        },
        {
          .keyword   = "gplv1,no-later",
          .cmd       = "gpl --v1 --jerkwad --link",
          .words     = "ersion 1,http",
          .recommend = 0
        },
        {
          .keyword   = "gplv1,no-later,mass",
          .cmd       = "gpl --v1 --jerkwad --mass",
          .words     = "ersion 1,Mass",
          .recommend = 0
        },
        {
          .keyword   = "gplv1+,mass",
          .cmd       = "gpl --v1 --mass",
          .words     = "ersion 1,Mass",
          .recommend = 1
        },
        {
          .keyword   = "gplv2,no-later,temple",
          .cmd       = "gpl --v2 --jerkwad --temple",
          .words     = "ersion 2,Temple",
          .recommend = 0
        },
        {
          .keyword   = "gplv2+,temple",
          .cmd       = "gpl --v2 --temple",
          .words     = "ersion 2,Temple",
          .recommend = 1
        },
        {
          .keyword   = "gplv2,no-later,franklin",
          .cmd       = "gpl --v2 --jerkwad --franklin",
          .words     = "ersion 2,Franklin",
          .recommend = 0
        },
        {
          .keyword   = "gplv2+,franklin",
          .cmd       = "gpl --v2 --franklin",
          .words     = "ersion 2,Franklin",
          .recommend = 1
        },
        {
          .keyword   = "gplv3,no-later,franklin",
          .cmd       = "gpl --v3 --jerkwad --franklin",
          .words     = "ersion 3,Franklin",
          .recommend = 0
        },
        {
          .keyword   = "gplv3+,franklin",
          .cmd       = "gpl --v3 --franklin",
          .words     = "ersion 3,Franklin",
          .recommend = 1
        },
        {
          .keyword   = NULL,
          .cmd       = NULL,
          .words     = NULL,
          .recommend = 0
        }
    }
};
