;;;; common.lisp --- FFI definitions for libxml-clisp

;;; Copyright (C) 2009 N. Raghavendra.  All rights reserved.
;;; 
;;; Redistribution and use in source and binary forms, with or without
;;; modification, are permitted provided that the following conditions
;;; are met:
;;; 1. Redistributions of source code must retain the above copyright
;;;    notice, this list of conditions and the following disclaimer.
;;; 2. Redistributions in binary form must reproduce the above
;;;    copyright notice, this list of conditions and the following
;;;    disclaimer in the documentation and/or other materials provided
;;;    with the distribution.
;;; 
;;; THIS SOFTWARE IS PROVIDED BY THE AUTHOR "AS IS" AND ANY EXPRESS OR
;;; IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
;;; WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
;;; ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY
;;; DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
;;; DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
;;; GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
;;; INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
;;; WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
;;; NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
;;; SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

;;; N. Raghavendra <raghu@retrotexts.net>
;;; 
;;; Created: 2009-07-09
;;; 
;;; $Hg: common.lisp,v ae6366215b15 2009-09-03T20:01:35+05:30 raghu $

(in-package "NET.RETROTEXTS.LIBXML-CLISP")



;;;; Library version and initialization

;;; See LIBXML_TEST_VERSION in `xmlversion.h'.

(defun init-library ()
  "Initialize libxml2.
Also checks potential ABI mismatches between the version it was
compiled for, and the version of the actual shared library used."
  ($xml-check-version ($lxcl-libxml-version)))



;;;; Conditions

(define-condition address-error (error)
  ((address :initarg :address :reader error-address))
  (:report (lambda (condition stream)
             (format stream "Unable to process the address ~A."
                     (error-address condition)))))

(define-condition null-address-error (error)
  ()
  (:report (lambda (condition stream)
             (format stream "The address ~A is null."
                     (error-address condition)))))

(define-condition item-error (error)
  ((item :initarg :item :reader error-item))
  (:report (lambda (condition stream)
             (format stream "Unable to process the Item ~A."
                     (error-item condition)))))

(define-condition null-item-error (item-error)
  ()
  (:report (lambda (condition stream)
             (format stream "The Item ~A is a null item."
                     (error-item condition)))))

(eval-when (:compile-toplevel :load-toplevel :execute)
  (setf (symbol-function 'error-pathname) #'file-error-pathname))

(define-condition xml-file-error (file-error)
  ()
  (:report (lambda (condition stream)
             (format stream "Unable to process file ~A."
                     (error-pathname condition)))))

(define-condition file-parse-error (xml-file-error)
  ()
  (:report (lambda (condition stream)
             (format stream "Unable to parse file ~A."
                     (error-pathname condition)))))

(define-condition file-exists-error (xml-file-error)
  ()
  (:report (lambda (condition stream)
             (format stream "File already exists: ~A."
                     (error-pathname condition)))))

(defun read-new-value ()
  "Read a value supplied by the user."
  (format t "Enter a new value: ")
  (multiple-value-list (eval (read))))



;;;; Memory management.

(defgeneric free-item (item)
  (:documentation
   "Free the memory occupied by the foreign object wrapped in ITEM.
If ITEM is a null item, nothing is done."))

(defmethod free-item :around (item)
  (unless (null-item-p item)
    (call-next-method)
    (setf (item-address item) nil)))

(defmethod free-item (item)
  ($lxcl-free (item-address item)))



;;;; Miscellany.

(defun funcall-if-true (function object)
  "Return the value of FUNCTION on OBJECT if the latter is true.
Returns false if OBJECT is false."
  (and object (funcall function object)))

(defvar *xml-output* *standard-output*
  "The default output stream for general libxml-clisp operations.")

(defvar *xml-input* *standard-input*
  "The default input stream for general libxml-clisp operations.")

(defmacro with-xml-output ((stream) &body body)
  "Evaluate BODY with STREAM as the default XML output stream."
  `(let ((*xml-output* ,stream))
     ,@body))

(defvar *encoding-names* (make-hash-table)
  "Map from CLISP encodings to IANA charset names.
See http://clisp.cons.org/impnotes/encoding.html#charset and
http://www.iana.org/assignments/character-sets")

;;; TODO.  Add below all the encodings at
;;; http://clisp.cons.org/impnotes/encoding.html#charset

(loop for (encodings name) in
     '((("UCS-2" "UNICODE-16" "UNICODE-16-BIG-ENDIAN") "UTF-16BE")
       (("UNICODE-16-LITTLE-ENDIAN") "UTF-16LE")
       (("UCS-4" "UNICODE-32" "UNICODE-32-BIG-ENDIAN") "UTF-32BE")
       (("UNICODE-32-LITTLE-ENDIAN") "UTF-32LE")
       ;; TODO.  Libxml2 treats UTF-16 as little-endian, whereas CLISP
       ;; treats it as big-endian.  When converting a UTF-16 foreign
       ;; string (i.e., octet-vector) returned by Libxml2 to a Lisp
       ;; string, this has to be taken care of.  See RFC 2781.
       (("UTF-16") "UTF-16")
       (("UTF-8") "UTF-8")
       (("ISO-8859-1") "ISO-8859-1")
       (("ASCII") "ASCII"))
   do (loop for encoding in encodings
         do (setf (gethash (ext:make-encoding :charset encoding)
                           *encoding-names*)
                    name)))

(defun encoding-name (encoding)
  "Return the name of the encoding ENCODING.
Returns false if ENCODING is false."
  (and encoding
       (gethash (ext:make-encoding :charset encoding) *encoding-names*)))



;;; Local Variables:
;;; mode: lisp
;;; comment-column: 32
;;; End:

;;;; common.lisp ends here