;;;; keyword-xpath.lisp --- example for libxml-clisp tutorial

;;; Copyright (C) 2009 N. Raghavendra.  All rights reserved.
;;; 
;;; Redistribution and use in source and binary forms, with or without
;;; modification, are permitted provided that the following conditions
;;; are met:
;;; 1. Redistributions of source code must retain the above copyright
;;;    notice, this list of conditions and the following disclaimer.
;;; 2. Redistributions in binary form must reproduce the above
;;;    copyright notice, this list of conditions and the following
;;;    disclaimer in the documentation and/or other materials provided
;;;    with the distribution.
;;; 
;;; THIS SOFTWARE IS PROVIDED BY THE AUTHOR "AS IS" AND ANY EXPRESS OR
;;; IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
;;; WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
;;; ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY
;;; DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
;;; DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
;;; GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
;;; INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
;;; WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
;;; NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
;;; SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

;;; N. Raghavendra <raghu@retrotexts.net>
;;; 
;;; Created: 2009-08-23
;;; 
;;; $Hg: keyword-xpath.lisp,v ae6366215b15 2009-09-03T20:01:35+05:30 raghu $

(in-package "NET.RETROTEXTS.LIBXML-CLISP.EXAMPLES")



(defun storyinfo-keywords-xpath (filename)
  "Return the content of all `storyinfo/keyword' Nodes in FILENAME.
FILENAME is a pathname designator for an XML file.  The return value
is an arbitrarily ordered list of strings, each of which is the
content of a `keyword' Node."
  (with-xml-file (story filename)
    (if (story-node-p (document-element story))
      (get-storyinfo-keywords-xpath story)
      (restart-case (error 'story-error :document story)
        (parse-new-file (new-filename)
          :report "Parse another file."
          :interactive read-new-value
          (storyinfo-keywords-xpath new-filename))))))

(defun get-storyinfo-keywords-xpath (story)
  (with-xpath-context (story-context story)
    (if (null-item-p story-context)
      (error "Unable to create XPath Context.")
      (with-xpath-value
          (keyword-object "/story/storyinfo/keyword" story-context)
        (if (null-item-p keyword-object)
          (error "Unable to evaluate XPath expression.")
          (node-set-map #'(lambda (node)
                            (node-list-text story node))
                        (xpath-object-node-set keyword-object)))))))

(defun test-storyinfo-keywords-xpath (string)
  (with-temp-file (test-file)
    (with-open-file (out test-file :direction :output)
      (write-string string out))
    (storyinfo-keywords-xpath test-file)))

(defvar *storyinfo-keywords-xpath-test-data*
  "<?xml version=\"1.0\"?>
<story>
  <storyinfo>
    <author>Foo B. Quux</author>
    <datewritten>2009-05-17</datewritten>
    <keyword>storyinfo-1/keyword-1</keyword>
    <keyword>storyinfo-1/keyword-2</keyword>
  </storyinfo>
  <storyinfo>
    <keyword>storyinfo-2/keyword-1</keyword>
    <keyword>storyinfo-2/keyword-2</keyword>
  </storyinfo>
  <body>
    <headline>The Headline</headline>
    <keyword>body/keyword</keyword>
    <para>This is the body text.</para>
  </body>
</story>")

(defvar *storyinfo-keywords-xpath-test-value*
        '("storyinfo-1/keyword-1" "storyinfo-1/keyword-2"
          "storyinfo-2/keyword-1" "storyinfo-2/keyword-2"))

(defun test-keyword-xpath ()
  (flet ((set-equal (set-1 set-2)
           (and (subsetp set-1 set-2 :test #'equal)
                (subsetp set-2 set-1 :test #'equal))))
    (test-libxml-clisp #'test-storyinfo-keywords-xpath
                       *storyinfo-keywords-xpath-test-data*
                       *storyinfo-keywords-xpath-test-value*
                       :test #'set-equal)))

(provide-example 'keyword-xpath)



;;; Local Variables:
;;; mode: lisp
;;; comment-column: 32
;;; End:

;;;; keyword-xpath.lisp ends here