;;;; writer.lisp --- FFI definitions for libxml-clisp

;;; N. Raghavendra <raghu@retrotexts.net>
;;; 
;;; Created: 2009-09-27
;;; 
;;; $Hg: writer.lisp,v fa97b54d6d45 2009-09-28T18:22:09+05:30 raghu $

(in-package "NET.RETROTEXTS.LIBXML-CLISP")



(defparameter *text-writer* nil
  "The current Text Writer.")

(defmethod free-item ((text-writer text-writer))
  ($xml-free-text-writer (item-address text-writer)))

(defun make-text-writer (&optional address)
  "Return a Text Writer with address ADDRESS."
  (make-instance 'text-writer :address address))

(define-condition text-writer-error (error)
  ((text-writer :initarg :text-writer :reader error-text-writer))
  (:report (lambda (condition stream)
             (format stream "Writer ~A cannot process data."
                     (error-text-writer condition)))))

(defun text-writer-set-indent (text-writer indent)
  "Indent the output of WRITER if INDENT is true.
If INDENT is false, the output of WRITER is not indented."
  (let ((result ($xml-text-writer-set-indent (item-address text-writer)
                                             indent)))
    (if (minusp result)
      (error 'text-writer-error :text-writer text-writer)
      (zerop result))))

(defun new-text-writer (stream encoding indent)
  "Return a writer which sends its output to STREAM.
If ENCODING is non-nil, it is assumed to be the encoding of the output
of WRITER.  If ENCODING is nil, the encoding of the output of WRITER
is assumed to be UTF-8.  If INDENT is true, then the output is
indented."
  (let* ((*xml-encoding* (or encoding *xml-encoding*))
         (encoding-name (encoding-name encoding))
         (writer-address
           ($xml-new-text-writer
            ($xml-output-buffer-create-io
             *$xml-output-write-callback* nil nil
             ($xml-find-char-encoding-handler encoding-name)))))
    (if writer-address
      (let ((text-writer (make-text-writer writer-address)))
        (text-writer-set-indent text-writer (if indent 1 0))
        text-writer)
      (error "Unable to create Text Writer for stream ~A." stream))))

(defmacro with-text-writer ((&key encoding indent) &body body)
  "Evaluate BODY using a Text Writer.
During the evaluation, the special variable *TEXT-WRITER* is bound to
a Text Writer which sends its output to the default XML output stream.
That Text Writer has dynamic extent, which ends when the form is
exited.

If ENCODING is non-nil, it is assumed to be the encoding of the output
of WRITER.  If ENCODING is nil, the encoding of the output of WRITER
is assumed to be UTF-8.  If INDENT is true, then the output is
indented."
  (let ((created (gensym)))
    `(let* ((,created nil)
            (*text-writer* (new-text-writer *xml-output* ,encoding ,indent)))
       (unwind-protect (progn (setf ,created t)
                              (init-library)
                              ,@body)
         (when ,created
           (free-item *text-writer*)
           (cleanup-parser))))))

(defun text-writer-start-document (&key (version "1.0")
                                   (encoding *xml-encoding*)
                                   standalone)
  "Write the start of a new XML document with WRITER using ENCODING.
VERSION must be nil or an XML String designator representing \"1.0\".
If ENCODING is nil, then UTF-8 is used.  STANDALONE must be nil or an
XML String designator representing \"yes\" or \"no\".  Returns the
number of octets written (may be 0 because of buffering)."
  (let ((result ($xml-text-writer-start-document
                 (item-address *text-writer*)
                 version
                 (encoding-name encoding)
                 (and standalone "yes"))))
    (if (minusp result)
      (error 'text-writer-error :text-writer *text-writer*)
      result)))

(defun text-writer-start-element (name &optional namespace-prefix
                                       namespace-name)
  "Write the start of an element named NAME with WRITER.
Returns the number of octets written (may be 0 because of buffering)."
  (let ((result ($xml-text-writer-start-element-ns
                 (item-address *text-writer*)
                 (and namespace-prefix (item-address
                                        (xml-string namespace-prefix)))
                 (item-address (xml-string name))
                 (and namespace-name (item-address
                                      (xml-string namespace-name))))))
    (if (minusp result)
      (error 'text-writer-error :text-writer *text-writer*)
      result)))

(defun text-writer-end-element ()
  "Write the end of the current element with WRITER.
Returns the number of octets written (may be 0 because of buffering)."
  (let ((result ($xml-text-writer-end-element (item-address *text-writer*))))
    (if (minusp result)
      (error 'text-writer-error :text-writer *text-writer*)
      result)))

(defun text-writer-write-attribute (name content &optional namespace-prefix
                                         namespace-name)
  "Write an attribute named NAME with WRITER.
Returns the number of octets written (may be 0 because of buffering)."
  (let ((result ($xml-text-writer-write-attribute-ns
                 (item-address *text-writer*)
                 (and namespace-prefix (item-address
                                        (xml-string namespace-prefix)))
                 (item-address (xml-string name))
                 (and namespace-name (item-address
                                      (xml-string namespace-name)))
                 (item-address (xml-string content)))))
    (if (minusp result)
      (error 'text-writer-error :text-writer *text-writer*)
      result)))

(defun text-writer-write-comment (content)
  "Write a comment with text CONTENT using WRITER.
Returns the number of octets written (may be 0 because of buffering)."
  (let ((result ($xml-text-writer-write-comment
                 (item-address *text-writer*)
                 (item-address (xml-string content)))))
    (if (minusp result)
      (error 'text-writer-error :text-writer *text-writer*)
      result)))

(defun text-writer-end-document ()
     "Write the end of the current document with WRITER.
Returns the number of octets written (may be 0 because of buffering)."
  (let ((result ($xml-text-writer-end-document (item-address *text-writer*))))
    (if (minusp result)
      (error 'text-writer-error :text-writer *text-writer*)
      result)))



;;; Local Variables:
;;; mode: lisp
;;; comment-column: 32
;;; End:

;;;; writer.lisp ends here