;;;; xpath.lisp --- FFI definitions for libxml-clisp

;;; Copyright (C) 2009 N. Raghavendra.  All rights reserved.
;;; 
;;; Redistribution and use in source and binary forms, with or without
;;; modification, are permitted provided that the following conditions
;;; are met:
;;; 1. Redistributions of source code must retain the above copyright
;;;    notice, this list of conditions and the following disclaimer.
;;; 2. Redistributions in binary form must reproduce the above
;;;    copyright notice, this list of conditions and the following
;;;    disclaimer in the documentation and/or other materials provided
;;;    with the distribution.
;;; 
;;; THIS SOFTWARE IS PROVIDED BY THE AUTHOR "AS IS" AND ANY EXPRESS OR
;;; IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED
;;; WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
;;; ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR BE LIABLE FOR ANY
;;; DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
;;; DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE
;;; GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS
;;; INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY,
;;; WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
;;; NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
;;; SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

;;; N. Raghavendra <raghu@retrotexts.net>
;;; 
;;; Created: 2009-08-23
;;; 
;;; $Hg: xpath.lisp,v ae6366215b15 2009-09-03T20:01:35+05:30 raghu $

(in-package "NET.RETROTEXTS.LIBXML-CLISP")



;;;; A Node-set.

(defun make-node-set (&optional address)
  "Return a Node-set with address ADDRESS."
  (make-instance 'node-set :address address))

(defun node-set-number (node-set)
  "Return the number of nodes in NODE-SET."
  ($lxcl-xpath-node-set-get-length (item-address node-set)))

(defun node-set-element (node-set index)
  "Return the INDEX-th node in NODE-SET."
  (make-node ($lxcl-xpath-node-set-item (item-address node-set) index)))

(defun empty-node-set-p (node-set)
  "Return true if NODE-SET contains no nodes."
  (not (zerop ($lxcl-xpath-node-set-is-empty (item-address node-set)))))



;;;; An XPath Object.

(defun make-xpath-object (&optional address)
  "Return an XPath Object with address ADDRESS."
  (make-instance 'xpath-object :address address))

(defun xpath-object-type (xpath-object)
  (funcall-if-true #'(lambda (address)
                       (enum-from-value '$xml-xpath-object-type
                                        ($lxcl-xpath-object-type address)))
                   (item-address xpath-object)))

(defun xpath-object-node-set (xpath-object)
  (make-node-set (funcall-if-true #'$lxcl-xpath-object-nodesetval
                                  (item-address xpath-object))))

(defmethod free-item ((xpath-object xpath-object))
  ($xml-xpath-free-object (item-address xpath-object)))



;;;; An XPath Context.

(defun make-xpath-context (&optional address)
  "Return an XPath Context with address ADDRESS."
  (make-instance 'xpath-context :address address))

(defmethod free-item ((xpath-context xpath-context))
  ($xml-xpath-free-context (item-address xpath-context)))

(defun xpath-context (document)
  "Return the XPath Context created from DOCUMENT."
  (make-xpath-context ($xml-xpath-new-context (item-address document))))

(defmacro with-xpath-context ((xpath-context document) &body body)
  "Evaluate BODY using the XPath Context created from DOCUMENT.
During the evaluation, the variable XPATH-CONTEXT is bound to the
XPath Context created from DOCUMENT.  That XPath Context has dynamic
extent, which ends when the form is exited."
  (let ((created (gensym)))
    `(let* ((,created nil)
            (,xpath-context (xpath-context ,document)))
       (unwind-protect (progn (setf ,created t)
                              ,@body)
         (when ,created
           (free-item ,xpath-context))))))



;;;; XPath expressions.

(defun xpath-expression (expression)
  "Return the XPath expression described by EXPRESSION.
EXPRESSION must be an XPath expression designator, i.e., an XML String
designator."
  (xml-string expression))

(defun eval-xpath-expression (expression xpath-context)
  "Return the XPath Object obtained by evaluating EXPRESSION.
EXPRESSION must be an XPath expression designator.  The evaluation is
performed with respect to the XPath Context XPATH-CONTEXT."
  (let ((xpath-expression (xpath-expression expression)))
    (prog1 
      (make-xpath-object ($xml-xpath-eval-expression
                          (item-address xpath-expression)
                          (item-address xpath-context)))
      (free-item xpath-expression))))

(defmacro with-xpath-value ((xpath-object expression xpath-context) &body body)
  "Execute BODY using the XPath Object obtained by evaluating EXPRESSION.
EXPRESSION must be an XPath expression designator.  During the
execution, the variable XPATH-OBJECT is bound to the XPath Object
obtained by evaluating EXPRESSION in the XPath Context XPATH-CONTEXT.
That XPath Object has dynamic extent, which ends when the form is
exited."
  (let ((created (gensym)))
    `(let* ((,created nil)
            (,xpath-object (eval-xpath-expression ,expression ,xpath-context)))
       (unwind-protect (progn (setf ,created t)
                              ,@body)
         (when ,created
           (free-item ,xpath-object))))))

(defun node-set-map (function node-set &optional predicate)
  "Apply FUNCTION to elements of NODE-SET that satisfy PREDICATE.
FUNCTION must be a function which takes a Node as argument.  The
return value is an arbitrarily ordered list containing the results of
applying FUNCTION to the elements of NODE-SET that satisfy PREDICATE.
If PREDICATE is not supplied, then FUNCTION is applied to all the
elements of NODE-SET."
  (loop for i from 0 below (node-set-number node-set)
     for node = (node-set-element node-set i)
     when (or (null predicate) (funcall predicate node))
     collect (funcall function node)))



;;; Local Variables:
;;; mode: lisp
;;; comment-column: 32
;;; End:

;;;; xpath.lisp ends here