/* libunwind - a platform-independent unwind library
   Copyright (C) 2001-2002 Hewlett-Packard Co
	Contributed by David Mosberger-Tang <davidm@hpl.hp.com>

This library is free software; you can redistribute it and/or modify
it under the terms of the GNU Lesser General Public License as
published by the Free Software Foundation; either version 2.1 of the
License, or (at your option) any later version.

This library is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

As a special exception, if you link this library with other files to
produce an executable, this library does not by itself cause the
resulting executable to be covered by the GNU General Public License.
This exception does not however invalidate any other reasons why the
executable file might be covered by the GNU General Public
License.  */

#include <signal.h>

#include "rse.h"
#include "unwind_i.h"

int
unw_step (unw_cursor_t *cursor)
{
  struct ia64_cursor *c = (struct ia64_cursor *) cursor;
  unw_word_t prev_ip, prev_sp, prev_bsp, ip, pr, num_regs, cfm;
  STAT(unsigned long start, flags;)

  STAT(++unw.stat.api.unwinds;	start = ia64_get_itc ());

  prev_ip = c->ip;
  prev_sp = c->sp;
  prev_bsp = c->bsp;

  /* restore the ip */
  if (!c->rp_loc)
    {
      debug (1, __FUNCTION__": failed to locate return link (ip=0x%lx)!\n",
	     c->ip);
      STAT(unw.stat.api.unwind_time += ia64_get_itc () - start);
      return -1;
    }
  if (ia64_get (c, c->rp_loc, &ip) < 0)
    return -1;
  c->ip = ip;

  /* restore the cfm: */
  if (!c->pfs_loc)
    {
      dprintf (__FUNCTION__": failed to locate ar.pfs!\n");
      STAT(unw.stat.api.unwind_time += ia64_get_itc () - start);
      return -1;
    }
  c->cfm_loc = c->pfs_loc;

  /* restore the bsp: */
  pr = c->pr;
  num_regs = 0;
  if ((c->flags & IA64_FLAG_SIGTRAMP))
    {
      unw_word_t sigcontext_addr, sigcontext_flags;

      if (ia64_get (c, c->sp + 0x10, &sigcontext_addr) < 0)
	return -1;

      if (ia64_get (c, (sigcontext_addr
			+ struct_offset (struct sigcontext, sc_flags)),
		    &sigcontext_flags) < 0)
	return -1;

      if ((sigcontext_flags & IA64_SC_FLAG_IN_SYSCALL_BIT) == 0)
	{
	  unw_word_t cfm;

	  if (ia64_get (c, c->cfm_loc, &cfm) < 0)
	    return -1;

	  num_regs = cfm & 0x7f;	/* size of frame */
	}
      c->pfs_loc = (c->sp + 0x10 + struct_offset (struct sigcontext,
						  sc_ar_pfs));
    }
  else
    {
      if (ia64_get (c, c->cfm_loc, &cfm) < 0)
	return -1;
      num_regs = (cfm >> 7) & 0x7f;	/* size of locals */
    }
  c->bsp = (unsigned long) ia64_rse_skip_regs ((unsigned long *) c->bsp,
					       -num_regs);

  /* restore the sp: */
  c->sp = c->psp;

  if (c->ip == prev_ip && c->sp == prev_sp && c->bsp == prev_bsp)
    {
      dprintf (__FUNCTION__": ip, sp, bsp remain unchanged; stopping here "
	       "(ip=0x%lx)\n", ip);
      STAT(unw.stat.api.unwind_time += ia64_get_itc () - start);
      return -1;
    }

  /* as we unwind, the saved ar.unat becomes the primary unat: */
  c->pri_unat_loc = c->unat_loc;

  /* finally, restore the predicates: */

  if (ia64_get (c, c->pr_loc, &c->pr) < 0)
    goto fail;

  if (__ia64_unw_find_save_locs (c) < 0)
    goto fail;

  STAT(unw.stat.api.unwind_time += ia64_get_itc () - start);

  return (c->ip == 0) ? 0 : 1;

 fail:
  STAT(unw.stat.api.unwind_time += ia64_get_itc () - start);
  return -1;
}
