/* libunwind - a platform-independent unwind library
   Copyright (C) 2001-2002 Hewlett-Packard Co
	Contributed by David Mosberger-Tang <davidm@hpl.hp.com>

This library is free software; you can redistribute it and/or modify
it under the terms of the GNU Lesser General Public License as
published by the Free Software Foundation; either version 2.1 of the
License, or (at your option) any later version.

This library is distributed in the hope that it will be useful, but
WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
Lesser General Public License for more details.

You should have received a copy of the GNU Lesser General Public
License along with this library; if not, write to the Free Software
Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA

As a special exception, if you link this library with other files to
produce an executable, this library does not by itself cause the
resulting executable to be covered by the GNU General Public License.
This exception does not however invalidate any other reasons why the
executable file might be covered by the GNU General Public
License.  */

#ifndef UNWIND_H
#define UNWIND_H

#include <ucontext.h>

/* This needs to be big enough to accommodate the unwind state of any
   architecture.  */
#define UNW_STATE_LEN	127

typedef enum
  {
    UNW_ESUCCESS = 0,		/* no error */
    UNW_EUNSPEC,		/* unspecified (general) error */
    UNW_ENOMEM,			/* out of memory */
    UNW_EBADREG,		/* bad register number */
    UNW_EREADONLYREG,		/* attempt to write read-only register */
  }
unw_error_t;

/* The following are pseudo-registers which have the same meaning
   across all platforms.  (RO) means read-only.  */
typedef enum
  {
    UNW_REG_IP = -1,		/* (ro) instruction pointer (pc) */
    UNW_REG_RP = -2,		/* (rw) return pointer */
    UNW_REG_SP = -3,		/* (ro) stack pointer */
    UNW_REG_FP = -4,		/* (ro) frame pointer */
    UNW_REG_FIRST_IP = -5,	/* (ro) starting addr. of procedure */
    UNW_REG_HANDLER = -6,	/* (ro) addr. of "personality routine" */
    UNW_REG_LSDA = -7		/* (ro) addr. of language-specific data area */
  }
unw_frame_regnum_t;

typedef int unw_regnum_t;

typedef unsigned long unw_word_t;

typedef struct unw_cursor
  {
    unw_word_t opaque[UNW_STATE_LEN];
  }
unw_cursor_t;

/* We will assume that "long double" is sufficiently large and aligned
   to hold the contents of a floating-point register.  Note that the
   fp register format is not usually the same format as a "long
   double".  Instead, the content of unw_fpreg_t should be manipulated
   only through the "raw.bits" member. */
typedef union
  {
    struct { unsigned long bits[1]; } raw;
    long double dummy;	/* dummy to force 16-byte alignment */
  }
unw_fpreg_t;

typedef struct unw_accessors
  {
    /* Lock for unwind info for address IP.  The architecture specific
       UNWIND_INFO is updated as necessary.  */
    int (*acquire_unwind_info) (unw_word_t ip, void *unwind_info);
    int (*release_unwind_info) (void *unwind_info);

    /* Access aligned word at address ADDR.  */
    int (*access_mem) (unw_word_t addr, unw_word_t *val, int write, void *arg);

    /* Access register number REG at address ADDR.  */
    int (*access_reg) (unw_regnum_t reg, unw_word_t *val, int write,
		       void *arg);

    /* Access register number REG at address ADDR.  */
    int (*access_fpreg) (unw_regnum_t reg, unw_fpreg_t *val, int write,
			 void *arg);

    int (*resume) (unw_cursor_t *c, void *arg);

    void *arg;		/* application-specific data */
  }
unw_accessors_t;

/* Initialize context such that unwinding starts at the point from
   which this function is called.  Returns zero on success, negative
   value on failure.  */
extern int unw_init_local (unw_cursor_t *c, ucontext_t *u);

/* Establish accessor routines.  Without this call, the unwinder will
   access the process memory directory.  Returns zero on success,
   negative value on error.

   Note: This function may not be supported by all implementations of
   this interface.  */
extern int unw_init_remote (unw_cursor_t *c, unw_accessors_t *a);

/* Move cursor by one step (toward earlier, less deeply nested
   frames).  Returns positive number if there are more frames to
   unwind, 0 if last frame has been reached, negative number in case
   of an error.  */
extern int unw_step (unw_cursor_t *c);

/* Resume execution at the point identified by the cursor.  */
extern int unw_resume (unw_cursor_t *c);

/* Register accessor routines.  Return zero on success, negative value
   on failure.  */
extern int unw_get_reg (unw_cursor_t *c, int regnum, unw_word_t *valp);
extern int unw_set_reg (unw_cursor_t *c, int regnum, unw_word_t val);

/* Floating-point accessor routines.  Return zero on success, negative
   value on failure.  */
extern int unw_get_fpreg (unw_cursor_t *c, int regnum, unw_fpreg_t *val);
extern int unw_set_fpreg (unw_cursor_t *c, int regnum, unw_fpreg_t val);

/* Returns non-zero value if frame is a signal frame.  */
extern int unw_is_signal_frame (unw_cursor_t *c);

#endif /* UNWIND_H */
