#include <glib.h>
#include <stdio.h>
#include <stdlib.h>

#include "../parse.h"
#include "../scheme.h"

typedef struct _parse_fixture {
  ruin_window_t *window;
  SCM doc;
} parse_fixture;

typedef struct _parse_fixture_component {
  struct _parse_fixture_component *parent;
  GHashTable *attributes;
  char *tag_name;
  char *content;
} parse_fixture_component;

static parse_fixture_component *parse_fixture_component_new 
(char *tag_name, GHashTable *attributes, char *content, 
 parse_fixture_component *parent)
{
  parse_fixture_component *c = calloc (1, sizeof (parse_fixture_component));

  c->attributes = attributes;
  c->tag_name = tag_name;
  c->content = content;
  c->parent = parent;

  return c;
}

static void parse_fixture_component_free (parse_fixture_component *c)
{
  free (c);
}

static void set_attribute (gpointer key, gpointer value, gpointer user_data)
{
  ruin_scheme_sdom_set_attribute ((ruin_window_t *) ((void **) user_data)[0], 
				  (SCM) ((void **) user_data)[1], 
				  (char *) key, (char *) value);
}

void setup_parse_fixture (parse_fixture *f, gconstpointer d)
{
  GList *components = (GList *) d;
  GList *component_ptr = components;
  GHashTable *component_nodes = g_hash_table_new 
    (g_direct_hash, g_direct_equal);

  f->window = ruin_window_new 
    (newwin (10, 10, 0, 0), fopen ("/dev/null", "w+"));
  setup_window_fixture (f->window, d);

  f->doc = SCM_BOOL_F;
  
  while (component_ptr != NULL)
    {
      SCM node = SCM_EOL;
      parse_fixture_component *c = (parse_fixture_component *) 
	component_ptr->data;

      if (f->doc == SCM_BOOL_F)
	{
	  f->doc = ruin_scheme_sdom_create_document 
	    (f->window, c->tag_name, SCM_BOOL_F);
	  scm_gc_protect_object (f->doc);

	  node = ruin_scheme_sdom_document_element (f->window, f->doc);
	}
      else if (c->tag_name != NULL)
	{
	  node = ruin_scheme_sdom_create_element 
	    (f->window, f->doc, c->tag_name);
	  if (c->attributes != NULL)
	    {
	      void *user_data[] = { f->window, node };
	      g_hash_table_foreach 
		(c->attributes, set_attribute, user_data);
	    }
	}
      else node = ruin_scheme_sdom_create_text_node 
	     (f->window, f->doc, c->content);

      g_hash_table_insert (component_nodes, c, node);
      if (c->parent != NULL)
	{
	  SCM parent_node = g_hash_table_lookup (component_nodes, c->parent);
	  ruin_scheme_sdom_append_child (f->window, parent_node, node);
	}

      component_ptr = component_ptr->next;
    }

  g_hash_table_destroy (component_nodes);
}

void teardown_parse_fixture (parse_fixture *f, gconstpointer d)
{
}

void test_parse_simple_element (parse_fixture *f, gconstpointer d)
{
  ruin_node_t *elt = ruin_parse_document (f->window, f->doc);
  
  g_assert (elt != NULL);
  g_assert (elt->parent == NULL);
  g_assert (elt->first_child == NULL);
  g_assert (elt->next_sibling == NULL);
}

void test_parse_simple_element_attributes (parse_fixture *f, gconstpointer d)
{
  ruin_node_t *elt = ruin_parse_document (f->window, f->doc);

  g_assert (elt != NULL);
  g_assert (elt->parent == NULL);
  g_assert (elt->first_child == NULL);
  g_assert (elt->next_sibling == NULL);
}

void test_parse_simple_element_content (parse_fixture *f, gconstpointer d)
{
  ruin_node_t *elt = ruin_parse_document (f->window, f->doc);
  ruin_node_text_t *text_node = NULL;

  g_assert (elt != NULL);
  g_assert (elt->parent == NULL);
  g_assert (elt->next_sibling == NULL);

  g_assert (elt->first_child != NULL);
  text_node = (ruin_node_text_t *) elt->first_child;

  g_assert_cmpstr ("content", ==, text_node->content);
}

void test_parse_complex_element_children (parse_fixture *f, gconstpointer d)
{
  ruin_node_t *elt = ruin_parse_document (f->window, f->doc);
  ruin_node_t *child1 = NULL;
  ruin_node_t *child2 = NULL;

  g_assert (elt != NULL);
  g_assert (elt->parent == NULL);
  g_assert (elt->first_child != NULL);
  g_assert (elt->next_sibling == NULL);

  child1 = elt->first_child;
  g_assert (child1->parent == elt);
  g_assert (child1->first_child == NULL);
  g_assert (child1->next_sibling != NULL);

  child2 = child1->next_sibling;
  g_assert (child2->parent == elt);
  g_assert (child2->first_child == NULL);
  g_assert (child2->next_sibling == NULL);
  
  g_assert (child1->next_sibling == child2);
}

void test_parse_complex_mixed_children (parse_fixture *f, gconstpointer d)
{
  ruin_node_t *elt = ruin_parse_document (f->window, f->doc);
  ruin_node_t *child1 = NULL;
  ruin_node_t *child2 = NULL;
  ruin_node_text_t *child1_text = NULL;

  g_assert (elt != NULL);
  g_assert (elt->parent == NULL);
  g_assert (elt->first_child != NULL);
  g_assert (elt->next_sibling == NULL);

  child1 = elt->first_child;
  child1_text = (ruin_node_text_t *) child1;

  g_assert (child1->parent == elt);
  g_assert (child1->first_child == NULL);
  g_assert_cmpstr (child1_text->content, ==, "content");
  g_assert (child1->next_sibling != NULL);

  child2 = child1->next_sibling;
  g_assert (child2->parent == elt);
  g_assert (child2->first_child == NULL);
  g_assert (child2->next_sibling == NULL);
  
  g_assert (child1->next_sibling == child2);
}

int main (int argc, char *argv[])
{
  int ret = 0;
  FILE *dev_null = fopen ("/dev/null", "w+");
  
  g_test_init (&argc, &argv, NULL);

  newterm (NULL, dev_null, stdin);
  scm_init_guile ();
  ruin_init ();

  { GList *elements = g_list_append 
      (NULL, parse_fixture_component_new ("element", NULL, NULL, NULL));
    g_test_add ("/parse/simple/element", parse_fixture, elements, 
		setup_parse_fixture, test_parse_simple_element, 
		teardown_parse_fixture);
  };

  { GHashTable *attrs = g_hash_table_new (g_str_hash, g_str_equal);
    GList *elements = g_list_append
      (NULL, parse_fixture_component_new ("element", attrs, NULL, NULL));
    g_hash_table_insert (attrs, "foo", "bar");
    g_test_add ("/parse/simple/element/attributes", parse_fixture, elements,
		setup_parse_fixture, test_parse_simple_element_attributes, 
		teardown_parse_fixture);
  };

  { parse_fixture_component *elt = 
      parse_fixture_component_new ("element", NULL, NULL, NULL);
    GList *text = g_list_append
      (g_list_append (NULL, elt),
       parse_fixture_component_new (NULL, NULL, "content", elt));

    g_test_add ("/parse/simple/element/content", parse_fixture, text, 
		setup_parse_fixture, test_parse_simple_element_content, 
		teardown_parse_fixture);
  };

  { parse_fixture_component *parent =
      parse_fixture_component_new ("parent", NULL, NULL, NULL);
    GList *elements = g_list_append 
      (g_list_append 
       (g_list_append (NULL, parent), 
	parse_fixture_component_new ("element", NULL, NULL, parent)), 
       parse_fixture_component_new ("element", NULL, NULL, parent));
    
    g_test_add ("/parse/complex/element/children", parse_fixture, elements,
		setup_parse_fixture, test_parse_complex_element_children,
		teardown_parse_fixture);
  };

  { parse_fixture_component *parent = 
      parse_fixture_component_new ("parent", NULL, NULL, NULL);
    GList *elements = g_list_append 
      (g_list_append 
       (g_list_append (NULL, parent), 
	parse_fixture_component_new (NULL, NULL, "content", parent)), 
       parse_fixture_component_new ("element", NULL, NULL, parent));

    g_test_add ("/parse/complex/mixed/children", parse_fixture, elements,
		setup_parse_fixture, test_parse_complex_mixed_children,
		teardown_parse_fixture);    
  };

  ret = g_test_run ();

  endwin ();
  fclose (dev_null);

  return ret;
}
