#include <glib.h>
#include <ncurses.h>
#include <stdio.h>
#include <stdlib.h>

#include "../box.h"
#include "../scheme.h"
#include "../window.h"
#include "util.h"

void test_layout_containingblock_initial (box_fixture *f, gconstpointer d)
{
  ruin_layout_layout_elements (g_list_append (NULL, f->root_box));

  g_assert_cmpint (20, ==, f->root_box->containing_block->width.used);
  g_assert_cmpint (10, ==, f->root_box->containing_block->height.used);
}

void test_layout_simple_block (box_fixture *f, gconstpointer d)
{
  ruin_layout_layout_elements (g_list_append (NULL, f->root_box));

  g_assert_cmpint (0, ==, f->root_box->height.used);
  g_assert_cmpint (20, ==, f->root_box->width.used);
}

void test_layout_simple_line (box_fixture *f, gconstpointer d)
{
  ruin_box_t *child_box = NULL;
  ruin_inline_box_t *inline_child_box = NULL;

  ruin_layout_layout_elements (g_list_append (NULL, f->root_box));
  
  g_assert_cmpint (1, ==, f->root_box->height.used);
  g_assert_cmpint (20, ==, f->root_box->width.used);

  g_assert (f->root_box->type == RUIN_LAYOUT_BOX_TYPE_LINE);
  g_assert_cmpint (1, ==, g_list_length (f->root_box->children));

  child_box = (ruin_box_t *) f->root_box->children->data;
  g_assert (child_box->type == RUIN_LAYOUT_BOX_TYPE_INLINE);
  g_assert_cmpint (4, ==, child_box->width.used);
  inline_child_box = (ruin_inline_box_t *) child_box;
}

void test_layout_style_letter_spacing (box_fixture *f, gconstpointer d)
{
  ruin_box_t *child_box = NULL;

  ruin_layout_layout_elements (g_list_append (NULL, f->root_box));

  g_assert (f->root_box->type == RUIN_LAYOUT_BOX_TYPE_LINE);
  g_assert_cmpint (1, ==, g_list_length (f->root_box->children));

  child_box = (ruin_box_t *) f->root_box->children->data;
  g_assert (child_box->type == RUIN_LAYOUT_BOX_TYPE_INLINE);
  g_assert_cmpint (7, ==, child_box->width.used);  
}

void test_layout_style_word_spacing (box_fixture *f, gconstpointer d)
{
  ruin_box_t *child_box = NULL;

  ruin_layout_layout_elements (g_list_append (NULL, f->root_box));

  g_assert (f->root_box->type == RUIN_LAYOUT_BOX_TYPE_LINE);
  g_assert_cmpint (1, ==, g_list_length (f->root_box->children));

  child_box = (ruin_box_t *) f->root_box->children->data;
  g_assert (child_box->type == RUIN_LAYOUT_BOX_TYPE_INLINE);
  g_assert_cmpint (10, ==, child_box->width.used);
}

void test_layout_style_white_space_normal (box_fixture *f, gconstpointer d)
{
  ruin_box_t *child_box = NULL;

  ruin_layout_layout_elements (g_list_append (NULL, f->root_box));

  g_assert (f->root_box->type == RUIN_LAYOUT_BOX_TYPE_LINE);
  g_assert_cmpint (1, ==, g_list_length (f->root_box->children));

  child_box = (ruin_box_t *) f->root_box->children->data;
  g_assert (child_box->type == RUIN_LAYOUT_BOX_TYPE_INLINE);
  g_assert_cmpint (9, ==, child_box->width.used);
}

void test_layout_style_white_space_pre (box_fixture *f, gconstpointer d)
{
  ruin_box_t *child_box = NULL;

  ruin_layout_layout_elements (g_list_append (NULL, f->root_box));

  g_assert (f->root_box->type == RUIN_LAYOUT_BOX_TYPE_LINE);
  g_assert_cmpint (1, ==, g_list_length (f->root_box->children));

  child_box = (ruin_box_t *) f->root_box->children->data;
  g_assert (child_box->type == RUIN_LAYOUT_BOX_TYPE_INLINE);
  g_assert_cmpint (17, ==, child_box->width.used);
}

int main (int argc, char *argv[])
{
  int ret = 0;
  FILE *dev_null = fopen ("/dev/null", "w+");

  GHashTable *simple_block = NULL;
  GHashTable *simple_inline = NULL;
  GHashTable *simple_inline_letter_spacing = NULL;
  GHashTable *simple_inline_word_spacing = NULL;
  GHashTable *simple_inline_white_space_normal = NULL;
  GHashTable *simple_inline_white_space_pre = NULL;

  box_layout_fixture_component_t simple_line_config;
  box_layout_fixture_component_t simple_inline_config;
  box_layout_fixture_component_t simple_inline_letter_spacing_config;
  box_layout_fixture_component_t simple_inline_word_spacing_config;
  box_layout_fixture_component_t simple_inline_white_space_normal_config;
  box_layout_fixture_component_t simple_inline_white_space_pre_config;
  
  g_test_init (&argc, &argv, NULL);

  newterm (NULL, dev_null, stdin);
  scm_init_guile ();
  ruin_init ();

  simple_block = g_hash_table_new (g_str_hash, g_str_equal);
  g_hash_table_insert (simple_block, "display", "block");
  
  simple_inline = g_hash_table_new (g_str_hash, g_str_equal);
  g_hash_table_insert (simple_inline, "display", "inline");

  simple_inline_letter_spacing = g_hash_table_new (g_str_hash, g_str_equal);
  g_hash_table_insert (simple_inline_letter_spacing, "display", "inline");
  g_hash_table_insert (simple_inline_letter_spacing, "letter-spacing", "1em");

  simple_inline_word_spacing = g_hash_table_new (g_str_hash, g_str_equal);
  g_hash_table_insert (simple_inline_word_spacing, "display", "inline");
  g_hash_table_insert (simple_inline_word_spacing, "word-spacing", "1em");
  
  simple_inline_white_space_pre = g_hash_table_new (g_str_hash, g_str_equal);
  g_hash_table_insert (simple_inline_white_space_pre, "display", "inline");
  g_hash_table_insert (simple_inline_white_space_pre, "white-space", "pre");  

  g_test_add ("/box/layout/containingblock/initial", box_fixture, 
	      box_layout_fixture_component_new (simple_block, NULL, NULL),
	      setup_box_layout_fixture_single, 
	      test_layout_containingblock_initial, teardown_box_fixture);

  g_test_add ("/box/layout/simple/block", box_fixture, 
	      box_layout_fixture_component_new (simple_block, NULL, NULL), 
	      setup_box_layout_fixture_single, test_layout_simple_block,
	      teardown_box_fixture);

  simple_line_config.style = g_hash_table_new (g_str_hash, g_str_equal);
  simple_line_config.type = RUIN_LAYOUT_BOX_TYPE_LINE;
  simple_line_config.content_ptr = NULL;
  simple_line_config.parent = NULL;
  simple_line_config.containing_block = NULL;

  simple_inline_config.style = simple_inline;
  simple_inline_config.type = RUIN_LAYOUT_BOX_TYPE_INLINE;
  simple_inline_config.content_ptr = "Test";
  simple_inline_config.parent = &simple_line_config;
  simple_inline_config.containing_block = NULL;
  
  g_test_add ("/box/layout/simple/line", box_fixture, 
	      g_list_append (g_list_append (NULL, &simple_line_config), 
			     &simple_inline_config),
	      setup_box_layout_fixture, test_layout_simple_line, 
	      teardown_box_fixture);

  simple_inline_letter_spacing_config.style = simple_inline_letter_spacing;
  simple_inline_letter_spacing_config.type = RUIN_LAYOUT_BOX_TYPE_INLINE;
  simple_inline_letter_spacing_config.content_ptr = "Test";
  simple_inline_letter_spacing_config.parent = &simple_line_config;
  simple_inline_letter_spacing_config.containing_block = NULL;

  g_test_add ("/box/layout/style/letter-spacing", box_fixture,
	      g_list_append (g_list_append (NULL, &simple_line_config),
			     &simple_inline_letter_spacing_config), 
	      setup_box_layout_fixture, test_layout_style_letter_spacing,
	      teardown_box_fixture);

  simple_inline_word_spacing_config.style = simple_inline_word_spacing;
  simple_inline_word_spacing_config.type = RUIN_LAYOUT_BOX_TYPE_INLINE;
  simple_inline_word_spacing_config.content_ptr = "Test Test";
  simple_inline_word_spacing_config.parent = &simple_line_config;
  simple_inline_word_spacing_config.containing_block = NULL;

  g_test_add ("/box/layout/style/word-spacing", box_fixture,
	      g_list_append (g_list_append (NULL, &simple_line_config),
			     &simple_inline_word_spacing_config), 
	      setup_box_layout_fixture, test_layout_style_word_spacing, 
	      teardown_box_fixture);

  simple_inline_white_space_normal_config.style = simple_inline;
  simple_inline_white_space_normal_config.type = RUIN_LAYOUT_BOX_TYPE_INLINE;
  simple_inline_white_space_normal_config.content_ptr = "   Test   Test   ";
  simple_inline_white_space_normal_config.parent = &simple_line_config;
  simple_inline_white_space_normal_config.containing_block = NULL;  

  g_test_add ("/box/layout/style/white-space/normal", box_fixture,
	      g_list_append (g_list_append (NULL, &simple_line_config),
			     &simple_inline_white_space_normal_config), 
	      setup_box_layout_fixture, test_layout_style_white_space_normal, 
	      teardown_box_fixture);

  simple_inline_white_space_pre_config.style = simple_inline_white_space_pre;
  simple_inline_white_space_pre_config.type = RUIN_LAYOUT_BOX_TYPE_INLINE;
  simple_inline_white_space_pre_config.content_ptr = "   Test   Test   ";
  simple_inline_white_space_pre_config.parent = &simple_line_config;
  simple_inline_white_space_pre_config.containing_block = NULL;

  g_test_add ("/box/layout/style/white-space/pre", box_fixture,
	      g_list_append (g_list_append (NULL, &simple_line_config),
			     &simple_inline_white_space_pre_config), 
	      setup_box_layout_fixture, test_layout_style_white_space_pre, 
	      teardown_box_fixture);

  ret = g_test_run ();

  endwin ();
  fclose (dev_null);

  return ret;
}
