/* tables.c: Table layout routines for libRUIN
 * Copyright (C) 2011 Julian Graham
 *
 * libRUIN is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <assert.h>
#include <glib.h>
#include <stdlib.h>

#include "box.h"
#include "css.h"
#include "tables.h"
#include "text.h"
#include "util.h"

typedef struct _min_max_width
{
  int min_width;
  int max_width;
} min_max_width;

static min_max_width *get_min_max_width (ruin_box_t *);

static ruin_layout_table_column *ruin_layout_table_column_new (ruin_box_t *b)
{
  ruin_layout_table_column *column = calloc 
    (1, sizeof (ruin_layout_table_column));

  column->node = b;
  column->width = RUIN_CSS_VALUE_NONE;
  
  return column;
}

static ruin_layout_table_cell *ruin_layout_table_cell_new (ruin_box_t *b)
{
  ruin_layout_table_cell *cell = calloc (1, sizeof (ruin_layout_table_cell));

  cell->node = b;

  return cell;
}

static gpointer get_analysis_text (gpointer box)
{
  ruin_layout_text_analysis *analysis = NULL;
  ruin_layout_text_line_width_constraint constraint = { INT_MAX, INT_MAX };
  ruin_layout_text_context context = { FALSE, TRUE, TRUE, TRUE, NULL };
  context.width_constraints = g_list_append (NULL, &constraint);

  analysis = ruin_layout_text_analyze ((ruin_inline_box_t *) box, &context);
  g_list_free (context.width_constraints);

  return analysis;
}

static gpointer get_analysis_inline (gpointer i)
{
  ruin_inline_box_t *ib = (ruin_inline_box_t *) i;
  if (ib->content_ptr == NULL)
    {
      ruin_box_t *box = (ruin_box_t *) ib;      
      ruin_layout_text_analysis *analysis = 
	calloc (1, sizeof (ruin_layout_text_analysis));
      GList *analyses = ruin_util_map (box->children, get_analysis_inline);
      GList *analysis_ptr = analyses;

      while (analysis_ptr != NULL)
	{
	  ruin_layout_text_analysis *child_analysis = 
	    (ruin_layout_text_analysis *) analysis_ptr->data;
	  analysis->fragments = g_list_concat 
	    (analysis->fragments, child_analysis->fragments);
	  ruin_layout_text_analysis_free (child_analysis, FALSE);

	  analysis_ptr = analysis_ptr->next;
	}

      g_list_free (analyses);
      return analysis;
    }
  else return get_analysis_text (ib);
}

static min_max_width *get_min_max_width_marker (ruin_marker_box_t *box)
{
  assert (1 == 0);
  return NULL;
}

static min_max_width *get_min_max_width_line (ruin_box_t *box)
{
  min_max_width *min_max = calloc (1, sizeof (min_max_width));
  
  GList *analyses = ruin_util_map (box->children, get_analysis_inline);
  GList *analysis_ptr = analyses;
  int running_max_width = 0;

  while (analysis_ptr != NULL)
    {
      ruin_layout_text_analysis *analysis = (ruin_layout_text_analysis *) 
	analysis_ptr->data;
      GList *fragment_ptr = analysis->fragments;      
      while (fragment_ptr != NULL)
	{
	  ruin_layout_text_fragment *fragment = (ruin_layout_text_fragment *)
	    fragment_ptr->data;

	  running_max_width += fragment->real_width;
	  if (fragment->real_width > min_max->min_width)
	    min_max->min_width = fragment->real_width;
	  if (fragment->break_required)
	    {
	      if (running_max_width > min_max->max_width)
		min_max->max_width = running_max_width;
	      running_max_width = 0;
	    }

	  fragment_ptr = fragment_ptr->next;
	}
      
      ruin_layout_text_analysis_free (analysis, FALSE);
      analysis_ptr = analysis_ptr->next;
    }
  if (running_max_width > min_max->max_width)
    min_max->max_width = running_max_width;

  g_list_free (analyses);

  return min_max;
}

static min_max_width *get_min_max_width_block_children (ruin_box_t *box)
{
  GList *child_ptr = box->children;
  min_max_width *min_max = calloc (1, sizeof (min_max_width));
  while (child_ptr != NULL)
    {
      min_max_width *child_min_max = NULL;
      ruin_box_t *child_box = (ruin_box_t *) child_ptr->data;
      
      if (child_box->type == RUIN_LAYOUT_BOX_TYPE_MARKER)
	assert (1 == 0);
      else if (child_box->type == RUIN_LAYOUT_BOX_TYPE_LINE)
	child_min_max = get_min_max_width_line (child_box);
      else child_min_max = get_min_max_width (child_box);

      min_max->min_width = MAX (min_max->min_width, child_min_max->min_width);
      min_max->max_width = MAX (min_max->max_width, child_min_max->max_width);
      free (child_min_max);
  
      child_ptr = child_ptr->next;
    }
  
  return min_max;
}

static min_max_width *get_min_max_width_block (ruin_box_t *box)
{
  min_max_width *min_max = get_min_max_width_block_children (box);

  ruin_length_t blw = ruin_css_parse_size 
    (ruin_box_css_lookup (box, "border-left-width"));
  ruin_length_t brw = ruin_css_parse_size
    (ruin_box_css_lookup (box, "border-right-width"));

  ruin_css_normalize_length 
    (&blw, &box->containing_block->width, box->window->font_width, 
     box->window->dpi, TRUE);  
  ruin_css_normalize_length 
    (&brw, &box->containing_block->width, box->window->font_width, 
     box->window->dpi, TRUE);  

  min_max->min_width += blw.used + brw.used;
  min_max->max_width += blw.used + brw.used;

  return min_max;
}

static min_max_width *get_min_max_width (ruin_box_t *box)
{
  switch (box->type)
    {
    case RUIN_LAYOUT_BOX_TYPE_ANONYMOUS_BLOCK:
    case RUIN_LAYOUT_BOX_TYPE_BLOCK:
      return get_min_max_width_block (box);
    case RUIN_LAYOUT_BOX_TYPE_LINE:
      assert (1 == 0);
    case RUIN_LAYOUT_BOX_TYPE_MARKER:
      return get_min_max_width_marker ((ruin_marker_box_t *) box);
    default: assert (1 == 0);
    }

  return NULL;
}

static ruin_layout_table_row *ruin_layout_table_row_new (ruin_box_t *box)
{
  ruin_layout_table_row *row = calloc (1, sizeof (ruin_layout_table_row));

  row->node = box;

  return row;
}

static gpointer generate_table_row (gpointer i, gpointer d)
{
  ruin_box_t *box = (ruin_box_t *) i;
  ruin_layout_table *table = (ruin_layout_table *) d;
  ruin_layout_table_row *row = ruin_layout_table_row_new (box);
  GList *column_ptr = table->columns;
  GList *child_ptr = box->children;
  GList *new_columns = NULL;
  
  while (child_ptr != NULL)
    {
      ruin_box_t *child = (ruin_box_t *) child_ptr->data;
      ruin_layout_table_cell *cell = ruin_layout_table_cell_new (child);
      row->cells = g_list_append (row->cells, cell);
      
      if (column_ptr == NULL)
	{
	  ruin_layout_table_column *column = 
	    ruin_layout_table_column_new (NULL);
	  column->cells = g_list_append (NULL, cell);
	  new_columns = g_list_append (new_columns, column);
	}
      else 
	{
	  ruin_layout_table_column *column = 
	    (ruin_layout_table_column *) column_ptr->data;
	  column->cells = g_list_append (column->cells, cell);
	  column_ptr = column_ptr->next;
	}

      child_ptr = child_ptr->next;
    }

  table->columns = g_list_concat (table->columns, new_columns);
  table->rows = g_list_append (table->rows, row);

  return row;
}

static ruin_layout_table *parse_table (ruin_box_t *box)
{
  ruin_layout_table *table = calloc (1, sizeof (ruin_layout_table));
  GList *child_ptr = box->children;
  int explicit_columns = 0;

  table->node = box;

  while (child_ptr != NULL)
    {
      ruin_box_t *child = (ruin_box_t *) child_ptr->data;
      char *display = ruin_box_css_lookup (child, "display");
      
      if (strcmp (display, "table-caption") == 0)
	table->caption = child;
      else if (strcmp (display, "table-column") == 0)
	{
	  ruin_layout_table_column *column = (ruin_layout_table_column *)
	    g_list_nth_data (table->columns, explicit_columns++);
	  if (column != NULL)
	    column->node = child;
	  else table->columns = g_list_append 
		 (table->columns, ruin_layout_table_column_new (child));
	}
      else if (strcmp (display, "table-column-group") == 0)
	{
	  GList *column_ptr = child->children;
	  while (column_ptr != NULL)
	    {
	      ruin_box_t *column_child = (ruin_box_t *) column_ptr->data;
	      ruin_layout_table_column *column = (ruin_layout_table_column *)
		g_list_nth_data (table->columns, explicit_columns++);
	      if (column != NULL)
		column->node = column_child;
	      else table->columns = g_list_append 
		     (table->columns, ruin_layout_table_column_new 
		      (column_child));
	    }	  
	}	    
      else if (strcmp (display, "table-header-group") == 0
	       || strcmp (display, "table-footer-group") == 0
	       || strcmp (display, "table-row-group") == 0)
	table->rows = g_list_concat 
	  (table->rows, ruin_util_map_context
	   (child->children, generate_table_row, table));
      else if (strcmp (display, "table-row") == 0)
	table->rows = g_list_append 
	  (table->rows, generate_table_row (child, table));

      child_ptr = child_ptr->next;
    }

  return table;
}

ruin_layout_table *ruin_layout_table_auto (ruin_box_t *box)
{
  int min_columns_width = 0;
  int max_columns_width = 0;

  ruin_layout_table *table = parse_table (box);
  ruin_length_t l = ruin_css_parse_size (ruin_box_css_lookup (box, "width"));
  GList *column_ptr = table->columns;
  GList *column_min_max_widths = NULL;

  while (column_ptr != NULL)
    {
      min_max_width *min_max = calloc (1, sizeof (min_max_width));
      ruin_layout_table_column *column =
	(ruin_layout_table_column *) column_ptr->data;
      GList *cell_ptr = column->cells;
      while (cell_ptr != NULL)
	{
	  ruin_layout_table_cell *cell = 
	    (ruin_layout_table_cell *) cell_ptr->data;
	  min_max_width *cell_min_max = get_min_max_width (cell->node);

	  if (cell_min_max->min_width > min_max->min_width)
	    min_max->min_width = cell_min_max->min_width;
	  if (cell_min_max->max_width > min_max->max_width)
	    min_max->max_width = cell_min_max->max_width;
	  
	  free (cell_min_max);
	  cell_ptr = cell_ptr->next;
	}

      column_min_max_widths = g_list_append (column_min_max_widths, min_max);
      
      min_columns_width += min_max->min_width;
      max_columns_width += min_max->max_width;

      column_ptr = column_ptr->next;
    }

  if (l.computed != RUIN_CSS_VALUE_AUTO)
    {
      GList *column_ptr = table->columns;
      GList *column_min_max_width_ptr = column_min_max_widths;
      int additional_width = 0;

      ruin_length_t t;
      
      t.units = RUIN_LAYOUT_UNITS_CHARS;
      t.used = table->node->containing_block->width.used;

      ruin_css_normalize_length 
	(&l, &t, table->node->window->font_width, table->node->window->dpi, 
	 TRUE);  

      l.used = MAX (l.used, min_columns_width);
      
      if (l.used > min_columns_width)
	additional_width = l.used - min_columns_width;
      
      while (column_ptr != NULL)
	{
	  ruin_layout_table_column *column = 
	    (ruin_layout_table_column *) column_ptr->data;
	  min_max_width *min_max = 
	    (min_max_width *) column_min_max_width_ptr->data;
	  
	  column->width = min_max->min_width;
	  
	  column_ptr = column_ptr->next;
	  column_min_max_width_ptr = column_min_max_width_ptr->next;
	}
      
      while (additional_width > 0)
	{
	  column_ptr = table->columns;
	  while (column_ptr != NULL)
	    {
	      ruin_layout_table_column *column = 
		(ruin_layout_table_column *) column_ptr->data;

	      column->width++;
	      additional_width--;

	      column_ptr = column_ptr->next;
	    }
	}
    }
  else
    {
      GList *column_ptr = table->columns;
      GList *column_min_max_width_ptr = column_min_max_widths;

      if (max_columns_width < box->containing_block->width.used)
	{
	  table->width = max_columns_width;

	  while (column_ptr != NULL)
	    {
	      ruin_layout_table_column *column = 
		(ruin_layout_table_column *) column_ptr->data;
	      min_max_width *min_max = 
		(min_max_width *) column_min_max_width_ptr->data;

	      column->width = min_max->max_width;

	      column_ptr = column_ptr->next;
	      column_min_max_width_ptr = column_min_max_width_ptr->next;
	    }
	}
      else 
	{
	  table->width = min_columns_width;

	  while (column_ptr != NULL)
	    {
	      ruin_layout_table_column *column = 
		(ruin_layout_table_column *) column_ptr->data;
	      min_max_width *min_max = 
		(min_max_width *) column_min_max_width_ptr->data;

	      column->width = min_max->min_width;

	      column_ptr = column_ptr->next;
	      column_min_max_width_ptr = column_min_max_width_ptr->next;
	    }
	}
    }

  return table;
}

static void size_column_fixed
(ruin_layout_table *table, ruin_layout_table_column *column, 
 ruin_layout_table_cell *first_row_cell)
{
  ruin_length_t t;
  ruin_length_t l = 
    { RUIN_LAYOUT_UNITS_CHARS, RUIN_CSS_VALUE_NONE, RUIN_CSS_VALUE_NONE };

  t.units = RUIN_LAYOUT_UNITS_CHARS;
  t.used = table->node->containing_block->width.used;

  if (column->node != NULL)
    l = ruin_css_parse_size (ruin_box_css_lookup (column->node, "width"));
  if (l.computed == RUIN_LAYOUT_VALUE_NONE || l.computed == RUIN_CSS_VALUE_AUTO)
    {
      if (first_row_cell != NULL)
	l = ruin_css_parse_size 
	  (ruin_box_css_lookup (first_row_cell->node, "width"));
      else 
	{
	  ruin_layout_table_cell *cell = (ruin_layout_table_cell *) 
	    g_list_nth_data (column->cells, 0);
	  l = ruin_css_parse_size (ruin_box_css_lookup (cell->node, "width"));
	}
    }

  ruin_css_normalize_length 
    (&l, &t, table->node->window->font_width, table->node->window->dpi, TRUE);  

  column->width = l.used;
}

ruin_layout_table *ruin_layout_table_fixed (ruin_box_t *box)
{
  ruin_layout_table *table = parse_table (box);
  GList *column_ptr = table->columns;
  GList *cell_ptr = NULL;
  int available_width = 0;

  ruin_length_t table_width = ruin_css_parse_size 
    (ruin_box_css_lookup (box, "width"));
  ruin_box_normalize_width (&table_width, box->containing_block, TRUE);

  table->width = table_width.used;
  available_width = table_width.used; 

  if (g_list_length (table->rows) > 0)
    {
      ruin_layout_table_row *row = (ruin_layout_table_row *) table->rows->data;
      cell_ptr = row->cells;
    }

  while (column_ptr != NULL)
    {
      ruin_layout_table_cell *cell = NULL;
      ruin_layout_table_column *column = (ruin_layout_table_column *) 
	column_ptr->data;

      if (cell_ptr != NULL)
	{
	  cell = (ruin_layout_table_cell *) cell_ptr->data;
	  cell_ptr = cell_ptr->next;
	}
      
      size_column_fixed (table, column, cell);
      column_ptr = column_ptr->next;
    }

  if (available_width > 0)
    {
      GList *widthless_columns = NULL;
      GList *child_ptr = table->columns;
      int per_column_width = 0;
            
      while (child_ptr != NULL)
	{
	  ruin_layout_table_column *column = (ruin_layout_table_column *) 
	    child_ptr->data;
	  if (column->width == RUIN_CSS_VALUE_AUTO
	      || column->width == RUIN_CSS_VALUE_NONE)
	    widthless_columns = g_list_append 
	      (widthless_columns, child_ptr->data);

	  child_ptr = child_ptr->next;
	}

      child_ptr = widthless_columns;
      if (child_ptr != NULL)
	{
	  per_column_width = available_width 
	    / g_list_length (widthless_columns);
           
	  while (child_ptr != NULL)
	    {
	      ruin_layout_table_column *column = (ruin_layout_table_column *) 
		child_ptr->data;
	      
	      column->width = per_column_width;
	      child_ptr = child_ptr->next;
	    }
	}
    }
    
  return table;
}

static void free_column (gpointer d)
{
  ruin_layout_table_column *column = (ruin_layout_table_column *) d;

  g_list_free (column->cells);
  free (column);
}

static void free_row (gpointer d)
{
  ruin_layout_table_row *row = (ruin_layout_table_row *) d;

  g_list_free (row->cells);
  free (row);
}

void ruin_layout_table_free (ruin_layout_table *table)
{
  g_list_free_full (table->columns, free_column);
  g_list_free_full (table->rows, free_row);
  free (table);
}
