/* render.c: Rendering and sizing routines for libRUIN
 * Copyright (C) 2011 Julian Graham
 *
 * libRUIN is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <assert.h>
#include <ctype.h>
#include <curses.h>
#include <glib.h>
#include <libguile.h>
#include <math.h>
#include <stdlib.h>
#include <string.h>

#include "css.h"
#include "layout.h"
#include "render.h"
#include "util.h"
#include "window.h"

/* Forward prototype... */

static void render_tree (ruin_box_t *, int);

static void set_matching_colors 
(ruin_window_t *w, int fg_attrs, short ncurses_fg, short ncurses_bg)
{
  int i;

  if ((ncurses_fg == COLOR_WHITE) && (ncurses_bg == COLOR_BLACK)) {
    wcolor_set (w->window, 1, NULL);
    return;
  }
  
  wattron (w->window, fg_attrs);
  for (i = 2; i < COLOR_PAIRS; i++) {
    short _fg = 0, _bg = 0;
    (void) pair_content (i, &_fg, &_bg); 
    if ((_fg == COLOR_BLACK) && (_bg == COLOR_BLACK)) {
      init_pair (i, ncurses_fg, ncurses_bg);
      wcolor_set (w->window, i, NULL);
      break;
    } else if ((_fg == ncurses_fg) && (_bg == ncurses_bg)) {
      wcolor_set (w->window, i, NULL);
      break;
    }
  }
}

static void set_colors (ruin_window_t *w, enum ruin_layout_foreground_color fg,
			enum ruin_layout_background_color bg) {
  short ncurses_fg = COLOR_BLACK, ncurses_bg = COLOR_BLACK;
  int fg_attrs = A_NORMAL;

  if (!has_colors ()) return;

  switch(fg) 
    {
    case RUIN_LAYOUT_FG_COLOR_BR_BLACK: fg_attrs |= A_BOLD;
    case RUIN_LAYOUT_FG_COLOR_BLACK:
      ncurses_fg = COLOR_BLACK; break;
    case RUIN_LAYOUT_FG_COLOR_BR_RED: fg_attrs |= A_BOLD;
    case RUIN_LAYOUT_FG_COLOR_RED:
      ncurses_fg = COLOR_RED; break;
    case RUIN_LAYOUT_FG_COLOR_BR_GREEN: fg_attrs |= A_BOLD;
    case RUIN_LAYOUT_FG_COLOR_GREEN:
      ncurses_fg = COLOR_GREEN; break;
    case RUIN_LAYOUT_FG_COLOR_BR_YELLOW: fg_attrs |= A_BOLD;
    case RUIN_LAYOUT_FG_COLOR_YELLOW:
      ncurses_fg = COLOR_YELLOW; break;
    case RUIN_LAYOUT_FG_COLOR_BR_BLUE: fg_attrs |= A_BOLD;
    case RUIN_LAYOUT_FG_COLOR_BLUE:
      ncurses_fg = COLOR_BLUE; break;
    case RUIN_LAYOUT_FG_COLOR_BR_MAGENTA: fg_attrs |= A_BOLD;
    case RUIN_LAYOUT_FG_COLOR_MAGENTA:
      ncurses_fg = COLOR_MAGENTA; break;
    case RUIN_LAYOUT_FG_COLOR_BR_CYAN: fg_attrs |= A_BOLD;
    case RUIN_LAYOUT_FG_COLOR_CYAN:
      ncurses_fg = COLOR_CYAN; break;
    case RUIN_LAYOUT_FG_COLOR_BR_WHITE: fg_attrs |= A_BOLD;
    case RUIN_LAYOUT_FG_COLOR_WHITE:
      ncurses_fg = COLOR_WHITE; break;
    }
  
  switch (bg) 
    {
    case RUIN_LAYOUT_BG_COLOR_BLACK: ncurses_bg = COLOR_BLACK; break;
    case RUIN_LAYOUT_BG_COLOR_RED: ncurses_bg = COLOR_RED; break;
    case RUIN_LAYOUT_BG_COLOR_GREEN: ncurses_bg = COLOR_GREEN; break;
    case RUIN_LAYOUT_BG_COLOR_YELLOW: ncurses_bg = COLOR_YELLOW; break;
    case RUIN_LAYOUT_BG_COLOR_BLUE: ncurses_bg = COLOR_BLUE; break;
    case RUIN_LAYOUT_BG_COLOR_MAGENTA: ncurses_bg = COLOR_MAGENTA; break;
    case RUIN_LAYOUT_BG_COLOR_CYAN: ncurses_bg = COLOR_CYAN; break;
    case RUIN_LAYOUT_BG_COLOR_WHITE: ncurses_bg = COLOR_WHITE; break;
    }
  
  wattrset (w->window, A_NORMAL);
  set_matching_colors (w, fg_attrs, ncurses_fg, ncurses_bg);
}

static void update_transparent_background (ruin_box_t *box)
{
  short fg = 0, bg = 0;
  short pair = 0;
  attr_t attr;
  chtype ch = winch (box->window->window);
  
  pair_content (PAIR_NUMBER (ch & A_COLOR), NULL, &bg);
  wattr_get (box->window->window, &attr, &pair, NULL);
  pair_content (pair, &fg, NULL);

  set_matching_colors (box->window, A_NORMAL, fg, bg);
}

static void set_text_decorations (ruin_box_t *box, int *text_decorations)
{
  char *td = ruin_box_css_lookup (box, "text-decoration");
  if (strstr (td, "underline") != NULL)
    *text_decorations |= RUIN_RENDER_TEXT_DECORATION_UNDERLINE;
  if (strstr (td, "overline") != NULL)
    *text_decorations |= RUIN_RENDER_TEXT_DECORATION_OVERLINE;
  if (strstr (td, "line-through") != NULL)
    *text_decorations |= RUIN_RENDER_TEXT_DECORATION_LINE_THROUGH;
  if (strstr (td, "blink") != NULL)
    *text_decorations |= RUIN_RENDER_TEXT_DECORATION_BLINK;
}

static void set_attrs (ruin_box_t *box, int text_decorations) 
{
  char *font_weight = ruin_box_css_lookup (box, "font-weight");
  if ((text_decorations & RUIN_RENDER_TEXT_DECORATION_UNDERLINE) ||
      (text_decorations & RUIN_RENDER_TEXT_DECORATION_OVERLINE) ||
      (text_decorations & RUIN_RENDER_TEXT_DECORATION_LINE_THROUGH))
    wattron (box->window->window, A_UNDERLINE);
  if (text_decorations & RUIN_RENDER_TEXT_DECORATION_BLINK)
    wattron (box->window->window, A_BLINK);
  
  if (strcmp (font_weight, "bold") == 0)
    wattron (box->window->window, A_BOLD);
  else if (strcmp (font_weight, "bolder") == 0)
    wattron (box->window->window, A_STANDOUT);
}

static void setup_colors (ruin_box_t *b, char *fg) 
{
  enum ruin_layout_foreground_color temp_fg;
  enum ruin_layout_background_color temp_bg;

  char *c = ruin_box_css_lookup (b, fg);
  char *bc = ruin_box_css_lookup (b, "background-color");

  temp_fg = ruin_css_match_foreground_color (b->window, c);
  temp_bg = ruin_css_match_background_color (b->window, bc, NULL);

  set_colors (b->window, temp_fg, temp_bg);
}

static void setup_printing (ruin_box_t *b, int text_decorations, char *fg) 
{
  setup_colors (b, fg);
  set_attrs (b, text_decorations);
}

static void draw_text 
(ruin_box_t *line, ruin_inline_box_t *ib, int text_decorations,
 ruin_layout_line_state_t *state)
{
  ruin_box_t *b = (ruin_box_t *) ib;
  char *white_space = ruin_box_css_lookup (b, "white-space");
  char *background_color = ruin_box_css_lookup (b, "background-color");
  int transparent_background = strcmp (background_color, "transparent") == 0;

  int need_print_setup = FALSE;
  int small_caps = FALSE;

  int left = b->left;
  int idx = 0;
  
  setup_printing (b, text_decorations, "color");
  small_caps = strcmp 
    (ruin_box_css_lookup (b, "font-variant"), "small-caps") == 0;

  if (strcmp (white_space, "normal") == 0) {
    int len = strlen (ib->content_ptr);
    int idx = 0;
    int seen_space = FALSE;
    char *last_word_start = NULL;
    char *content_ptr = ib->content_ptr;

    while (idx < len && left - b->left < b->width.used)
      {
	char c = content_ptr[idx++];

	if (isspace (c)) 
	  {
	    short more_non_space = FALSE;
	    seen_space = TRUE;
	    while (idx < len)
	      {
		if (!isspace (content_ptr[idx++]))
		  {
		    more_non_space = TRUE;
		    idx--;
		    break;
		  }
	      }

	    if ((last_word_start != NULL 
		 && (more_non_space || !state->is_last)) 
		|| (!state->is_first && !isspace (state->last_char)))
	      {
		int i = 0;
		for (; i < b->word_spacing.used + 1; i++)
		  {
		    wmove (b->window->window, b->top, left);
		    if (transparent_background)
		      update_transparent_background (b);
		    waddch (b->window->window, ' ');
		    left++;
		  }
		state->last_char = c;
	      }
	  }
	else 
	  {
	    if (seen_space)
	      seen_space = FALSE;
	    if (last_word_start == NULL)
	      last_word_start = content_ptr;
	    
	    wmove (b->window->window, b->top, left);
	    if (need_print_setup)
	      {
		need_print_setup = FALSE;

		set_text_decorations (b, &text_decorations);
		setup_printing (b, text_decorations, "color");
		small_caps = strcmp 
		  (ruin_box_css_lookup (b, "font-variant"), "small-caps") == 0;
	      }
	    if (transparent_background)
	      update_transparent_background (b);

	    waddch (b->window->window, small_caps ? toupper (c) : c);

	    if (ruin_css_pseudo_element_is_active 
		(b->window, RUIN_CSS_PSEUDO_ELEMENT_FIRST_LETTER)
		&& !ispunct (c))
	      {
		ruin_css_deactivate_pseudo_element 
		  (b->window, RUIN_CSS_PSEUDO_ELEMENT_FIRST_LETTER);
		need_print_setup = TRUE;
	      }

	    left += 1;
	    state->last_char = c;
	    if (idx < len)
	      left += b->letter_spacing.used;
	  }
      }
  } else if (strcmp (white_space, "pre") == 0) {
    char *next_line = strchr (ib->content_ptr, '\a');    
    int len = next_line == NULL ? strlen (ib->content_ptr) 
      : next_line - ib->content_ptr;

    while (idx < len)
      {
	char c = ib->content_ptr[idx++];
	wmove (b->window->window, line->top, left);
	if (transparent_background)
	  update_transparent_background (b);
	if (need_print_setup)
	  {
	    need_print_setup = FALSE;
	    
	    set_text_decorations (b, &text_decorations);
	    setup_printing (b, text_decorations, "color");
	    small_caps = strcmp 
	      (ruin_box_css_lookup (b, "font-variant"), "small-caps") == 0;
	  }

	waddch (line->window->window, small_caps ? toupper (c) : c);
	if (isspace (c) && !isspace (state->last_char))
	  left += b->word_spacing.used;
	else 
	  {
	    if (ruin_css_pseudo_element_is_active 
		(b->window, RUIN_CSS_PSEUDO_ELEMENT_FIRST_LETTER)
		&& !ispunct (c))
	      {
		ruin_css_deactivate_pseudo_element 
		  (b->window, RUIN_CSS_PSEUDO_ELEMENT_FIRST_LETTER);
		need_print_setup = TRUE;
	      }

	    if (idx < len) left += b->letter_spacing.used;
	  }
	state->last_char = c;
	left += 1;
      }
  } else if (strcmp (white_space, "nowrap") == 0) {
    assert (1 == 0);
  } else if (strcmp (white_space, "pre-wrap") == 0) {
    assert (1 == 0);
  } else if (strcmp (white_space, "pre-line") == 0) {
    assert (1 == 0);
  }

}

static void draw_inline
(ruin_box_t *line, ruin_box_t *b, int text_decorations, 
 ruin_layout_line_state_t *state) 
{
  ruin_inline_box_t *ib = (ruin_inline_box_t *) b;
  set_text_decorations (b, &text_decorations);

  if (ib->content_ptr != NULL)
    draw_text (line, ib, text_decorations, state);
  else 
    {
      GList *child_ptr = b->children;
      while (child_ptr != NULL)
	{
	  ruin_box_t *child = child_ptr->data;
	  draw_inline (line, child, text_decorations, state);
	  child_ptr = child_ptr->next;
	}
    }
}

static void draw_line (ruin_box_t *b, int text_decorations)
{
  GList *child_ptr = b->children;

  ruin_layout_line_state_t state;

  state.is_first = TRUE;
  state.is_last = g_list_length (child_ptr) == 1;
  state.last_char = 0x0;

  while (child_ptr != NULL)
    {
      ruin_box_t *child = child_ptr->data;
      draw_inline (b, child, text_decorations, &state);
      state.is_first = FALSE;
      child_ptr = child_ptr->next;
    }

  if (b->window->render_state->active_pseudo_elements & 
      RUIN_CSS_PSEUDO_ELEMENT_FIRST_LINE)
    ruin_css_deactivate_pseudo_element 
      (b->window, RUIN_CSS_PSEUDO_ELEMENT_FIRST_LINE);
}

static int _get_border_char(char *s, enum _ruin_render_border_char c) {
  if ((strcmp(s, "solid") == 0) || (strcmp(s, "inset") == 0) ||
      (strcmp(s, "outset") == 0)) {
    switch(c) {
    case RUIN_RENDER_BORDER_ULCORNER: return ACS_ULCORNER;
    case RUIN_RENDER_BORDER_URCORNER: return ACS_URCORNER;
    case RUIN_RENDER_BORDER_LLCORNER: return ACS_LLCORNER;
    case RUIN_RENDER_BORDER_LRCORNER: return ACS_LRCORNER;
    case RUIN_RENDER_BORDER_HLINE: return ACS_HLINE;
    case RUIN_RENDER_BORDER_VLINE: return ACS_VLINE;
    case RUIN_RENDER_BORDER_LTEE: return ACS_LTEE;
    case RUIN_RENDER_BORDER_RTEE: return ACS_RTEE;
    case RUIN_RENDER_BORDER_BTEE: return ACS_BTEE;
    case RUIN_RENDER_BORDER_TTEE: return ACS_TTEE;
    default: return ' ';
    }
  } else if (strcmp(s, "dashed") == 0) {
    switch(c) {
    case RUIN_RENDER_BORDER_ULCORNER:
    case RUIN_RENDER_BORDER_URCORNER:
    case RUIN_RENDER_BORDER_LLCORNER:
    case RUIN_RENDER_BORDER_LRCORNER:
    case RUIN_RENDER_BORDER_LTEE:
    case RUIN_RENDER_BORDER_RTEE:
    case RUIN_RENDER_BORDER_BTEE:
    case RUIN_RENDER_BORDER_TTEE: return '+';
    case RUIN_RENDER_BORDER_HLINE: return '-';
    case RUIN_RENDER_BORDER_VLINE: return '|';
    default: return ' ';
    }
  } else if (strcmp(s, "dotted") == 0) {
    switch(c) {
    case RUIN_RENDER_BORDER_ULCORNER: 
    case RUIN_RENDER_BORDER_HLINE:
    case RUIN_RENDER_BORDER_URCORNER: 
    case RUIN_RENDER_BORDER_TTEE: return '.';
    case RUIN_RENDER_BORDER_LLCORNER:
    case RUIN_RENDER_BORDER_LRCORNER:
    case RUIN_RENDER_BORDER_VLINE: 
    case RUIN_RENDER_BORDER_LTEE:
    case RUIN_RENDER_BORDER_RTEE:
    case RUIN_RENDER_BORDER_BTEE: return ':';
    default: return ' ';
    }
  } else return ' ';
}

static void draw_border (ruin_box_t *b, int top, int left) 
{
  int i = 0;
  int w = b->width.used + b->padding_left.used + 
    b->padding_right.used + b->border_left_width.used + 
    b->border_right_width.used;
  int h = b->height.used + b->padding_top.used + 
    b->padding_bottom.used + b->border_top_width.used +
    b->border_bottom_width.used;

  char *border_style = ruin_box_css_lookup(b, "border-top-style");

  /* Draw the top... */

  if ((strcmp(border_style, "none") != 0) &&
      (b->border_top_width.used > 0)) {

    if (strcmp(border_style, "inset") == 0) {
    } else if (strcmp(border_style, "outset") == 0) {
    } else setup_printing (b, 0, "border-top-color");

    for (i = 0; i < b->border_top_width.used; i++) {
      int start = (b->border_left_width.used * i) / 
	b->border_top_width.used;
      int end = (b->border_right_width.used * i) /
	b->border_top_width.used;
      move(top + i, left + start);
      hline(_get_border_char(border_style, RUIN_RENDER_BORDER_ULCORNER), 1);
      move(top + i, left + start + 1);
      hline(_get_border_char(border_style, RUIN_RENDER_BORDER_HLINE), 
	    w - end - start - 1);
      move(top + i, left + w - end - 1);
      hline(_get_border_char(border_style, RUIN_RENDER_BORDER_URCORNER), 1);
    }
  }

  /* Draw the bottom... */
  
  border_style = ruin_box_css_lookup (b, "border-bottom-style");
  if ((strcmp(border_style, "none") != 0) &&
      (b->border_bottom_width.used > 0)) {

    if (strcmp(border_style, "inset") == 0) {
    } else if (strcmp(border_style, "outset") == 0) {
    } else setup_printing (b, 0, "border-bottom-color");

    for (i = b->border_bottom_width.used; i > 0; i--) {
      int start = 
	(b->border_left_width.used * (b->border_bottom_width.used - i)) /
	b->border_bottom_width.used;
      int end = 
	(b->border_right_width.used * 
	 (b->border_bottom_width.used - i)) / 
	b->border_bottom_width.used;
      move(top + h - b->border_bottom_width.used + i - 1, left + start);
      hline(_get_border_char(border_style, RUIN_RENDER_BORDER_LLCORNER), 1);
      move(top + h - b->border_bottom_width.used + i - 1, left + start + 1);
      hline(_get_border_char(border_style, RUIN_RENDER_BORDER_HLINE), 
	    w - end - start - 1);
      move(top + h - b->border_bottom_width.used + i - 1, 
	   left + w - end - 1);
      hline(_get_border_char(border_style, RUIN_RENDER_BORDER_LRCORNER), 1);
    }
  }

  /* Draw the left... */

  if ((strcmp(border_style = ruin_box_css_lookup (b, "border-left-style"), 
	      "none") != 0) && (b->border_left_width.used > 0)) {

    if (strcmp(border_style, "inset") == 0) {
    } else if (strcmp(border_style, "outset") == 0) {
    } else setup_printing (b, 0, "border-left-color");

    for (i = 0; i < b->border_left_width.used; i++) {
      int start = (b->border_top_width.used / 
		   b->border_left_width.used) * (i + 1);
      int end = (b->border_bottom_width.used / b->border_left_width.used)
	* (i + 1);
      move(top + start, left + i);
      vline(_get_border_char(border_style, RUIN_RENDER_BORDER_VLINE), 
	    h - start - end);
    }
  }

  /* Draw the right... */

  if ((strcmp(border_style = ruin_box_css_lookup (b, "border-right-style"),
	      "none") != 0) && (b->border_right_width.used > 0)) {

    if (strcmp(border_style, "inset") == 0) {
    } else if (strcmp(border_style, "outset") == 0) {
    } else setup_printing (b, 0, "border-right-color");

    for (i = b->border_right_width.used; i > 0; i--) {
      int start = (b->border_top_width.used / 
		   b->border_right_width.used) * 
	(b->border_right_width.used - i + 1);
      int end = (b->border_bottom_width.used /
		 b->border_right_width.used) * 
	(b->border_right_width.used - i + 1);
      move(top + start, left + w - b->border_right_width.used + i - 1);
      vline(_get_border_char(border_style, RUIN_RENDER_BORDER_VLINE), 
	    h - start - end);
    }
  }
}

static void 
draw_block (ruin_box_t *b, int text_decorations) 
{
  char *background_color = ruin_box_css_lookup (b, "background-color");
  int render_top, render_left, full_width, full_height;
  if ((b == NULL) || (!b->visible))
    return;

  full_width = b->margin_left.used + b->border_left_width.used +
    b->padding_left.used + b->width.used + b->padding_right.used +
    b->border_right_width.used + b->margin_right.used;
  full_height = b->margin_top.used + b->border_top_width.used +
    b->padding_top.used + b->height.used + b->padding_bottom.used +
    b->border_bottom_width.used + b->margin_bottom.used;

  /* We're not going to render 0-size elements.  The border should be factored
     into the node's current_width/height value, so if it's zero, there's no
     border, either. */

  render_top = b->top + b->margin_top.used + b->border_top_width.used;
  render_left = b->left + b->margin_left.used + b->border_left_width.used;
  
  /* First paint the background. Don't use the full setup_printing function
     because we don't want to paint attributes like underline onto the
     background. */
  
  setup_colors (b, "color");
  
  if (strcmp (background_color, "transparent") != 0)
    { int i;
      int num_lines = full_height - b->margin_top.used - b->margin_bottom.used;
      int width = full_width - b->margin_left.used - b->margin_right.used;
      char *line = calloc (width + 1, sizeof(char));
      (void) memset (line, (int) ' ', width);
      for (i = 0; i < num_lines; i++) {
	move (render_top - b->border_top_width.used + i, 
	      render_left - b->border_left_width.used);
	waddstr (b->window->window, line);
      }
    }
  
  /* Now render the children. */

  ruin_css_activate_pseudo_element 
    (b->window, RUIN_CSS_PSEUDO_ELEMENT_FIRST_LINE);
  ruin_css_activate_pseudo_element 
    (b->window, RUIN_CSS_PSEUDO_ELEMENT_FIRST_LETTER);
  { GList *elt_ptr = b->children;
    while(elt_ptr != NULL) 
      {
	render_tree ((ruin_box_t *) elt_ptr->data, text_decorations);
	elt_ptr = elt_ptr->next; 
      }
  }

  /* The children might have changed the colors... */

  setup_printing(b, text_decorations, "color");
  
  /* Draw the border. */

  draw_border(b, 
	      render_top - b->border_top_width.used,
	      render_left - b->border_left_width.used);
  return;
}

static int decimal_width (int n)
{
  return ceil (log (n + 1) / log (10));
}

static void 
draw_marker (ruin_box_t *b, int text_decorations)
{
  ruin_marker_box_t *m = (ruin_marker_box_t *) b;
  char *marker_str = NULL, *format_str = NULL;
  int required_digits, required_format_digits = 0;

  setup_colors (b, "color");
  move (b->top, b->left);

  switch (m->style)
    {
    case RUIN_LAYOUT_LIST_STYLE_TYPE_DISC:
      waddch (b->window->window, '*'); break;
    case RUIN_LAYOUT_LIST_STYLE_TYPE_CIRCLE:
      waddch (b->window->window, 'o'); break;
    case RUIN_LAYOUT_LIST_STYLE_TYPE_SQUARE:
      waddch (b->window->window, '#'); break;
    case RUIN_LAYOUT_LIST_STYLE_TYPE_LOWER_ALPHA:
    case RUIN_LAYOUT_LIST_STYLE_TYPE_LOWER_LATIN:
      waddch (b->window->window, 98 + ((m->ordinal - 1) % 26)); break;
    case RUIN_LAYOUT_LIST_STYLE_TYPE_UPPER_LATIN:
    case RUIN_LAYOUT_LIST_STYLE_TYPE_UPPER_ALPHA:
      waddch (b->window->window, 66 + ((m->ordinal - 1) % 26)); break;
    case RUIN_LAYOUT_LIST_STYLE_TYPE_DECIMAL:
      required_digits = decimal_width (m->ordinal + 1);
      
      marker_str = calloc (required_digits + 2, sizeof (char));
      snprintf (marker_str, required_digits + 2, "%d.", m->ordinal + 1);
      
      waddstr (b->window->window, marker_str);
      free (marker_str);
      break;
      
    case RUIN_LAYOUT_LIST_STYLE_TYPE_DECIMAL_LEADING_ZERO:
      required_digits = decimal_width (m->ordinal + 1);
      required_format_digits = decimal_width (m->length);

      format_str = calloc (required_format_digits + 4, sizeof (char));
      marker_str = calloc (required_digits + 2, sizeof (char));
      snprintf (format_str, required_format_digits + 4, "%%0.%dd.", m->length);
      snprintf (marker_str, required_digits + 2, format_str, m->ordinal + 1);
      
      waddstr (b->window->window, marker_str);
      free (format_str);
      free (marker_str);
      break;

    case RUIN_LAYOUT_LIST_STYLE_TYPE_LOWER_ROMAN:
    case RUIN_LAYOUT_LIST_STYLE_TYPE_UPPER_ROMAN:
      marker_str = ruin_util_arabic_to_roman 
	(m->ordinal + 1, m->style == RUIN_LAYOUT_LIST_STYLE_TYPE_UPPER_ROMAN);
      waddstr (b->window->window, marker_str);
      free (marker_str);
      break;

    case RUIN_LAYOUT_LIST_STYLE_TYPE_LOWER_GREEK:
    case RUIN_LAYOUT_LIST_STYLE_TYPE_ARMENIAN:
    case RUIN_LAYOUT_LIST_STYLE_TYPE_GEORGIAN: assert (1 == 0);
    case RUIN_LAYOUT_LIST_STYLE_TYPE_NONE: break;
    }
}

static void render_tree (ruin_box_t *b, int text_decorations) 
{
  set_text_decorations (b, &text_decorations);
  switch (b->type)
    {
    case RUIN_LAYOUT_BOX_TYPE_ANONYMOUS_BLOCK: 
    case RUIN_LAYOUT_BOX_TYPE_BLOCK: draw_block (b, text_decorations); 
      break;
    case RUIN_LAYOUT_BOX_TYPE_LINE: draw_line (b, text_decorations); 
      break;
    case RUIN_LAYOUT_BOX_TYPE_MARKER: draw_marker (b, text_decorations);
      break;
    default: break;      
    }
}

void ruin_render_render_tree (ruin_box_t *b) 
{
  render_tree (b, 0);
}

