/* parse.c: Parsing utilities for libRUIN
 * Copyright (C) 2011 Julian Graham
 *
 * libRUIN is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <assert.h>
#include <glib.h>
#include <libguile.h>
#include <stdlib.h>

#include "css.h"
#include "parse.h"
#include "scheme.h"
#include "util.h"
#include "xml.h"

static GHashTable *nodes_by_scm = NULL;
static ruin_node_t *parse_node (ruin_window_t *, SCM, SCM, ruin_node_t *);

SCM parse_in_document_style(ruin_window_t *w, SCM d, char *s) {
  return ruin_scheme_scss_css_to_scss
    (w, s, ruin_scheme_sdom_document_uri (w, d));
}

static void init_node 
(ruin_node_t *node, enum ruin_parse_node_type t, SCM d, SCM n, ruin_node_t *p) 
{
  node->type = t;
  node->node = n;
  node->doc = d;
  node->parent = p;
}

ruin_node_element_t *ruin_node_element_new (SCM d, SCM n, ruin_node_t *p)
{
  ruin_node_element_t *elt = calloc (sizeof (ruin_node_element_t), 1);
  init_node ((ruin_node_t *) elt, RUIN_PARSE_NODE_TYPE_ELEMENT, d, n, p);

  elt->inherent_attribute_style = SCM_EOL;
  elt->additional_attribute_style = SCM_EOL;

  return elt;
}

ruin_node_prototype_t *ruin_node_prototype_new (GHashTable *properties)
{
  ruin_node_prototype_t *proto = calloc (sizeof (ruin_node_prototype_t), 1);
  init_node ((ruin_node_t *) proto, RUIN_PARSE_NODE_TYPE_PROTOTYPE, 
	     SCM_BOOL_F, SCM_BOOL_F, NULL);
  
  proto->properties = properties;
  
  return proto;
}

ruin_node_text_t *ruin_node_text_new (SCM d, SCM n, ruin_node_t *p)
{
  ruin_node_text_t *t = calloc (sizeof (ruin_node_text_t), 1);
  init_node ((ruin_node_t *) t, RUIN_PARSE_NODE_TYPE_TEXT, d, n, p);
  return t;
}

void ruin_node_free (ruin_node_t *node)
{
  free (node);
}

void ruin_parse_init ()
{
  nodes_by_scm = g_hash_table_new (g_direct_hash, g_direct_equal);
}

static void parse_attributes 
(ruin_window_t *w, SCM d, ruin_node_element_t *e, SCM attrlst)
{
  while (!scm_is_null (attrlst))
    {
      SCM attr = SCM_CAR (attrlst);
  
      if (strcmp (ruin_scheme_sdom_node_name (w, attr), "style") == 0) 
	{
	  char *attr_str = ruin_scheme_sdom_value (w, attr);
	  SCM parsed_style = parse_in_document_style
	    (w, d, scm_to_locale_string
	     (scm_string_append (scm_list_3 (scm_from_locale_string ("* {"),
					     scm_from_locale_string (attr_str),
					     scm_from_locale_string ("}")))));
	  if (scm_is_null (e->additional_attribute_style)) 
	    {
	      e->additional_attribute_style = parsed_style;
	      scm_gc_protect_object (e->additional_attribute_style);
	    } 
	  else scm_append_x (scm_list_2 (e->additional_attribute_style, 
					 SCM_CADAR (parsed_style)));
	}

      attrlst = SCM_CDR (attrlst);
    }
}

static ruin_node_element_t *parse_element 
(ruin_window_t *w, SCM d, SCM n, ruin_node_t *p) 
{
  ruin_node_element_t *elt = ruin_node_element_new (d, n, p);
  ruin_node_t *node = (ruin_node_t *) elt;

  SCM attrs = ruin_scheme_sdom_attributes (w, n);
  SCM first_child = ruin_scheme_sdom_first_child (w, n);

  node->doc = d;
  node->parent = p;
  
  if (strcmp (ruin_scheme_sdom_node_name (w, n), "style") == 0)
    {
      SCM scss = parse_in_document_style 
	(w, d, ruin_scheme_sdom_text_content (w, n));
      ruin_scheme_scss_set_cascade_author (w, w->cascade, scss);
    }
  
  if (attrs != SCM_BOOL_F)
    parse_attributes (w, d, elt, attrs);
  
  if (first_child != SCM_BOOL_F)
    {
      ruin_node_t *c_old = NULL;
      while (first_child != SCM_BOOL_F) 
	{
	  ruin_node_t *c = parse_node (w, d, first_child, node);
	  if (c != NULL)
	    {
	      if (node->first_child == NULL)
		node->first_child = c;
	      node->children = g_list_append (node->children, c);
	      if (c_old != NULL)
		c_old->next_sibling = c;
	      c_old = c;
	    }
	  first_child = ruin_scheme_sdom_next_sibling (w, first_child);
	}
    }
  return elt;
} 

static ruin_node_text_t *parse_text_node 
(ruin_window_t *w, SCM d, SCM n, ruin_node_t *p)
{
  ruin_node_text_t *t = ruin_node_text_new (d, n, p);
  ruin_node_t *node = (ruin_node_t *) t;

  t->content = ruin_scheme_sdom_node_value (w, n);
  
  node->doc = d;
  node->parent = p;
  
  return t;
}

static ruin_node_t *parse_node (ruin_window_t *w, SCM d, SCM n, ruin_node_t *p)
{
  if (ruin_scheme_sdom_element_p (w, n)) 
    return (ruin_node_t *) parse_element (w, d, n, p);
  else if (ruin_scheme_sdom_text_node_p (w, n)) 
    return (ruin_node_t *) parse_text_node (w, d, n, p);
  else return NULL;
}

ruin_node_t *ruin_parse_document (ruin_window_t *w, SCM doc)
{
  return (ruin_node_t *) parse_element 
    (w, doc, ruin_scheme_sdom_document_element (w, doc), NULL);
}

ruin_node_t *ruin_parse_get_node_by_scm (SCM node)
{
  return (ruin_node_t *) g_hash_table_lookup (nodes_by_scm, node);
}

enum ruin_xml_dialect ruin_parse_determine_dialect (ruin_window_t *w, SCM doc)
{
  SCM dtd = ruin_scheme_sdom_doctype (w, doc);

  if (dtd == SCM_BOOL_F || dtd == SCM_EOL)
    {
      ruin_util_log (w, "could not determine XML dialect; assuming XHTML");
      return RUIN_XML_DIALECT_XHTML;
    }
  else 
    {
      char *type = ruin_scheme_sdom_node_name (w, dtd);
      if (strcmp (type, "xhtml") == 0 || strcmp (type, "html") == 0)
	return RUIN_XML_DIALECT_XHTML;
      else if (strcmp (type, "xul") == 0)
	return RUIN_XML_DIALECT_XUL;
      else 
	{
	  ruin_util_log (w, "unknown XML dialect %s", type);
	  assert (1 == 0);
	}
    }
}
