/* css.c: Cascade-management routines for libRUIN
 * Copyright (C) 2011 Julian Graham
 *
 * libRUIN is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <assert.h>
#include <glib.h>
#include <libguile.h>
#include <math.h>
#include <stdarg.h>
#include <stdio.h>
#include <string.h>

#include "css.h"
#include "layout.h"
#include "scheme.h"
#include "window.h"
#include "xml.h"

GHashTable *default_style_cache = NULL;
GHashTable *specification_delegation = NULL;

/* The orders here have to match the orders of the enumerations in the 
   header file! */

const char *ruin_css_fg_color_hex[16] = {
  "#000000", "#800000", "#008000", "#808000", "#000080", "#800080", "#008080",
  "#c0c0c0", "#808080", "#ff0000", "#00ff00", "#ffff00", "#0000ff", "#ff00ff",
  "#00ffff", "#ffffff"
};

const char *ruin_css_bg_color_hex[8] = {
  "#000000", "#800000", "#008000", "#808000", "#000080", "#800080", "#008080",
  "#c0c0c0"
};

long full_value_select_time = 0;
int full_value_select_num = 0;

static char *css_lookup (ruin_window_t *, ruin_node_element_t *, char *);

int ruin_css_get_rgb(ruin_window_t *win, char *c) {
  SCM sc = SCM_EOL;
  SCM val = scm_from_int32(0);

  if (c[0] == '#')
    val = scm_string_to_number(scm_from_locale_string (c + 1), 
			       scm_from_int32(16));
  else if (strstr(c, "rgb(") == c) {
    double c1f = 0, c2f = 0, c3f = 0;
    char *save = NULL;

    c+= 4;

    c1f = strtod(c, &save);
    if (save != NULL && save[0] == '%')
      c1f = (255 * MIN(c1f, 100)) / 100;
    else c1f = MIN(c1f, 255);

    c = index(c, ',') + 1;

    c2f = strtod(c, &save);
    if (save != NULL && save[0] == '%')
      c2f = (255 * MIN(c2f, 100)) / 100;
    else c2f = MIN(c2f, 255);

    c = index(c, ',') + 1;

    c3f = strtod(c, &save);
    if (save != NULL && save[0] == '%')
      c3f = (255 * MIN(c3f, 100)) / 100;
    else c3f = MIN(c3f, 255);

    return ((int) c1f << 16) + ((int) c2f << 8) + (int) c3f;
  } else {
    sc = ruin_scheme_scss_color_to_hex(win, c);
    if (scm_eq_p(sc, SCM_EOL) != SCM_BOOL_T) 
      val = scm_string_to_number(scm_substring(sc,
					       scm_from_int32(1), 
					       scm_from_int32(7)), 
				 scm_from_int32(16));
  }
  return scm_to_int32(val);
}

double ruin_css_get_color_distance(int c1, int c2) {
  int d_r = (c2 >> 16) - (c1 >> 16);
  int d_g = ((c2 & 0xff00) >> 8) - ((c1 & 0xff00) >> 8);
  int d_b = (c2 & 0xff) - (c1 & 0xff);

  return sqrt((d_r * d_r) + (d_g * d_g) + (d_b * d_b));
}

enum ruin_layout_foreground_color ruin_css_match_foreground_color
(ruin_window_t *win, char *c) {
  int rgb = ruin_css_get_rgb(win, c);
  int temp_rgb = 0;
  enum ruin_layout_foreground_color min_distance_color = 
    RUIN_LAYOUT_FG_COLOR_BLACK;
  double min_distance = -1;
  int i;
  for (i = 0; i < 16; i++) {
    double distance;
    temp_rgb = ruin_css_get_rgb(win, (char *) ruin_css_fg_color_hex[i]);
    distance = ruin_css_get_color_distance(rgb, temp_rgb);
    if ((min_distance == -1) || (distance < min_distance)) {
      min_distance = distance;
      min_distance_color = i;
    }
  }
  return min_distance_color;
}

enum ruin_layout_background_color ruin_css_match_background_color
(ruin_window_t *win, char *c, GList *l) 
{
  int rgb = -1;
  int temp_rgb = 0;
  enum ruin_layout_background_color min_distance_color = 
    RUIN_LAYOUT_FG_COLOR_BLACK;
  double min_distance = -1;
  int i;

  if (strcmp (c, "transparent") == 0) 
    {
      while (l != NULL) 
	{
	  ruin_node_element_t *t = (ruin_node_element_t *) l->data;
	  c = css_lookup (win, t, "background-color");
	  if (strcmp (c, "transparent") != 0) 
	    {
	      rgb = ruin_css_get_rgb (win, c);
	      break;
	    }
	  l = l->next;
	}
      if (rgb == -1)
	return min_distance_color;
    } else rgb = ruin_css_get_rgb (win, c);

  for (i = 0; i < 8; i++) 
    {
      double distance;
      temp_rgb = ruin_css_get_rgb (win, (char *) ruin_css_bg_color_hex[i]);
      distance = ruin_css_get_color_distance (rgb, temp_rgb);
      
      if ((min_distance == -1) || (distance < min_distance)) 
	{
	  min_distance = distance;
	  min_distance_color = i;
	}
    }
  return min_distance_color;
}

void ruin_css_activate_pseudo_element(ruin_window_t *w,
				      enum _ruin_css_pseudo_element e) 
{
  ruin_window_render_state_t *s = w->render_state;
  s->active_pseudo_elements |= e;
}

int ruin_css_pseudo_element_is_active (ruin_window_t *w,
				       enum _ruin_css_pseudo_element e) 
{
  ruin_window_render_state_t *s = w->render_state;
  return s->active_pseudo_elements & e;
}

void ruin_css_deactivate_pseudo_element(ruin_window_t *w,
					enum _ruin_css_pseudo_element e) 
{
  ruin_window_render_state_t *s = w->render_state;
  if (s->active_pseudo_elements & e) 
    s->active_pseudo_elements ^= e;
}

static GHashTable **_ruin_css_get_hash_ptr 
(ruin_node_element_t *tree, int mask) {
  if (mask & RUIN_CSS_PSEUDO_ELEMENT_FIRST_LETTER)
    return &tree->first_letter_style_cache;
  else if (mask & RUIN_CSS_PSEUDO_ELEMENT_FIRST_LINE)
    return &tree->first_line_style_cache;
  else if (mask & RUIN_CSS_PSEUDO_ELEMENT_BEFORE)
    return &tree->before_style_cache;
  else if (mask & RUIN_CSS_PSEUDO_ELEMENT_AFTER)
    return &tree->after_style_cache;
  else return &tree->style_cache;
}

char *specify_value (ruin_window_t *w, GHashTable *target, char *property)
{
  GList *general_properties = g_hash_table_lookup
    (specification_delegation, property);
  GList *general_property_ptr = general_properties;

  while (general_property_ptr != NULL)
    {
      char *general_property = (char *) general_property_ptr->data;
      char *general_value = g_hash_table_lookup (target, general_property);
    
      if (general_value == NULL)
	general_value = specify_value (w, target, general_property);
      if (general_value != NULL)
	return ruin_scheme_scss_specify (w, property, general_value);

      general_property_ptr = general_property_ptr->next;
    }

  return NULL;
}

static char *alternate_lookup_strategy 
(ruin_window_t *window, ruin_node_element_t *tree, char *prop)
{
  if (strcmp (prop, "border-top-color") == 0 ||
      strcmp (prop, "border-left-color") == 0 ||
      strcmp (prop, "border-bottom-color") == 0 ||
      strcmp (prop, "border-right-color") == 0)
    return css_lookup (window, tree, "color");
  return NULL;
}

static SCM make_synthetic_selection_context (ruin_window_t *window, char *css)
{
  int len = strlen (css);
  char *wrapper = calloc (len + 4, sizeof (char));
  SCM explicit_stylesheet = SCM_BOOL_F, cascade = SCM_BOOL_F;

  snprintf (wrapper, len + 3, "*{%s}", css);
  explicit_stylesheet = ruin_scheme_scss_css_to_scss (window, wrapper, NULL);
  cascade = ruin_scheme_scss_make_cascade (window);
  ruin_scheme_scss_set_cascade_agent (window, cascade, explicit_stylesheet);

  free (wrapper);

  return ruin_scheme_scss_make_selection_context 
    (window, ruin_scheme_scss_document_interface_sdom, SCM_BOOL_F, cascade);
}

static void select_values_inner
(ruin_window_t *window, SCM selection_context, GHashTable *target, SCM node)
{
  SCM selected_values = ruin_scheme_scss_select_values 
    (window, selection_context, node);
  while (selected_values != SCM_EOL)
    {
      SCM selected_value = SCM_CAR (selected_values);
      char *property = ruin_scheme_scss_selected_value_property 
	(window, selected_value);

      if (g_hash_table_lookup (target, property) == NULL)
	g_hash_table_insert (target, property, selected_value);

      selected_values = SCM_CDR (selected_values);
    }
}

static int calc_specificity (ruin_window_t *window, SCM selected_value)
{
  char *source = ruin_scheme_scss_selected_value_source 
    (window, selected_value);
  if (strcmp (source, "agent") == 0) {
    return 1;
  } else if (strcmp (source, "user") == 0) {
    return ruin_scheme_scss_selected_value_important_p 
      (window, selected_value) ? 5 : 2;
  } else if (strcmp (source, "author") == 0) {
    return ruin_scheme_scss_selected_value_important_p 
      (window, selected_value) ? 4 : 3;
  } else assert (1 == 0);
  return -1;
}

static void 
overwrite_with_explicit_style (gpointer key, gpointer value, gpointer user_data)
{
  ruin_window_t *window = (ruin_window_t *) ((void **) user_data)[0];
  GHashTable *target = (GHashTable *) ((void **) user_data)[1];

  if (g_hash_table_lookup (target, key) == NULL
      || (calc_specificity (window, (SCM) value) >=
	  calc_specificity (window, (SCM) g_hash_table_lookup (target, key))))
    g_hash_table_insert (target, key, value);
}

static void
selected_values_to_strings (gpointer key, gpointer value, gpointer user_data)
{
  ruin_window_t *window = (ruin_window_t *) ((void **) user_data)[0];
  GHashTable *target = (GHashTable *) ((void **) user_data)[1];

  g_hash_table_insert 
    (target, key, ruin_scheme_scss_selected_value_value (window, (SCM) value));
}

static void select_values 
(ruin_window_t *window, ruin_node_element_t *tree, GHashTable *target)
{
  void *user_data[2];
  ruin_node_t *node = (ruin_node_t *) tree;
  GHashTable *intermediate_target = g_hash_table_new (g_str_hash, g_str_equal);
  
  select_values_inner 
    (window, window->selection_context, intermediate_target, node->node);

  if (window->render_state->dialect == RUIN_XML_DIALECT_XHTML) 
    {
      char *explicit_style = ruin_scheme_sdom_get_attribute 
	(window, node->node, "style");
      if (explicit_style != NULL)
	{
	  GHashTable *explicit_target = g_hash_table_new 
	    (g_str_hash, g_str_equal);
	  SCM synthetic_context = make_synthetic_selection_context
	    (window, explicit_style);

	  void *intermediate_user_data[2];
	  intermediate_user_data[0] = window;
	  intermediate_user_data[1] = intermediate_target;

	  select_values_inner 
	    (window, synthetic_context, explicit_target, node->node);
	  g_hash_table_foreach (explicit_target, 
				overwrite_with_explicit_style, 
				intermediate_user_data);

	  g_hash_table_destroy (explicit_target);
	}
    }

  user_data[0] = window;
  user_data[1] = target;

  g_hash_table_foreach 
    (intermediate_target, selected_values_to_strings, user_data);
  g_hash_table_destroy (intermediate_target);
}

/* If the element has style based on attribute values, we temporarily append it
   to the head of the author stylesheet for the purposes of doing the cascade
   lookup. If the element has a "style" attribute or child or something, this
   data takes precedence over everything else. */

/* Returns a string giving the effective value for the specified property.  The
   lookup strategy is as follows.  First, the effective style cache is
   consulted.  If a value is not found, the appropriate contextual cache is
   checked.  If the cache is empty, a selection is performed into it against the
   cascade.  If the cache still doesn't contain a value for the property, the
   default value for that property is used, unless there is another way of
   obtaining a value, such as for the border-*-color properties.

   Some special processing is performed against the value before it is inserted
   into the effective style cache: For example, if the value is "inherit," the
   appropriate value will be looked up from a parent element. */

static char *css_lookup 
(ruin_window_t *window, ruin_node_element_t *tree, char *prop) 
{
  SCM backup_sheet = SCM_EOL;
  enum _ruin_css_pseudo_element pseudo_elt_mask = 
    window->render_state->active_pseudo_elements;
  GHashTable **hash_ptr = NULL; 
  char *value = NULL;

  hash_ptr = _ruin_css_get_hash_ptr (tree, pseudo_elt_mask);

  if (*hash_ptr == NULL) 
    {
      *hash_ptr = g_hash_table_new (g_str_hash, g_str_equal);
      
      /* CSS2.1 6.4.4 says that HTML attribute-based style goes into the author
	 sheet, style from other document languages goes into the agent 
	 sheet. */
  
      if (scm_eq_p (tree->inherent_attribute_style, SCM_EOL) != SCM_BOOL_T) 
	{
	  SCM new_sheet = tree->inherent_attribute_style;
	  SCM css_sym = scm_from_locale_symbol ("css");
      
	  if (window->render_state->dialect == RUIN_XML_DIALECT_XHTML) 
	    {
	      backup_sheet = SCM_CADDR (window->cascade);
	      new_sheet = scm_cons
		(css_sym, 
		 scm_cons (SCM_CADR (new_sheet),
			   (scm_eq_p (backup_sheet, SCM_EOL) == SCM_BOOL_T ? 
			    SCM_EOL : SCM_CDR (backup_sheet))));
	      ruin_scheme_scss_set_cascade_author
		(window, window->cascade, new_sheet);
	    } 
	  else 
	    {
	      backup_sheet = SCM_CADR (window->cascade);
	      new_sheet = scm_cons
		(css_sym, 
		 scm_cons(SCM_CADR (new_sheet), 
			  (scm_eq_p (backup_sheet, SCM_EOL) == SCM_BOOL_T ? 
			   SCM_EOL : SCM_CDR (backup_sheet))));
	      ruin_scheme_scss_set_cascade_agent
		(window, window->cascade, new_sheet);
	    }
	}

      select_values (window, tree, *hash_ptr);
      
      if (scm_eq_p (tree->inherent_attribute_style, SCM_EOL) != SCM_BOOL_T) {
	if (window->render_state->dialect == RUIN_XML_DIALECT_XHTML)
	  ruin_scheme_scss_set_cascade_author
	    (window, window->cascade, backup_sheet);
	else ruin_scheme_scss_set_cascade_agent
	       (window, window->cascade, backup_sheet);
      }
    }
  
  value = g_hash_table_lookup (*hash_ptr, prop);

  if (value == NULL)
    value = specify_value (window, *hash_ptr, prop);
  if (value == NULL)
    value = alternate_lookup_strategy (window, tree, prop);
  if ((value == NULL && ruin_scheme_scss_is_inherited (window, prop)) ||
      (value != NULL && strcmp (value, "inherit") == 0))
    {
      ruin_node_t *n = (ruin_node_t *) tree;
      if (n->parent != NULL)
	value = css_lookup (window, (ruin_node_element_t *) n->parent, prop);
    }

  return value;
}

char *ruin_css_lookup_element_style 
(ruin_window_t *w, ruin_node_element_t *e, char *property)
{
  char *value = css_lookup (w, e, property);  
  if (value == NULL)
    {
      enum _ruin_css_pseudo_element pseudo_elt_mask = 
	w->render_state->active_pseudo_elements;
      GHashTable **hash_ptr = _ruin_css_get_hash_ptr (e, pseudo_elt_mask);

      value = ruin_css_initial_value (w, property);
      g_hash_table_insert (*hash_ptr, property, value);
    }
  return value;
}

char *ruin_css_lookup_text_style 
(ruin_window_t *w, ruin_node_text_t *t, char *property)
{
  ruin_node_t *n = (ruin_node_t *) t;
  if (n->parent == NULL)
    return ruin_scheme_scss_get_default_value (w, property);
  else return ruin_css_lookup_element_style 
	 (w, (ruin_node_element_t *) n->parent, property);
}

char *ruin_css_lookup_prototype_style
(ruin_window_t *w, ruin_node_prototype_t *t, char *property)
{
  return (char *) g_hash_table_lookup (t->properties, property);  
}

char *ruin_css_lookup (ruin_window_t *w, ruin_node_t *n, char *property)
{
  switch (n->type)
    {
    case RUIN_PARSE_NODE_TYPE_ELEMENT:
      return ruin_css_lookup_element_style 
	(w, (ruin_node_element_t *) n, property);
    case RUIN_PARSE_NODE_TYPE_PROTOTYPE:
      return ruin_css_lookup_prototype_style
	(w, (ruin_node_prototype_t *) n, property);
    case RUIN_PARSE_NODE_TYPE_TEXT:
      return ruin_css_lookup_text_style (w, (ruin_node_text_t *) n, property);

    default: assert (1 == 0);
    }
}

void ruin_css_add_named_style 
(SCM *list, char *selector, char *prop, char *value)
{
  SCM addition = scm_list_2 
    (scm_from_locale_symbol (selector), scm_list_2 
     (scm_from_locale_symbol (prop), scm_from_locale_string (value)));
  if (scm_eq_p (*list, SCM_EOL) != SCM_BOOL_T)
    scm_append_x (scm_list_2 (SCM_CDR (*list), scm_list_1 (addition)));
  else 
    {
      *list = scm_list_2 (scm_from_locale_symbol ("css"), addition);
      scm_gc_protect_object (*list);
    }
}

void ruin_css_add_style (SCM *list, char *prop, char *value) 
{
  ruin_css_add_named_style (list, "*", prop, value);
}

int ruin_css_is_inherited (ruin_window_t *window, char *prop)
{
  return ruin_scheme_scss_is_inherited (window, prop);
}

char *ruin_css_initial_value (ruin_window_t *window, char *property)
{
  char *value = g_hash_table_lookup (default_style_cache, property);
  if (value == NULL) 
    {
      value = ruin_scheme_scss_get_default_value (window, property);
      g_hash_table_insert (default_style_cache, property, value);
    }

  return value;
}

ruin_length_t ruin_css_parse_size (char *size)
{
  ruin_length_t l = 
    { RUIN_CSS_VALUE_NONE, RUIN_CSS_VALUE_NONE, RUIN_CSS_VALUE_NONE };

  if (strcmp (size, "auto") == 0)
    l.computed = RUIN_CSS_VALUE_AUTO;
  else if (strcmp (size, "thin") == 0 || strcmp (size, "medium") == 0)
    {
      l.computed = 1;
      l.units = RUIN_LAYOUT_UNITS_CHARS;
    } 
  else if ((strcmp(size, "thick") == 0)) 
    {
      l.computed = 2;
      l.units = RUIN_LAYOUT_UNITS_CHARS;
    } 
  else 
    {
      float d = 0;
      int matches = 0;
      char units[3]; units[0] = 0; units[1] = 0; units[2] = 0;
      if (size[0] == '+') size++;
      matches = sscanf (size, "%f%c%c", &d, &units[0], &units[1]);
      if (matches == 3) 
	{
	  l.computed = d;
	  if (strcmp (units, "em") == 0 || strcmp (units, "ex") == 0) 
	    l.units = RUIN_LAYOUT_UNITS_CHARS;
	  else if (strcmp (units, "px") == 0) 
	    l.units = RUIN_LAYOUT_UNITS_PIXELS;
	  else if (strcmp (units, "pt") == 0) l.units = RUIN_LAYOUT_UNITS_PT;
	  else if (strcmp (units, "pc") == 0) l.units = RUIN_LAYOUT_UNITS_PC;
	  else if (strcmp (units, "in") == 0) l.units = RUIN_LAYOUT_UNITS_IN;
	  else if (strcmp (units, "cm") == 0) l.units = RUIN_LAYOUT_UNITS_CM;
	  else if (strcmp (units, "mm") == 0) l.units = RUIN_LAYOUT_UNITS_MM;
	} 
      else if (matches == 2) 
	{
	  l.computed = d;
	  if (strcmp(units, "%") == 0) 
	    l.units = RUIN_LAYOUT_UNITS_PERCENT;
	} 
      else if (matches == 1) 
	{
	  l.computed = d;
	  l.units = RUIN_LAYOUT_UNITS_CHARS;
	}
    }

  return l;
}

ruin_length_t ruin_css_parse_spacing (char *size)
{
  if (strcmp (size, "normal") == 0)
    {
      ruin_length_t l = { RUIN_LAYOUT_UNITS_CHARS, 0, 0 };
      return l;
    }
  else return ruin_css_parse_size (size);
}

void ruin_css_normalize_length 
(ruin_length_t *l, ruin_length_t *t, int font_length, int dpi, 
 int round_to_zero)
{
  if ((l->computed == RUIN_CSS_VALUE_AUTO) ||
      (l->computed == RUIN_CSS_VALUE_NONE))
    return;

  switch (l->units) 
    {
    case RUIN_LAYOUT_UNITS_PIXELS: 
      l->used = floorf (l->computed / font_length); break;
    case RUIN_LAYOUT_UNITS_IN: 
      l->used = floorf ((l->computed * dpi) / font_length); break;
    case RUIN_LAYOUT_UNITS_CM: 
      l->used = floorf ((l->computed * dpi) / (font_length * 2.54)); break;
    case RUIN_LAYOUT_UNITS_MM: 
      l->used = floorf ((l->computed * dpi) / (font_length * 25.4)); break;
    case RUIN_LAYOUT_UNITS_PT: 
      l->used = floorf ((l->computed * dpi) / (72 * font_length)); break;
    case RUIN_LAYOUT_UNITS_PC: 
      l->used = floorf (((l->computed * dpi) / 6) / font_length); break;
    case RUIN_LAYOUT_UNITS_PERCENT: 
      l->used = floorf ((l->computed * t->used) / 100); break;
    default: 
      l->used = l->computed;
    }

  if (!round_to_zero && l->computed != 0 && l->used == 0)
    l->used = 1;

  return;
}

static void register_delegation (char *property, ...)
{
  GList *l = NULL;

  va_list delegate_properties;
  va_start (delegate_properties, property);

  while (TRUE)
    {
      char *delegate_property = va_arg (delegate_properties, char *);
      if (delegate_property == NULL)
	break;
      else l = g_list_append (l, delegate_property);
    }

  va_end (delegate_properties);

  g_hash_table_insert (specification_delegation, property, l);
}

void _ruin_css_init ()
{
  default_style_cache = g_hash_table_new (g_str_hash, g_str_equal);
  specification_delegation = g_hash_table_new (g_str_hash, g_str_equal);

  register_delegation ("background-attachment", "background", NULL);
  register_delegation ("background-color", "background", NULL);
  register_delegation ("background-image", "background", NULL);
  register_delegation ("background-position", "background", NULL);
  register_delegation ("background-target", "background", NULL);

  register_delegation ("border-top-color", "border-top", "border-color", NULL);
  register_delegation ("border-top-style", "border-top", "border-style", NULL);
  register_delegation ("border-top-width", "border-top", "border-width", NULL);

  register_delegation 
    ("border-right-color", "border-right", "border-color", NULL);
  register_delegation 
    ("border-right-style", "border-right", "border-style", NULL);
  register_delegation 
    ("border-right-width", "border-right", "border-width", NULL);

  register_delegation 
    ("border-bottom-color", "border-bottom", "border-color", NULL);
  register_delegation 
    ("border-bottom-style", "border-bottom", "border-style", NULL);
  register_delegation 
    ("border-bottom-width", "border-bottom", "border-width", NULL);

  register_delegation 
    ("border-left-color", "border-left", "border-color", NULL);
  register_delegation 
    ("border-left-style", "border-left", "border-style", NULL);
  register_delegation 
    ("border-left-width", "border-left", "border-width", NULL);

  register_delegation ("border-top", "border", NULL);
  register_delegation ("border-right", "border", NULL);
  register_delegation ("border-bottom", "border", NULL);
  register_delegation ("border-left", "border", NULL);

  register_delegation ("list-style-type", "list-style", NULL);
  register_delegation ("list-style-position", "list-style", NULL);
  register_delegation ("list-style-image", "list-style", NULL);  
}
