/* box.h: Box generation routines for libRUIN
 * Copyright (C) 2011 Julian Graham
 *
 * libRUIN is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef RUIN_BOX_H
#define RUIN_BOX_H

#include <glib.h>

#include "parse.h"
#include "window.h"

enum ruin_layout_unit {
  RUIN_LAYOUT_UNITS_CHARS, RUIN_LAYOUT_UNITS_PERCENT, RUIN_LAYOUT_UNITS_PIXELS,
  RUIN_LAYOUT_UNITS_IN, RUIN_LAYOUT_UNITS_CM, RUIN_LAYOUT_UNITS_MM,
  RUIN_LAYOUT_UNITS_PT, RUIN_LAYOUT_UNITS_PC
};

enum ruin_layout_list_style_type {
  RUIN_LAYOUT_LIST_STYLE_TYPE_DISC, RUIN_LAYOUT_LIST_STYLE_TYPE_CIRCLE, 
  RUIN_LAYOUT_LIST_STYLE_TYPE_SQUARE, RUIN_LAYOUT_LIST_STYLE_TYPE_DECIMAL,
  RUIN_LAYOUT_LIST_STYLE_TYPE_DECIMAL_LEADING_ZERO, 
  RUIN_LAYOUT_LIST_STYLE_TYPE_LOWER_ROMAN, 
  RUIN_LAYOUT_LIST_STYLE_TYPE_UPPER_ROMAN,
  RUIN_LAYOUT_LIST_STYLE_TYPE_LOWER_GREEK,
  RUIN_LAYOUT_LIST_STYLE_TYPE_LOWER_LATIN,
  RUIN_LAYOUT_LIST_STYLE_TYPE_UPPER_LATIN,
  RUIN_LAYOUT_LIST_STYLE_TYPE_ARMENIAN, RUIN_LAYOUT_LIST_STYLE_TYPE_GEORGIAN,
  RUIN_LAYOUT_LIST_STYLE_TYPE_LOWER_ALPHA, 
  RUIN_LAYOUT_LIST_STYLE_TYPE_UPPER_ALPHA, RUIN_LAYOUT_LIST_STYLE_TYPE_NONE
};

enum ruin_layout_box_type {
  RUIN_LAYOUT_BOX_TYPE_BLOCK,
  RUIN_LAYOUT_BOX_TYPE_ANONYMOUS_BLOCK,
  RUIN_LAYOUT_BOX_TYPE_INLINE,
  RUIN_LAYOUT_BOX_TYPE_ANONYMOUS_INLINE,
  RUIN_LAYOUT_BOX_TYPE_LINE,
  RUIN_LAYOUT_BOX_TYPE_MARKER,
  RUIN_LAYOUT_BOX_TYPE_NONE
};

typedef struct _ruin_length_t {
  enum ruin_layout_unit units;
  float computed;
  short used;
} ruin_length_t;

typedef struct _ruin_box_t {
  ruin_window_t *window;
  ruin_node_t *generator;

  enum ruin_layout_box_type type;
  struct _ruin_box_t *parent;
  struct _ruin_box_t *containing_block;

  int visible;
  int level;

  int top;
  int left;

  GList *children;

  ruin_length_t width;
  ruin_length_t height;

  ruin_length_t min_width;
  ruin_length_t max_width;

  ruin_length_t min_height;
  ruin_length_t max_height;

  ruin_length_t margin_top;
  ruin_length_t margin_left;
  ruin_length_t margin_bottom;
  ruin_length_t margin_right;

  ruin_length_t padding_top;
  ruin_length_t padding_left;
  ruin_length_t padding_bottom;
  ruin_length_t padding_right;

  ruin_length_t border_top_width;
  ruin_length_t border_left_width;
  ruin_length_t border_bottom_width;
  ruin_length_t border_right_width;

  ruin_length_t letter_spacing;
  ruin_length_t word_spacing;
  ruin_length_t line_height;

  ruin_length_t text_indent;

} ruin_box_t;

typedef struct _ruin_inline_box_t {
  struct _ruin_box_t base;
  char *content_ptr;
} ruin_inline_box_t;

typedef struct _ruin_marker_box_t {
  struct _ruin_box_t base;
  enum ruin_layout_list_style_type style;
  int ordinal;
  int length;
} ruin_marker_box_t;

GList *ruin_box_generate (ruin_window_t *, ruin_node_t *);

ruin_box_t *ruin_box_new
(enum ruin_layout_box_type, ruin_box_t *, ruin_box_t *, ruin_window_t *, 
 ruin_node_t *);
ruin_inline_box_t *ruin_inline_box_new 
(enum ruin_layout_box_type, ruin_box_t *, ruin_box_t *, ruin_window_t *,
 ruin_node_t *, char *);
ruin_marker_box_t *ruin_marker_box_new
(enum ruin_layout_box_type, ruin_box_t *, ruin_box_t *, ruin_window_t *,
 ruin_node_t *, enum ruin_layout_list_style_type, int, int);

ruin_box_t *ruin_box_clone (ruin_box_t *, short);
void ruin_box_free (ruin_box_t *);

char *ruin_box_css_lookup (ruin_box_t *, char *);

void ruin_box_normalize_width (ruin_length_t *, ruin_box_t *, int);
void ruin_box_normalize_height (ruin_length_t *, ruin_box_t *, int);

#endif /* RUIN_BOX_H */
