/* box.c: Box generation routines for libRUIN
 * Copyright (C) 2011 Julian Graham
 *
 * libRUIN is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include <assert.h>
#include <glib.h>
#include <libguile.h>
#include <math.h>
#include <strings.h>

#include "box.h"
#include "css.h"
#include "parse.h"
#include "window.h"

static GHashTable *display_table = NULL; 
static GHashTable *display_table_cell = NULL;
static GHashTable *display_table_column = NULL;
static GHashTable *display_table_row = NULL;

static GList *box_generate 
(ruin_window_t *, ruin_node_t *, ruin_box_t *, ruin_box_t *);

typedef struct {  
  ruin_node_t *node;
  char *display;
} ruin_node_display_t;

int inline_box_type (enum ruin_layout_box_type t)
{
  return t == RUIN_LAYOUT_BOX_TYPE_INLINE ||
    t == RUIN_LAYOUT_BOX_TYPE_ANONYMOUS_INLINE;
}

int inline_display (char *display)
{
  return strcmp (display, "inline") == 0;
}

static int row_group_box (char *display)
{
  return strcmp (display, "table-row-group") == 0
    || strcmp (display, "table-header-group") == 0
    || strcmp (display, "table-footer-group") == 0;
}

static int proper_table_child (char *display)
{
  return strcmp (display, "table-caption") == 0
    || strcmp (display, "table-column") == 0
    || strcmp (display, "table-column-group") == 0
    || strcmp (display, "table-row") == 0
    || row_group_box (display);
}

static enum ruin_layout_box_type display_to_box_type (char *disp)
{
  if (strcmp (disp, "block") == 0)
    return RUIN_LAYOUT_BOX_TYPE_BLOCK;
  else if (strcmp (disp, "list-item") == 0)
    return RUIN_LAYOUT_BOX_TYPE_BLOCK;
  else if (strcmp (disp, "inline") == 0)
    return RUIN_LAYOUT_BOX_TYPE_INLINE;
  else if (strcmp (disp, "none") == 0)
    return RUIN_LAYOUT_BOX_TYPE_NONE;
  else if (strcmp (disp, "table") == 0 
	   || strcmp (disp, "table-caption") == 0
	   || strcmp (disp, "table-cell") == 0
	   || strcmp (disp, "table-column") == 0
	   || strcmp (disp, "table-column-group") == 0
	   || strcmp (disp, "table-row") == 0
	   || row_group_box (disp))
    return RUIN_LAYOUT_BOX_TYPE_BLOCK;
  
  assert (1 == 0);
}

static enum ruin_layout_box_type get_box_type (ruin_window_t *w, ruin_node_t *n)
{
  switch (n->type)
    {
    case RUIN_PARSE_NODE_TYPE_ELEMENT:
      return display_to_box_type (ruin_css_lookup (w, n, "display"));
    case RUIN_PARSE_NODE_TYPE_PROTOTYPE:
      return RUIN_LAYOUT_BOX_TYPE_ANONYMOUS_BLOCK;
    case RUIN_PARSE_NODE_TYPE_TEXT: 
      return RUIN_LAYOUT_BOX_TYPE_ANONYMOUS_INLINE;
    default: assert (1 == 0);
    }
}

ruin_box_t *ruin_box_new 
(enum ruin_layout_box_type type, ruin_box_t *p, ruin_box_t *cb, 
 ruin_window_t *w, ruin_node_t *t) 
{
  ruin_box_t *box = NULL;
  if (type == RUIN_LAYOUT_BOX_TYPE_INLINE || 
      type == RUIN_LAYOUT_BOX_TYPE_ANONYMOUS_INLINE)
    box = calloc (1, sizeof (ruin_inline_box_t));
  else if (type == RUIN_LAYOUT_BOX_TYPE_MARKER)
    box = calloc (1, sizeof (ruin_marker_box_t));
  else box = calloc (1, sizeof (ruin_box_t));

  if (type == RUIN_LAYOUT_BOX_TYPE_LINE)
    box->height.used = 1;

  box->line_height.used = 1;

  box->type = type;
  box->containing_block = cb;
  box->parent = p;
  box->window = w;
  box->generator = t;

  box->visible = TRUE;

  return box;
}

ruin_inline_box_t *ruin_inline_box_new 
(enum ruin_layout_box_type type, ruin_box_t *p, ruin_box_t *cb, 
 ruin_window_t *w, ruin_node_t *t, char *content_ptr) 
{
  ruin_inline_box_t *box = (ruin_inline_box_t *) 
    ruin_box_new (type, p, cb, w, t);
  box->content_ptr = content_ptr;
  return box;
}

ruin_marker_box_t *ruin_marker_box_new
(enum ruin_layout_box_type type, ruin_box_t *p, ruin_box_t *cb,
 ruin_window_t *w, ruin_node_t *t, enum ruin_layout_list_style_type style,
 int ordinal, int length)
{
  ruin_marker_box_t *box = (ruin_marker_box_t *)
    ruin_box_new (type, p, cb, w, t);

  box->style = style;
  box->ordinal = ordinal;
  box->length = length;
  
  return box;
}

ruin_box_t *ruin_box_clone (ruin_box_t *src, short deep)
{
  ruin_box_t *b = NULL;

  assert (deep == FALSE); /* for now */

  switch (src->type)
    {
    case RUIN_LAYOUT_BOX_TYPE_INLINE:
    case RUIN_LAYOUT_BOX_TYPE_ANONYMOUS_INLINE:
      b = (ruin_box_t *) ruin_inline_box_new
	(src->type, src->parent, src->containing_block, src->window, 
	 src->generator, ((ruin_inline_box_t *) src)->content_ptr);
      break;
    case RUIN_LAYOUT_BOX_TYPE_MARKER:
      b = (ruin_box_t *) ruin_marker_box_new
	(src->type, src->parent, src->containing_block, src->window, 
	 src->generator, ((ruin_marker_box_t *) src)->style,
	 ((ruin_marker_box_t *) src)->ordinal, 
	 ((ruin_marker_box_t *) src)->length);
      break;
    default:
      b = ruin_box_new 
	(src->type, src->parent, src->containing_block, src->window, 
	 src->generator);
    }

  b->visible = src->visible;
  b->level = src->level;

  b->top = src->top;
  b->left = src->left;

  b->width = src->width;
  b->height = src->height;

  b->min_width = src->min_width;
  b->max_width = src->max_width;

  b->min_height = src->min_height;
  b->max_height = src->max_height;

  b->margin_top = src->margin_top;
  b->margin_left = src->margin_left;
  b->margin_bottom = src->margin_bottom;
  b->margin_right = src->margin_right;

  b->padding_top = src->padding_top;
  b->padding_left = src->padding_left;
  b->padding_bottom = src->padding_bottom;
  b->padding_right = src->padding_right;

  b->border_top_width = src->border_top_width;
  b->border_left_width = src->border_left_width;
  b->border_bottom_width = src->border_bottom_width;
  b->border_right_width = src->border_right_width;

  b->letter_spacing = src->letter_spacing;
  b->word_spacing = src->word_spacing;
  b->line_height = src->line_height;

  b->text_indent = src->text_indent;

  return b;
}

void ruin_box_free (ruin_box_t *box)
{
  free (box);
}

static ruin_box_t *generate_block 
(ruin_window_t *w, ruin_node_t *e, ruin_box_t *p, ruin_box_t *cb)
{
  ruin_box_t *b = ruin_box_new (RUIN_LAYOUT_BOX_TYPE_BLOCK, p, cb, w, e);
  if (e->first_child != NULL)
    b->children = box_generate (w, e->first_child, b, b);
  return b;
}

static ruin_box_t *generate_table_caption
(ruin_window_t *w, ruin_node_t *e, ruin_box_t *p, ruin_box_t *cb)
{
  ruin_box_t *b = ruin_box_new (get_box_type (w, e), p, cb, w, e);
  if (e->first_child != NULL)
    b->children = box_generate (w, e->first_child, b, b);
  return b;
}

static ruin_box_t *generate_table_cell
(ruin_window_t *w, ruin_node_t *e, ruin_box_t *p, ruin_box_t *cb)
{
  ruin_box_t *b = ruin_box_new (get_box_type (w, e), p, cb, w, e);
  if (e->first_child != NULL)
    b->children = box_generate (w, e->first_child, b, b);
  return b;
}

typedef ruin_box_t *(*generation_function)
  (ruin_window_t *, ruin_node_t *, ruin_box_t *, ruin_box_t *);

typedef struct _generation_context
{
  ruin_window_t *window;
  ruin_box_t *parent;
  ruin_box_t *containing_block;

  generation_function generator;
} generation_context;

typedef struct _prototype_wrapper_generation_context
{
  generation_context *subcontext;

  int (*display_predicate)(char *);
  GHashTable *prototype_properties;
  GList *wrap_children;
} prototype_wrapper_generation_context;

static void generation_context_init 
(generation_context *context, ruin_window_t *window, ruin_box_t *parent, 
 ruin_box_t *containing_block, generation_function generator)
{
  context->window = window;
  context->parent = parent;
  context->containing_block = containing_block;
  context->generator = generator;
}

static void prototype_wrapper_generation_context_init
(prototype_wrapper_generation_context *context, generation_context *subcontext,
 int (*display_predicate)(char *), GHashTable *prototype_properties,
 GList *wrap_children)
{
  context->subcontext = subcontext;
  context->display_predicate = display_predicate;
  context->prototype_properties = prototype_properties;
  context->wrap_children = wrap_children;
}

static ruin_box_t *generate_child 
(ruin_node_t *e, generation_context *context)
{
  return context->generator 
    (context->window, e, context->parent, context->containing_block);
}

static int is_table_cell (char *display) 
{ 
  return strcmp (display, "table-cell") == 0; 
}

static int is_table_column (char *display) 
{ 
  return strcmp (display, "table-column") == 0; 
}

static int is_table_row (char *display) 
{ 
  return strcmp (display, "table-row") == 0; 
}

static void generate_prototype_child 
(prototype_wrapper_generation_context *context)
{
  ruin_box_t *b = context->subcontext->parent;
  ruin_node_t *prototype = (ruin_node_t *) ruin_node_prototype_new 
    (context->prototype_properties);

  prototype->children = context->wrap_children;
  prototype->first_child = g_list_nth_data (prototype->children, 0);

  b->children = g_list_append 
    (b->children, generate_child (prototype, context->subcontext));
  context->wrap_children = NULL;
}

static void generate_wrapped_child (gpointer data, gpointer user_data)
{
  ruin_node_t *c = (ruin_node_t *) data;

  prototype_wrapper_generation_context *context = 
    (prototype_wrapper_generation_context *) user_data;

  ruin_window_t *w = context->subcontext->window;
  ruin_box_t *b = context->subcontext->parent;

  char *display = ruin_css_lookup (w, c, "display");
  if (context->display_predicate (display)) 
    {
      if (g_list_length (context->wrap_children) > 0)
	generate_prototype_child (context);
      b->children = g_list_append 
	(b->children, generate_child (c, context->subcontext));
    }
  else context->wrap_children = g_list_append (context->wrap_children, c);
  if (c->next_sibling == NULL && g_list_length (context->wrap_children) > 0)
    generate_prototype_child (context);
}				   

static ruin_box_t *generate_table_row
(ruin_window_t *w, ruin_node_t *e, ruin_box_t *p, ruin_box_t *cb)
{
  ruin_box_t *b = ruin_box_new (get_box_type (w, e), p, cb, w, e);
  
  generation_context context;
  prototype_wrapper_generation_context wrapping_context; 

  generation_context_init (&context, w, b, b, generate_table_cell);
  prototype_wrapper_generation_context_init 
    (&wrapping_context, &context, is_table_cell, display_table_cell, NULL);
  
  g_list_foreach (e->children, generate_wrapped_child, &wrapping_context);  
  return b;
}

static ruin_box_t *generate_table_row_group
(ruin_window_t *w, ruin_node_t *e, ruin_box_t *p, ruin_box_t *cb)
{
  ruin_box_t *b = ruin_box_new (get_box_type (w, e), p, cb, w, e);

  generation_context context;
  prototype_wrapper_generation_context wrapping_context;

  generation_context_init (&context, w, b, b, generate_table_row);
  prototype_wrapper_generation_context_init 
    (&wrapping_context, &context, is_table_row, display_table_row, NULL);

  g_list_foreach (e->children, generate_wrapped_child, &wrapping_context);  
  return b;
}

static ruin_box_t *generate_table_column
(ruin_window_t *w, ruin_node_t *e, ruin_box_t *p, ruin_box_t *cb)
{
  ruin_box_t *b = ruin_box_new (get_box_type (w, e), p, cb, w, e);

  generation_context context;
  prototype_wrapper_generation_context wrapping_context;

  generation_context_init (&context, w, b, b, generate_table_cell);
  prototype_wrapper_generation_context_init 
    (&wrapping_context, &context, is_table_cell, display_table_cell, NULL);

  g_list_foreach (e->children, generate_wrapped_child, &wrapping_context);  
  return b;
}

static ruin_box_t *generate_table_column_group
(ruin_window_t *w, ruin_node_t *e, ruin_box_t *p, ruin_box_t *cb)
{
  ruin_box_t *b = ruin_box_new (get_box_type (w, e), p, cb, w, e);

  generation_context context;
  prototype_wrapper_generation_context wrapping_context;

  generation_context_init (&context, w, b, b, generate_table_column);
  prototype_wrapper_generation_context_init 
    (&wrapping_context, &context, is_table_column, display_table_column, NULL);

  g_list_foreach (e->children, generate_wrapped_child, &wrapping_context);  
  return b;
}

static ruin_box_t *generate_table
(ruin_window_t *w, ruin_node_t *e, ruin_box_t *p, ruin_box_t *cb)
{
  ruin_box_t *b = ruin_box_new (get_box_type (w, e), p, cb, w, e);
  ruin_node_t *c = e->first_child;
  while (c != NULL)
    {
      char *display = ruin_css_lookup (w, c, "display");
      if (proper_table_child (display))
	{
	  ruin_box_t *child_box = NULL;
	  if (row_group_box (display))
	    child_box = generate_table_row_group (w, c, b, b);
	  else if (strcmp (display, "table-caption") == 0)
	    child_box = generate_table_caption (w, c, b, b);
	  else if (strcmp (display, "table-row") == 0)
	    child_box = generate_table_row (w, c, b, b);
	  else if (strcmp (display, "table-column") == 0)
	    child_box = generate_table_column (w, c, b, b);
	  else if (strcmp (display, "table-column-group") == 0)
	    child_box = generate_table_column_group (w, c, b, b);

	  b->children = g_list_append (b->children, child_box);
	  c = c->next_sibling;
	}
      else 
	{
	  ruin_node_t *prototype_table_row = (ruin_node_t *) 
	    ruin_node_prototype_new (display_table_row);
	  while (!proper_table_child (display))
	     {
	       if (prototype_table_row->first_child == NULL)
		 prototype_table_row->first_child = c;

	       prototype_table_row->children = 
		 g_list_append (prototype_table_row->children, c);
	       c = c->next_sibling;
	       if (c == NULL)
		 break;
	       display = ruin_css_lookup (w, c, "display");
	     }

	  b->children = g_list_append 
	    (b->children, generate_table_row (w, prototype_table_row, b, b));
	}
    }

  return b;
}

static enum ruin_layout_list_style_type parse_list_style_type (char *style)
{
  if (strcmp (style, "disc") == 0)
    return RUIN_LAYOUT_LIST_STYLE_TYPE_DISC;
  else if (strcmp (style, "circle") == 0)
    return RUIN_LAYOUT_LIST_STYLE_TYPE_CIRCLE;
  else if (strcmp (style, "square") == 0)
    return RUIN_LAYOUT_LIST_STYLE_TYPE_SQUARE;
  else if (strcmp (style, "decimal") == 0)
    return RUIN_LAYOUT_LIST_STYLE_TYPE_DECIMAL;
  else if (strcmp (style, "decimal-leading-zero") == 0)
    return RUIN_LAYOUT_LIST_STYLE_TYPE_DECIMAL_LEADING_ZERO;
  else if (strcmp (style, "lower-roman") == 0)
    return RUIN_LAYOUT_LIST_STYLE_TYPE_LOWER_ROMAN;
  else if (strcmp (style, "upper-roman") == 0)
    return RUIN_LAYOUT_LIST_STYLE_TYPE_UPPER_ROMAN;
  else if (strcmp (style, "lower-greek") == 0)
    return RUIN_LAYOUT_LIST_STYLE_TYPE_LOWER_GREEK;
  else if (strcmp (style, "lower-latin") == 0)
    return RUIN_LAYOUT_LIST_STYLE_TYPE_LOWER_LATIN;
  else if (strcmp (style, "upper-latin") == 0)
    return RUIN_LAYOUT_LIST_STYLE_TYPE_UPPER_LATIN;
  else if (strcmp (style, "armenian") == 0)
    return RUIN_LAYOUT_LIST_STYLE_TYPE_ARMENIAN;
  else if (strcmp (style, "georgian") == 0)
    return RUIN_LAYOUT_LIST_STYLE_TYPE_GEORGIAN;
  else if (strcmp (style, "lower-alpha") == 0)
    return RUIN_LAYOUT_LIST_STYLE_TYPE_LOWER_ALPHA;
  else if (strcmp (style, "upper-alpha") == 0)
    return RUIN_LAYOUT_LIST_STYLE_TYPE_UPPER_ALPHA;
  else if (strcmp (style, "none") == 0)
    return RUIN_LAYOUT_LIST_STYLE_TYPE_NONE;
  else assert (1 == 0);
}

static GList *generate_list_item
(ruin_window_t *w, ruin_node_t *e, ruin_box_t *p, ruin_box_t *cb, int o, int l)
{
  char *position = ruin_css_lookup (w, e, "list-style-position");
  enum ruin_layout_list_style_type s = parse_list_style_type 
    (ruin_css_lookup (w, e, "list-style-type"));
  ruin_box_t *principal_box = ruin_box_new 
    (RUIN_LAYOUT_BOX_TYPE_BLOCK, p, cb, w, e);

  if (strcmp (position, "inside") == 0)
    {
      ruin_box_t *line_box = ruin_box_new 
	(RUIN_LAYOUT_BOX_TYPE_LINE, principal_box, principal_box, w, e);
      ruin_box_t *marker_box = (ruin_box_t *) ruin_marker_box_new 
	(RUIN_LAYOUT_BOX_TYPE_MARKER, line_box, principal_box, w, e, s, o, l);

      line_box->children = g_list_append (NULL, marker_box);
      line_box->children = g_list_concat
	(line_box->children, box_generate 
	 (w, e->first_child, principal_box, principal_box));

      principal_box->children = g_list_append (NULL, line_box);

      return g_list_append (NULL, principal_box);
    }
  else if (strcmp (position, "outside") == 0)
    {
      ruin_box_t *marker_box = (ruin_box_t *) 
	ruin_marker_box_new (RUIN_LAYOUT_BOX_TYPE_MARKER, p, cb, w, e, s, o, l);
      principal_box->children = box_generate 
	(w, e->first_child, principal_box, principal_box);

      return g_list_append (g_list_append (NULL, marker_box), principal_box);
    }
  assert (1 == 0);
}

static ruin_box_t *box_clone (ruin_box_t *b)
{
  if (inline_box_type (b->type))
    return (ruin_box_t *) ruin_inline_box_new 
      (b->type, b->parent, b->containing_block, b->window, b->generator, 
       ((ruin_inline_box_t *) b)->content_ptr);
  return ruin_box_new 
    (b->type, b->parent, b->containing_block, b->window, b->generator);
}

static GList *generate_non_inline
(ruin_window_t *w, ruin_node_t **e, ruin_box_t *p, ruin_box_t *cb)
{
  char *display = ruin_css_lookup (w, *e, "display");
  GList *ret = NULL;

  if (strcmp (display, "block") == 0)
    ret = g_list_append (NULL, generate_block (w, *e, p, cb));
  else if (strcmp (display, "list-item") == 0)
    {
      int len = 0, ord = 0;
      ruin_node_t *e_ptr = *e;
      do
	{
	  len++;

	  *e = (*e)->next_sibling;
	  if (*e == NULL)
	    break;

	  display = ruin_css_lookup (w, *e, "display");	  
	}
      while (strcmp (display, "list-item") == 0);

      *e = e_ptr;
      for (ord = 0; ord < len; ord++)
	{
	  ret = g_list_concat 
	    (ret, generate_list_item (w, *e, p, cb, ord, len));
	  *e = (*e)->next_sibling;
	}

      return ret;
    }
  else if (strcmp (display, "table") == 0)
    ret = g_list_append (NULL, generate_table (w, *e, p, cb));
  else if (proper_table_child (display))
    {
      ruin_node_t *prototype_table = (ruin_node_t *) 
	ruin_node_prototype_new (display_table);
      do 
	{
	  if (prototype_table->first_child == NULL)
	    prototype_table->first_child = *e;
	  prototype_table->children = g_list_append 
	    (prototype_table->children, *e);
	  
	  *e = (*e)->next_sibling;
	  if (*e == NULL)
	    break;
	  
	  display = ruin_css_lookup (w, *e, "display");
	} 
      while (proper_table_child (display));
      return g_list_append (NULL, generate_table (w, prototype_table, p, cb));
    }
  else if (strcmp (display, "none") == 0);
  else assert (1 == 0);

  *e = (*e)->next_sibling;
  return ret;
}

static GList *generate_inline
(ruin_window_t *w, ruin_node_t *e, ruin_box_t *p, ruin_box_t *cb)
{
  enum ruin_layout_box_type box_type = get_box_type (w, e);
  char *content = NULL;

  if (box_type != RUIN_LAYOUT_BOX_TYPE_INLINE &&
      box_type != RUIN_LAYOUT_BOX_TYPE_ANONYMOUS_INLINE)
    return generate_non_inline (w, &e, p, cb);
  if (e->type == RUIN_PARSE_NODE_TYPE_TEXT)
    content = ((ruin_node_text_t *) e)->content;
  
  if (content != NULL)
    {
      ruin_inline_box_t *b = ruin_inline_box_new 
	(RUIN_LAYOUT_BOX_TYPE_INLINE, p, cb, w, e, content);
      return g_list_append (NULL, b);
    }
  else
    {
      ruin_inline_box_t *b = ruin_inline_box_new
	(RUIN_LAYOUT_BOX_TYPE_INLINE, p, cb, w, e, NULL);
      GList *ret = g_list_append (NULL, b);
      GList *child_ptr = e->children;
      int inline_context_latch = 0;

      while (child_ptr != NULL)
	{
	  GList *boxes = generate_inline 
	    (w, (ruin_node_t *) child_ptr->data, (ruin_box_t *) b, cb);
	  GList *box_ptr = boxes;
	  
	  while (box_ptr != NULL)
	    {
	      ruin_box_t *box = box_ptr->data;
	      if (!inline_box_type (box->type))
		{
		  inline_context_latch = 1;
		  ret = g_list_append (ret, box);
		}
	      else
		{
		  if (inline_context_latch)
		    {
		      inline_context_latch = 0;
		      b = (ruin_inline_box_t *) box_clone ((ruin_box_t *) b);
		      ret = g_list_append (ret, b);
		    }
		  box->parent = (ruin_box_t *) b;
		  ((ruin_box_t *) b)->children = 
		    g_list_append (((ruin_box_t *) b)->children, box);
		}
	      box_ptr = box_ptr->next;
	    }

	  child_ptr = child_ptr->next;
	}      

      return ret;
    }
}

static GList *generate_lines 
(ruin_window_t *w, ruin_node_t **e, ruin_box_t *p, 
 ruin_box_t *cb, int non_inline_siblings)
{
  GList *ret = NULL;
  ruin_box_t *current_line = NULL;

  while (*e != NULL)
    {
      GList *children = NULL;
      GList *child_ptr = NULL;
      enum ruin_layout_box_type box_type = get_box_type (w, *e);

      if (box_type != RUIN_LAYOUT_BOX_TYPE_INLINE &&
	  box_type != RUIN_LAYOUT_BOX_TYPE_ANONYMOUS_INLINE)
	{
	  if (g_list_length (ret) == 1 && !non_inline_siblings)
	    {
	      ruin_box_t *b = ruin_box_new 
		(RUIN_LAYOUT_BOX_TYPE_ANONYMOUS_BLOCK, p, cb, w, NULL);
	      ruin_box_t *l = ret->data;
	      
	      l->parent = b;
	      l->containing_block = b;
	      b->children = g_list_append (b->children, l);

	      return g_list_append (NULL, b);
	    }
	  break;
	}

      children = generate_inline (w, *e, p, cb);
      child_ptr = children;
      while (child_ptr != NULL)
	{
	  ruin_box_t *child = (ruin_box_t *) child_ptr->data;
	  if (!inline_box_type (child->type))
	    {
	      current_line = NULL;
	      if (g_list_length (ret) == 1 && !non_inline_siblings)
		{
		  ruin_box_t *b = ruin_box_new 
		    (RUIN_LAYOUT_BOX_TYPE_ANONYMOUS_BLOCK, p, cb, w, NULL);
		  ruin_box_t *l = ret->data;
	      
		  l->parent = b;
		  l->containing_block = b;
		  b->children = g_list_append (b->children, l);

		  ret = g_list_append (NULL, b);
		  non_inline_siblings = TRUE;
		}
	      
	      ret = g_list_append (ret, child);
	    }
	  else
	    {
	      if (current_line == NULL)
		{
		  if (non_inline_siblings)
		    {
		      ruin_box_t *b = ruin_box_new 
			(RUIN_LAYOUT_BOX_TYPE_ANONYMOUS_BLOCK, p, cb, w, NULL);
		      current_line = ruin_box_new
			(RUIN_LAYOUT_BOX_TYPE_LINE, b, b, w, *e);
		      b->children = g_list_append (b->children, current_line);
		      ret = g_list_append (ret, b);
		    }
		  else 
		    {
		      current_line = ruin_box_new
			(RUIN_LAYOUT_BOX_TYPE_LINE, p, cb, w, *e);
		      ret = g_list_append (ret, current_line);
		    }
		}
	      child->parent = current_line;
	      current_line->children = g_list_append 
		(current_line->children, child);
	    }
	  child_ptr = child_ptr->next;
	}
      
      *e = (*e)->next_sibling;
    }
  
  return ret;
}

static GList *box_generate 
(ruin_window_t *w, ruin_node_t *e, ruin_box_t *p, ruin_box_t *cb)
{
  GList *ret = NULL;

  while (e != NULL)
    {
      enum ruin_layout_box_type box_type = get_box_type (w, e);
      if (box_type == RUIN_LAYOUT_BOX_TYPE_INLINE ||
	  box_type == RUIN_LAYOUT_BOX_TYPE_ANONYMOUS_INLINE)
	{
	  ret = g_list_concat 
	    (ret, generate_lines (w, &e, p, cb, g_list_length (ret) > 0));
	  continue;
	}
      else ret = g_list_concat (ret, generate_non_inline (w, &e, p, cb));
    }
  
  return ret;
}

GList *ruin_box_generate (ruin_window_t *w, ruin_node_t *e)
{
  ruin_box_t *viewport_box = ruin_box_new 
    (RUIN_LAYOUT_BOX_TYPE_ANONYMOUS_BLOCK, NULL, NULL, w, NULL);
  assert (e->next_sibling == NULL);
  return box_generate (w, e, NULL, viewport_box);
}

char *ruin_box_css_lookup (ruin_box_t *box, char *prop)
{
  if (box->type == RUIN_LAYOUT_BOX_TYPE_ANONYMOUS_BLOCK
      || box->type == RUIN_LAYOUT_BOX_TYPE_ANONYMOUS_INLINE
      || box->type == RUIN_LAYOUT_BOX_TYPE_LINE)
    {
      if (box->parent == NULL)
	return ruin_css_initial_value (box->window, prop);
      else if (ruin_css_is_inherited (box->window, prop))
	return ruin_box_css_lookup (box->parent, prop);
      else return ruin_css_initial_value (box->window, prop);	
    }
  else if (box->type == RUIN_LAYOUT_BOX_TYPE_BLOCK
	   || box->type == RUIN_LAYOUT_BOX_TYPE_INLINE
	   || box->type == RUIN_LAYOUT_BOX_TYPE_MARKER)
    {
      assert (box->generator != NULL);
      return ruin_css_lookup (box->window, box->generator, prop);
    }
  else assert (1 == 0);
  return NULL;
}

void ruin_box_normalize_width 
(ruin_length_t *l, ruin_box_t *containing_block, int round_to_zero)
{
  ruin_css_normalize_length 
    (l, &containing_block->width, containing_block->window->font_width, 
     containing_block->window->dpi, round_to_zero);
}

void ruin_box_normalize_height 
(ruin_length_t *l, ruin_box_t *containing_block, int round_to_zero)
{
  ruin_css_normalize_length 
    (l, &containing_block->height, containing_block->window->font_height,
     containing_block->window->dpi, round_to_zero);
}

void _ruin_box_init ()
{
  display_table = g_hash_table_new (g_str_hash, g_str_equal);
  display_table_cell = g_hash_table_new (g_str_hash, g_str_equal);
  display_table_column = g_hash_table_new (g_str_hash, g_str_equal);
  display_table_row = g_hash_table_new (g_str_hash, g_str_equal);

  g_hash_table_insert (display_table, "display", "table");
  g_hash_table_insert (display_table_cell, "display", "table-cell");
  g_hash_table_insert (display_table_column, "display", "table-column");
  g_hash_table_insert (display_table_row, "display", "table-row");
}
