/* util.h: Prototypes and declarations for util.c
 * Copyright (C) 2007 Julian Graham
 *
 * libRUIN is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * libRUIN is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with libRUIN; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 */

#ifndef RUIN_UTIL_H
#define RUIN_UTIL_H

#include <pthread.h>

#ifndef TRUE
#define FALSE 0
#define TRUE 1
#endif

#define MAX(x,y) (x > y ? x : y)
#define MIN(x,y) (x < y ? x : y)

/* Hash function from: [JENK97] Bob Jenkins. "Algorithm Alley: Hash Functions".
   Dr. Dobb's Journal. Semptember 1997. */

#define RUIN_UTIL_HASH_SIZE(n) ((int) 1 << (n))
#define RUIN_UTIL_HASH_MASK(n) (RUIN_UTIL_HASH_SIZE (n) - 1)

#define RUIN_UTIL_HASH_MIX(a, b, c) { \
  a -= b; a -= c; a ^= (c >> 13); \
  b -= c; b -= a; b ^= (a << 8); \
  c -= a; c -= b; c ^= (b >> 13); \
  a -= b; a -= c; a ^= (c >> 12);  \
  b -= c; b -= a; b ^= (a << 16); \
  c -= a; c -= b; c ^= (b >> 5); \
  a -= b; a -= c; a ^= (c >> 3);  \
  b -= c; b -= a; b ^= (a << 10); \
  c -= a; c -= b; c ^= (b >> 15); \
}

#define RUIN_UTIL_HASH_DEFAULT_SIZE 128
#define RUIN_UTIL_HASH_GROWTH_FACTOR 2

typedef struct _ruin_util_hash {
  pthread_mutex_t *lock;
  int table_size;
  int num_keys;
  int *map;
  char **keys;
  void **values;
} ruin_util_hash;

typedef struct _ruin_util_list {
  void *data;
  struct _ruin_util_list *next;
  struct _ruin_util_list *prev;
} ruin_util_list;

ruin_util_hash *ruin_util_hash_new();
void ruin_util_hash_insert(ruin_util_hash *hash, char *key, void *value);
void *ruin_util_hash_retrieve(ruin_util_hash *hash, char *key);
void ruin_util_hash_remove(ruin_util_hash *hash, char *key);
char **ruin_util_hash_get_keys(ruin_util_hash *hash, int *key_count);
void ruin_util_hash_clear(ruin_util_hash *);
void ruin_util_hash_free(ruin_util_hash *hash);

ruin_util_list *ruin_util_list_new(void *);
int ruin_util_list_length(ruin_util_list *list);
ruin_util_list *ruin_util_list_push_front(ruin_util_list *, ruin_util_list *);
ruin_util_list *ruin_util_list_append(ruin_util_list *, ruin_util_list *);
ruin_util_list *ruin_util_list_get_ith(ruin_util_list *, int);
void ruin_util_list_free(ruin_util_list *);

ruin_util_hash *ruin_property_string_to_hash(char *properties);
char *ruin_util_int_to_string(int value);
char *ruin_util_ptr_to_string(void *ptr);
char *ruin_util_long_to_string(long);
void *ruin_util_string_to_ptr(char *ptr);
long ruin_util_string_to_long(char *);

long ruin_util_generate_id();

char *ruin_util_arabic_to_roman(int, int);
char *ruin_util_get_parent_directory(char *);

long ruin_util_current_time_millis();

SCM ruin_util_string2scm(char *);

#endif
