/* win-fns.c: Window-management functions for the libRUIN Scheme API
 * Copyright (C) 2007 Julian Graham
 *
 * libRUIN is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * libRUIN is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with libRUIN; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 */

#include <libguile.h>

#include "../css.h"
#include "../layout.h"
#include "../util.h"
#include "../window.h"
#include "win-fns.h"

extern ruin_windows_t *_ruin_windows;

SCM ruin_scm_api_window_focus_next(SCM selt) {
  ruin_element_t *elt = ruin_window_lookup_scm(selt);
  ruin_window_t *w = NULL;
  ruin_util_list *tab_order_ptr = NULL;

  int i = 0, len = 0, call_next = FALSE;

  if (elt == NULL)
    return SCM_BOOL_F;
  
  w = elt->parent_window;
  if (w->focused == NULL)
    return SCM_BOOL_F;

  len = ruin_util_list_length(w->tab_order);

  if (len == 0)
    return SCM_BOOL_F;

  tab_order_ptr = w->tab_order;
  for (; i < len; i++) {
    ruin_element_t *t = (ruin_element_t *) tab_order_ptr->data;
    if (call_next)
      return ruin_scm_api_window_focus(t->element);
    else if (t == w->focused)
      call_next = TRUE;
    tab_order_ptr = tab_order_ptr->next;
  }
  return ruin_scm_api_window_focus
    (((ruin_element_t *) w->tab_order->data)->element);
}

SCM ruin_scm_api_window_focus_prev(SCM selt) {
  ruin_element_t *elt = ruin_window_lookup_scm(selt);
  ruin_window_t *w = NULL;
  ruin_util_list *tab_order_ptr = NULL;

  int i = 0, len = 0;

  if (elt == NULL)
    return SCM_BOOL_F;
  
  w = elt->parent_window;
  if (w->focused == NULL)
    return SCM_BOOL_F;

  len = ruin_util_list_length(w->tab_order);

  if (len == 0)
    return SCM_BOOL_F;

  tab_order_ptr = w->tab_order;
  for (; i < len; i++) {
    ruin_element_t *t = (ruin_element_t *) tab_order_ptr->data;
    if (t == w->focused) {
      if (elt != NULL)
	return ruin_scm_api_window_focus(t->element);
      else return ruin_scm_api_window_focus
	     (((ruin_element_t *) tab_order_ptr->prev->data)->element);
    }
    else elt = t; 
    tab_order_ptr = tab_order_ptr->next;
  }
  return SCM_BOOL_F;
}

SCM ruin_scm_api_window_focus(SCM selt) {
  ruin_element_t *elt = ruin_window_lookup_scm(selt);
  ruin_window_t *w = NULL;

  if (elt == NULL)
    return SCM_BOOL_F;
  
  w = elt->parent_window;
  
  /* Are we already focused? */
  
  if (w->focused == elt)
    return SCM_BOOL_T;
  else {
    
    /* Are we in the list of focusable elements? */
    
    int found_it = FALSE;
    int tab_length = ruin_util_list_length(w->tab_order), i = 0;
    ruin_util_list *tab_order_ptr = w->tab_order;
    
    for (; i < tab_length; i++) {
      ruin_element_t *tmp = tab_order_ptr->data;
      if (tmp == elt) {
	found_it = TRUE;
	break;
      }	  
      tab_order_ptr = tab_order_ptr->next;
    }
    
    if (found_it) {
      ruin_element_t *old = w->focused;
      w->focused = elt;
      if (old != NULL) {
	ruin_util_log(w, "dispatching event sdom:event-dom-focus-out "
		      "on element '%s'\n", old->id);
	scm_call_4(scm_c_eval_string("sdom:dispatch-event"),
		   w->focused->element, 
		   scm_str2symbol("sdom:event-dom-focus-out"), 
		   SCM_EOL, SCM_EOL);
      }
      ruin_util_log(w, "dispatching event sdom:event-dom-focus-in on "
		    "element '%s'\n", w->focused->id);
      scm_call_4(scm_c_eval_string("sdom:dispatch-event"),
		 w->focused->element, 
		 scm_str2symbol("sdom:event-dom-focus-in"), 
		 SCM_EOL, SCM_EOL);

      ruin_css_clear_style_cache(w->focused);
      ruin_scm_api_window_render(w->focused->element);
    }
  }
  return SCM_BOOL_F;
}

SCM ruin_scm_api_window_render(SCM selt) {
  ruin_element_t *elt = ruin_window_lookup_scm(selt);

  if (elt == NULL)
    return SCM_BOOL_F;
  
  elt->parent_window->render_requested = TRUE;
  return SCM_BOOL_T;
}
