/* debug.c: Debug functions for libRUIN
 * Copyright (C) 2006 Julian Graham
 *
 * libRUIN is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * libRUIN is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with libRUIN; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 */

#include <libguile.h>
#include <string.h>

#include "css.h"
#include "debug.h"
#include "layout.h"
#include "scheme.h"
#include "util.h"

static SCM scd_p = SCM_EOL;
static SCM scn_p = SCM_EOL;
static SCM ssax_p = SCM_EOL;
static SCM snte_n = SCM_EOL;
static SCM sacx_p = SCM_EOL;

static SCM pt_s = SCM_EOL;
static SCM pl_s = SCM_EOL;
static SCM pb_s = SCM_EOL;
static SCM pr_s = SCM_EOL;

static SCM bt_s = SCM_EOL;
static SCM bl_s = SCM_EOL;
static SCM bb_s = SCM_EOL;
static SCM br_s = SCM_EOL;

static SCM mt_s = SCM_EOL;
static SCM ml_s = SCM_EOL;
static SCM mb_s = SCM_EOL;
static SCM mr_s = SCM_EOL;

static SCM w_s = SCM_EOL;

static SCM get_str(int val) {
  return scm_number_to_string(scm_from_int32(val), scm_from_int32(10));
}

static SCM add_node(SCM doc, ruin_element_t *tree) {
  SCM name = SCM_EOL;
  SCM node = SCM_EOL;

  ruin_element_t *child_ptr = NULL;

  if (tree == NULL)
    return SCM_EOL;
  { char *cname = ruin_css_lookup(tree, "display");
    if (strcmp(cname, "none") == 0)
      return SCM_EOL;
    name = ruin_util_string2scm(cname);
  }
  node = scm_apply_3(scn_p, doc, snte_n, name, SCM_EOL);
  if (scm_string_p(tree->element) == SCM_BOOL_T)
    scm_apply_3(ssax_p, node,
		ruin_util_string2scm("generated"),
		ruin_util_string2scm("true"),
		SCM_EOL);

  scm_apply_3(ssax_p, node, pt_s, get_str(tree->padding_top.used), SCM_EOL);
  scm_apply_3(ssax_p, node, pl_s, get_str(tree->padding_left.used), SCM_EOL);
  scm_apply_3(ssax_p, node, pb_s, get_str(tree->padding_bottom.used), SCM_EOL);
  scm_apply_3(ssax_p, node, pr_s, get_str(tree->padding_right.used), SCM_EOL);

  scm_apply_3(ssax_p, node, mt_s, get_str(tree->margin_top.used), SCM_EOL);
  scm_apply_3(ssax_p, node, ml_s, get_str(tree->margin_left.used), SCM_EOL);
  scm_apply_3(ssax_p, node, mb_s, get_str(tree->margin_bottom.used), SCM_EOL);
  scm_apply_3(ssax_p, node, mr_s, get_str(tree->margin_right.used), SCM_EOL);

  scm_apply_3
    (ssax_p, node, bt_s, get_str(tree->border_top_width.used), SCM_EOL);
  scm_apply_3
    (ssax_p, node, bl_s, get_str(tree->border_left_width.used), SCM_EOL);
  scm_apply_3
    (ssax_p, node, bb_s, get_str(tree->border_bottom_width.used), SCM_EOL);
  scm_apply_3
    (ssax_p, node, br_s, get_str(tree->border_right_width.used), SCM_EOL);

  scm_apply_3(ssax_p, node, w_s, get_str(tree->width.used), SCM_EOL);

  child_ptr = tree->first_child;
  while (child_ptr != NULL) {
    SCM child = add_node(doc, child_ptr);
    if (child != SCM_EOL)
      scm_apply_2(sacx_p, node, child, SCM_EOL);
    child_ptr = child_ptr->next_sibling;
  }
  return node;
}

SCM ruin_debug_dump_layout_tree(ruin_element_t *tree) {
  SCM doc = SCM_EOL;
  SCM doc_elem = SCM_EOL;
  if (scd_p == SCM_EOL) {
    scd_p = scm_c_eval_string("sdom:create-document");
    scn_p = scm_c_eval_string("sdom:create-node");
    ssax_p = scm_c_eval_string("sdom:set-attribute!");
    snte_n = scm_c_eval_string("sdom:node-type-element");
    sacx_p = scm_c_eval_string("sdom:append-child!");

    pt_s = scm_mem2string("padding-top", 11);
    pl_s = scm_mem2string("padding-left", 12);
    pb_s = scm_mem2string("padding-bottom", 14);
    pr_s = scm_mem2string("padding-right", 13);

    bt_s = scm_mem2string("border-top", 10);
    bl_s = scm_mem2string("border-left", 11);
    bb_s = scm_mem2string("border-bottom", 13);
    br_s = scm_mem2string("border-right", 12);

    mt_s = scm_mem2string("margin-top", 10);
    ml_s = scm_mem2string("margin-left", 11);
    mb_s = scm_mem2string("margin-bottom", 13);
    mr_s = scm_mem2string("margin-right", 12);

    w_s = scm_mem2string("width", 5);

    scm_gc_protect_object(scd_p);
    scm_gc_protect_object(scn_p);
    scm_gc_protect_object(ssax_p);
    scm_gc_protect_object(snte_n);
    scm_gc_protect_object(sacx_p);
    scm_gc_protect_object(pt_s);
    scm_gc_protect_object(pl_s);
    scm_gc_protect_object(pb_s);
    scm_gc_protect_object(pr_s);
    scm_gc_protect_object(bt_s);
    scm_gc_protect_object(bl_s);
    scm_gc_protect_object(bb_s);
    scm_gc_protect_object(br_s);
    scm_gc_protect_object(mt_s);
    scm_gc_protect_object(ml_s);
    scm_gc_protect_object(mb_s);
    scm_gc_protect_object(mr_s);
    scm_gc_protect_object(w_s);
  }
  doc = scm_apply_3(scd_p, SCM_EOL, SCM_EOL, SCM_EOL, SCM_EOL);
  doc_elem = add_node(doc, tree);
  (void) scm_apply_2(sacx_p, doc, doc_elem, SCM_EOL);
  return doc;
}
