/* window.c: Window-management routines for libRUIN
 * Copyright (C) 2005 Julian Graham
 *
 * libRUIN is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * libRUIN is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with libRUIN; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 */

#include <stdlib.h>

#include "layout.h"
#include "util.h"
#include "window.h"

extern ruin_windows_t *_ruin_windows;

ruin_windows_t *ruin_windows_new() {
  extern char xulstyle[];
  extern char xhtmlstyle[];

  ruin_windows_t *t = calloc(1, sizeof(ruin_windows_t));
  t->windows = ruin_util_hash_new();
  t->xul_agent_css = 
    scm_call_1(scm_c_eval_string("scss:css->scss"),
	       scm_open_input_string(scm_makfrom0str(xulstyle)));
  scm_gc_protect_object(t->xul_agent_css);
  t->xhtml_agent_css =
    scm_call_1(scm_c_eval_string("scss:css->scss"),
	       scm_open_input_string(scm_makfrom0str(xhtmlstyle)));
  scm_gc_protect_object(t->xhtml_agent_css);
  return t;
}

void ruin_windows_free(ruin_windows_t *t) {
  ruin_util_hash_free(t->windows);
  scm_gc_unprotect_object(t->xul_agent_css);
  scm_gc_unprotect_object(t->xhtml_agent_css);
  free(t);
  return;
}

ruin_window_t *ruin_window_new(WINDOW *w, FILE *f) {
  ruin_window_t *t = calloc(1, sizeof(ruin_window_t));
  t->window = w;
  t->log = f;

  t->internal_id = ruin_util_generate_id();
  ruin_util_hash_insert(_ruin_windows->windows, 
			ruin_util_long_to_string(t->internal_id), 
			ruin_util_ptr_to_string((void *) t));
  
  t->ids = ruin_util_hash_new();
  t->internal_ids = ruin_util_hash_new();

  t->scm_hash = scm_call_1(scm_c_eval_string("make-hash-table"),
			   scm_int2num(RUIN_WINDOW_SCM_HASH_SIZE));
  scm_gc_protect_object(t->scm_hash);

  t->tab_order = ruin_util_list_new();

  t->font_height = RUIN_WINDOW_DEFAULT_FONT_HEIGHT;
  t->font_width = RUIN_WINDOW_DEFAULT_FONT_WIDTH;
  t->dpi = RUIN_WINDOW_DEFAULT_DPI;

  return t;
}

void ruin_window_free(ruin_window_t *t) {
  ruin_util_hash_remove(_ruin_windows->windows, 
			ruin_util_long_to_string(t->internal_id));
  ruin_util_hash_free(t->ids);
  ruin_util_hash_free(t->internal_ids);
  scm_gc_unprotect_object(t->scm_hash);
  ruin_util_list_free(t->tab_order);
  free(t);
  return;
}

ruin_window_t *ruin_window_lookup_scm(SCM elt) {
  SCM doc = SCM_EOL;
  SCM type = scm_call_1(scm_c_eval_string("sdom:node-type"), elt);

  char **keys = NULL;
  int num_keys, i;
  
  if (scm_eqv_p(type, scm_c_eval_string("sdom:node-type-document")) == 
      SCM_BOOL_T)
    doc = elt;
  else doc = scm_call_2(scm_c_eval_string("sdom:get-dom-property"), elt,
			scm_makfrom0str("sdom:owner-document"));

  keys = ruin_util_hash_get_keys(_ruin_windows->windows, &num_keys);
  for (i = 0; i < num_keys; i++) {
    ruin_window_t *win = (ruin_window_t *)
      (ruin_util_string_to_ptr
       (ruin_util_hash_retrieve(_ruin_windows->windows, keys[i])));
    if (scm_eq_p(win->top->doc, doc) == SCM_BOOL_T) return win;
  }
  return NULL;  
}

