/* layout.h: Prototypes and declarations for layout.c
 * Copyright (C) 2005 Julian Graham
 *
 * libRUIN is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * libRUIN is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with libRUIN; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA 
 */

#ifndef RUIN_LAYOUT_H
#define RUIN_LAYOUT_H

#include <libguile.h>

#include "load.h"
#include "util.h"
#include "window.h"

#define RUIN_LAYOUT_VALUE_AUTO		-1
#define RUIN_LAYOUT_VALUE_INHERIT	-2
#define RUIN_LAYOUT_VALUE_NONE          -3

enum ruin_layout_unit {
  RUIN_LAYOUT_UNITS_CHARS, RUIN_LAYOUT_UNITS_PERCENT, RUIN_LAYOUT_UNITS_PIXELS,
  RUIN_LAYOUT_UNITS_IN, RUIN_LAYOUT_UNITS_CM, RUIN_LAYOUT_UNITS_MM,
  RUIN_LAYOUT_UNITS_PT, RUIN_LAYOUT_UNITS_PC
};

enum ruin_layout_border_style {
  RUIN_LAYOUT_BORDER_STYLE_NONE, RUIN_LAYOUT_BORDER_STYLE_SOLID, 
  RUIN_LAYOUT_BORDER_STYLE_DOUBLE, RUIN_LAYOUT_BORDER_STYLE_DASHED
};

enum ruin_layout_orientation {
  RUIN_LAYOUT_ORIENT_VERTICAL, RUIN_LAYOUT_ORIENT_HORIZONTAL
};

enum ruin_layout_horizontal_alignment {
  RUIN_LAYOUT_HORIZONTAL_ALIGNMENT_LEFT, 
  RUIN_LAYOUT_HORIZONTAL_ALIGNMENT_CENTER,
  RUIN_LAYOUT_HORIZONTAL_ALIGNMENT_RIGHT
};

enum ruin_layout_vertical_alignment {
  RUIN_LAYOUT_VERTICAL_ALIGNMENT_TOP, RUIN_LAYOUT_VERTICAL_ALIGNMENT_CENTER,
  RUIN_LAYOUT_VERTICAL_ALIGNMENT_BOTTOM
};

enum ruin_layout_crop_side {
  RUIN_LAYOUT_CROP_SIDE_NONE, RUIN_LAYOUT_CROP_SIDE_START, 
  RUIN_LAYOUT_CROP_SIDE_END, RUIN_LAYOUT_CROP_SIDE_BOTH
};

enum ruin_extra_content {
  RUIN_LAYOUT_EXTRA_CONTENT_NONE, RUIN_LAYOUT_EXTRA_CONTENT_CHECKBOX,
  RUIN_LAYOUT_EXTRA_CONTENT_RADIO, RUIN_LAYOUT_EXTRA_CONTENT_COLORPICKER
};

enum ruin_layout_foreground_color {
  RUIN_LAYOUT_FG_COLOR_BLACK = 0, RUIN_LAYOUT_FG_COLOR_RED, 
  RUIN_LAYOUT_FG_COLOR_GREEN, RUIN_LAYOUT_FG_COLOR_YELLOW, 
  RUIN_LAYOUT_FG_COLOR_BLUE, RUIN_LAYOUT_FG_COLOR_MAGENTA, 
  RUIN_LAYOUT_FG_COLOR_CYAN, RUIN_LAYOUT_FG_COLOR_WHITE, 
  RUIN_LAYOUT_FG_COLOR_BR_BLACK, RUIN_LAYOUT_FG_COLOR_BR_RED, 
  RUIN_LAYOUT_FG_COLOR_BR_GREEN, RUIN_LAYOUT_FG_COLOR_BR_YELLOW, 
  RUIN_LAYOUT_FG_COLOR_BR_BLUE, RUIN_LAYOUT_FG_COLOR_BR_MAGENTA, 
  RUIN_LAYOUT_FG_COLOR_BR_CYAN, RUIN_LAYOUT_FG_COLOR_BR_WHITE
};

enum ruin_layout_background_color {
  RUIN_LAYOUT_BG_COLOR_BLACK = 0, RUIN_LAYOUT_BG_COLOR_RED, 
  RUIN_LAYOUT_BG_COLOR_GREEN, RUIN_LAYOUT_BG_COLOR_YELLOW, 
  RUIN_LAYOUT_BG_COLOR_BLUE, RUIN_LAYOUT_BG_COLOR_MAGENTA, 
  RUIN_LAYOUT_BG_COLOR_CYAN, RUIN_LAYOUT_BG_COLOR_WHITE
};

enum ruin_layout_editable_type {
  RUIN_LAYOUT_EDITABLE_TYPE_NONE, RUIN_LAYOUT_EDITABLE_TYPE_SINGLE_LINE,
  RUIN_LAYOUT_EDITABLE_TYPE_MULTI_LINE
};

enum ruin_layout_display {
  RUIN_LAYOUT_DISPLAY_BLOCK = 0x1, RUIN_LAYOUT_DISPLAY_INLINE_BLOCK = 0x2,
  RUIN_LAYOUT_DISPLAY_INLINE = 0x4, RUIN_LAYOUT_DISPLAY_LIST_ITEM = 0x8,
  RUIN_LAYOUT_DISPLAY_NONE = 0x10, RUIN_LAYOUT_DISPLAY_RUN_IN = 0x20, 
  RUIN_LAYOUT_DISPLAY_TABLE = 0x40, RUIN_LAYOUT_DISPLAY_INLINE_TABLE = 0x80,
  RUIN_LAYOUT_DISPLAY_TABLE_ROW_GROUP = 0x100, 
  RUIN_LAYOUT_DISPLAY_TABLE_COLUMN = 0x200,
  RUIN_LAYOUT_DISPLAY_TABLE_COLUMN_GROUP = 0x400, 
  RUIN_LAYOUT_DISPLAY_TABLE_HEADER_GROUP = 0x800, 
  RUIN_LAYOUT_DISPLAY_TABLE_FOOTER_GROUP = 0x1000,
  RUIN_LAYOUT_DISPLAY_TABLE_ROW = 0x2000, 
  RUIN_LAYOUT_DISPLAY_TABLE_CELL = 0x4000,
  RUIN_LAYOUT_DISPLAY_TABLE_CAPTION = 0x8000
};

enum ruin_layout_direction {
  RUIN_LAYOUT_DIRECTION_LTR, RUIN_LAYOUT_DIRECTION_RTL
};

typedef struct _ruin_length_t {
  enum ruin_layout_unit units;
  float computed;
  short used;
} ruin_length_t;


/* NOTES: We refer to the plain values we get from the "cascade" as the 
   "computed values" -- these values can be absolute, relative, etc.  They
   haven't been normalized.  After we normalize them, we refer to them as 
   "used" values.  Finally, if the used values are found to violate any
   constraints on the size of their bounding boxes, we may need to adjust them
   so that they come into compliance -- the results are referred to as
   "resolved" values. */

typedef struct _ruin_element_t {
  long internal_id;

  enum ruin_xml_dialect dialect;

  SCM doc;
  SCM element;
  SCM attributes;

  struct _ruin_element_t *caption;
  struct _ruin_element_t *first_child;
  struct _ruin_element_t *parent;
  struct _ruin_element_t *next_sibling;
  struct _ruin_element_t *prev_sibling;

  struct _ruin_window_t *parent_window;

  struct _ruin_element_t *next_tab;
  struct _ruin_element_t *prev_tab;

  SCM cascade;
  SCM inherent_attribute_style;
  SCM additional_attribute_style;
  ruin_util_list *current_pseudo_elements;

  ruin_util_hash *style_cache;

  ruin_util_hash *capture_events;
  ruin_util_hash *target_events;
  ruin_util_hash *bubble_events;
  
  char *id;
  ruin_util_hash *ids;

  enum ruin_extra_content extra_content;

  char *content;
  int visible;

  int top;
  int left;
  ruin_length_t width;
  ruin_length_t height;

  ruin_length_t max_height;
  ruin_length_t max_width;
  ruin_length_t min_height;
  ruin_length_t min_width;

  enum ruin_layout_foreground_color color;
  enum ruin_layout_background_color background_color;

  ruin_length_t border_top_width;
  enum ruin_layout_foreground_color border_top_color;
  ruin_length_t border_left_width;
  enum ruin_layout_foreground_color border_left_color;
  ruin_length_t border_bottom_width;
  enum ruin_layout_foreground_color border_bottom_color;
  ruin_length_t border_right_width;
  enum ruin_layout_foreground_color border_right_color;

  ruin_length_t outline_width;
  ruin_length_t letter_spacing;
  ruin_length_t word_spacing;

  ruin_length_t padding_top;
  ruin_length_t padding_left;
  ruin_length_t padding_bottom;
  ruin_length_t padding_right;

  ruin_length_t margin_top;
  ruin_length_t margin_left;
  ruin_length_t margin_bottom;
  ruin_length_t margin_right;

  ruin_length_t text_indent;

  int selected;

  int submit_key;

  int focusable;
  int tab_index;

  int last_line_length;
  int first_line_start;
  
  enum ruin_layout_editable_type editable;
} ruin_element_t;

ruin_element_t *ruin_element_new();
void ruin_element_free(ruin_element_t *t);

typedef struct _ruin_layout_size_t {
  int first_line;
  int height;
  int width;
  int last_line;
} ruin_layout_size_t;

ruin_layout_size_t *ruin_layout_size_new(int, int);
void ruin_layout_append_child(ruin_element_t *, ruin_element_t *);
void ruin_layout_size_free(ruin_layout_size_t *t);

void ruin_layout_add_style(SCM *, char *, char *);

/* Shouldn't have to expose this... */
void ruin_layout_normalize_lengths(ruin_element_t *, ruin_util_list *, int);
ruin_element_t *ruin_layout_find_containing_block(ruin_util_list *, int);

/* These are the layout management functions. */

ruin_layout_size_t ruin_layout_size_tree(ruin_element_t *t, ruin_util_list *,
					 int, int);

void ruin_generate_tree_parse_attrs(ruin_element_t *t);

#endif
