/*===========================================================================*/
/*
 * This file is part of libpersist - a c++ library for object persistence
 *
 * Copyright (C) 2006  Elaine Tsiang YueLien
 *
 * libpersist is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to
 * Free Software Foundation, Inc.
 * 51 Franklin Street, Fifth Floor
 * Boston, MA  02110-1301, USA
 *
 *===========================================================================*/
/*                                                                           */
/* Examples::Sync Object-derived thread synchronizer implementation          */
/*                                                                           */
/*===========================================================================*/
#ifdef __GNUG__
#pragma implementation
#endif


#include	<Sync.H>
#include	<persist/GetPutDelTemplates.H>


namespace	Persistence
{
  using namespace Examples;

  //
  // instantiate template functions
  //
  template
  Sync *	get(
		    const char *	name
		    );

  template
  Sync *	get(
		    const Id		id
		    );
  template
  Status	put(
		    Sync &	cla
		    );
  template
  Status	del(
		    Sync & 	cla
		    );
  template
  Sync *	first();

  template
  Sync *	next(const Sync	&);

}


namespace	Examples
{
  using namespace Persistence;

  Id		Sync::ClassId(ExampleSync);
  Name		Sync::ClassName("Sync");
  LongName	Sync::ClassDecl(SyncClassDecl);
  LongName	Sync::ClassImpl("Sync.C $Id$ keyword substitution from version control");

  //---------------------------------------------------------------------------
  Sync::Sync(
	     const	char *	nm
	     )
    : Obj(nm, ClassId)
      , OwnsFilts(0)		// default number of filters
      , syncing(false)
      , nSyncing(0)
  {
    init(
	 ClassId		// Class ClassId
	 ,ClassName.cstr()
	 ,ClassDecl.cstr()
	 ,ClassImpl.cstr()	// Class Class Impl
	 ,0			// don't keep track of Sync objects
	 );


    mrefFilts mr(
		 *this
		 ,objName + ".filters"
		 );
    filts() = mr;

  }

  //------------------------------------------------------------------------------
  void  Sync::insert(
		     Filter &		o
		     )
  {
    this->OwnsFilts::insert_filt(o);
    nToSync = size_filts();	// quicker for runtime
  }

  void	Sync::remove(
		     Filter &		o
		     )
  {
    this->OwnsFilts::remove_filt(o);
    nToSync = size_filts();

  }

  //------------------------------------------------------------------------------
  void	Sync::dump(
		   ostream &	out
		   ) const
  {
    this->Object::dump(out);

    out << "Number of filters to sync = " << nToSync << std::endl;
    out << "Sync filters : " << std::endl;
    this->OwnsFilts::dump_filts(out);
  }
  //------------------------------------------------------------------------------
  Status
  Sync::put()
  {
    Status s(Persistence::put<Sync>(*this));

    if ( s.succeeded() )
      return(s);
    else
      {
	Status sta("Sync"
		   ,"put"
		   ,cname()
		   );
	sta.os() << " failed" << std::endl;
	sta &= s;
	return(sta);
      }
  }

  Status
  Sync::del()
  {
    Status s(Persistence::del<Sync>(*this));
    
    if ( s.succeeded() )
      return(s);
    else
      {
	Status sta("Sync"
		   ,"del"
		   ,cname()
		   );
	sta.os() << " failed" << std::endl;
	sta &= s;
	return(sta);
      }
  }

  //------------------------------------------------------------------------------
  void
  Sync::sync()
  {
    pthread_mutex_lock(&syncLock);
    {
      if ( syncing )
	{
	  // this thread has reached the next sync point
	  // but somebody still hasn't heard
	  // the end of the story of the last sync point
	  pthread_cond_wait(&endSyncCond, &syncLock);
	}

      // everyone is going for the next sync point
      ++nSyncing;

      if ( nSyncing < nToSync )
	{
	  // not everyone has reached sync point
	  // hurry up and wait
	  pthread_cond_wait(&beginSyncCond, &syncLock);
	}
      else
	{
	  // last thread to reach sync point
	  // let everybody know we are now syncing
	  syncing = true;
	  pthread_cond_broadcast(&beginSyncCond);
	}

      // this thread knows
      // everyone has reached sync point
      --nSyncing;
      
      if ( 0 >= nSyncing )
	{
	  // last thread to hear
	  // sync point has been reached
	  // let everybody know everybody has heard
	  // that we are done syncing
	  syncing = false;
	  pthread_cond_broadcast(&endSyncCond);
	}
      /* else
	 not everyone has heard
	 that the sync point has been reached
	 but this thread is off to do its thing
      */
    }
    pthread_mutex_unlock(&syncLock);
  }
}

