/*===========================================================================*/
/*
 * This file is part of libpersist - a c++ library for object persistence
 *
 * Copyright (C) 2006  Elaine Tsiang YueLien
 *
 * libpersist is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to
 * Free Software Foundation, Inc.
 * 51 Franklin Street, Fifth Floor
 * Boston, MA  02110-1301, USA
 *
 *===========================================================================*/
/*                                                                           */
/* Examples::Que - interface                                                 */
/*                                                                           */
/*===========================================================================*/
#ifndef	ExampleQue
#define	ExampleQue 251

#define	QueClassDecl  "Que.H $Id$ keyword substitution from version control"

#ifdef __GNUG__
#pragma interface
#endif

#include	<valarray>

namespace	Examples
{
  class Que;
}

#include	<Sync.H>

namespace	Examples
{
  using std::slice;

  typedef std::valarray<float>		Array;
  typedef std::slice_array<float>	SubArray;

  class Que;
  typedef tref<Que, Filter>	trefFilt;
  typedef ref<Que *>		refQue;
  typedef ref<Filter *>		refFilt;

  class Que : public Object, public DtorCommand, public ClassInfo
  {
    friend class ClassInfo;
    // class data and methods
  private:
    // common to all persistent classes
    static
    Id			ClassId;
    static
    Name		ClassName;
    static
    LongName		ClassDecl;
    static
    LongName		ClassImpl;
  public:
    static
    Id    		classId()
    {
      return(ClassId);
    }
    
    static
    const char *	className()
    {
      return(ClassName.cstr());
    }
    
    static
    const char *	classDecl()
    {
      return(ClassDecl.cstr());
    }

    static
    const char *	classImpl()
    {
      return(ClassImpl.cstr());
    }

  public:
    Que()
      : Obj()
	, inputFilter()
	, outputFilter()
    {}

    Que(
	const char *	nm
	,size_t		width
	,size_t		len
	);

    ~Que()
    {}
      

  private:
    trefFilt		inputFilter;	// relocate to be with the writer of this que
    refFilt		outputFilter;	// may not be with the reader of this que

  public:
    Filter *	input() const
    {
      return(inputFilter.operator->());
    }

    Filter *	output() const
    {
      return(outputFilter.operator->());
    }
    
    void	input(Id i)
    {
      inputFilter = trefFilt(this->id(), i);
    }
   
    void	output(Id i)
    {
      outputFilter = refFilt(i);
    }
   

    Status	start();
    Status	finish();

  public:
    struct itereader
    {
      Que *	area;
      size_t	frameWidth;
      size_t	offset;

      itereader() throw()
      {}

      itereader(Que & a) throw()
	: area(&a)
	  , frameWidth(a.width())
	  , offset(a.roffset())
      {}

      itereader(Que & a
		,size_t off
		) throw()
	: area(&a)
	  ,frameWidth(a.width())
	  ,offset(off)
      {}

      itereader(itereader & i) throw()
	: area(i.area)
	  , frameWidth(i.frameWidth)
	  , offset(i.offset)
      {}

      itereader(const itereader & i) throw()
	: area(i.area)
	  , frameWidth(i.frameWidth)
	  , offset(i.offset)
      {}

      SubArray
      operator*() const
      {
	return(area->frames()[slice(offset
				    ,frameWidth,1
				    )]
	       );
      }
	       
      itereader &
      operator=(itereader & i)
      {
	area = i.area;
	frameWidth = i.frameWidth;
	offset = i.offset;
	return(*this);
      }
      
      itereader &
      operator++()
      {
	area->raed();
	offset = area->read();
	return(*this);
      }

      bool
      operator==(itereader i)
      {
	return(offset == i.offset);
      }
      bool
      operator!=(itereader i)
      {
	return(offset != i.offset);
      }
    }
    ;

    struct iterwriter
    {
      Que *	area;
      size_t	frameWidth;
      size_t	offset;

      iterwriter() throw()
      {}

      iterwriter(Que & a) throw()
	: area(&a)
	, frameWidth(a.width())
	, offset(a.woffset())
      {}

      iterwriter(iterwriter & i) throw()
	: area(i.area)
	  , frameWidth(i.frameWidth)
	  , offset(i.offset)
      {}

      iterwriter(const iterwriter & i) throw()
	: area(i.area)
	  , frameWidth(i.frameWidth)
	  , offset(i.offset)
      {}

      SubArray
      operator*() const
      {
	return(area->frames()[slice(offset
				    ,frameWidth,1
				    )]
	       );
      }
	       
      iterwriter &
      operator=(iterwriter & i)
      {
	area = i.area;
	frameWidth = i.frameWidth;
	offset = i.offset;
	return(*this);
      }
      
      iterwriter &
      operator++()
      {
	area->wrote();
	offset = area->write();
	return(*this);
      }

      ~iterwriter()
      {
	area->endWrite();
      }
    }
    ;
      
  private:
    size_t		length;
    size_t		frameWidth;
    size_t		area;
    Array		* rframes;


    size_t		writeInc;
    size_t		writeOffset;
    size_t		writeArea;		// initialized to area
    pthread_cond_t	writeCond;
    pthread_mutex_t	writeLock;

    size_t		readInc;
    size_t		readOffset;
    size_t		readArea;
    pthread_cond_t	readCond;
    pthread_mutex_t	readLock;

    bool		writeEnded;

    size_t 	write();			// guarded request to write next frame
		 				// writeArea is checked and decremented
    						// writing itself is not guarded
    void	wrote();

    size_t	read();				// guarded request to read next frame

    void	raed();

    size_t	roffset()
    {
      return(readOffset);
    }

    size_t	woffset()
    {
      return(writeOffset);
    }

  public:
    size_t	width() const
    {
      return(frameWidth);
    }
    void	initReadWrite();
    itereader	beginRead();
    itereader	endRead();
    
    iterwriter	beginWrite();
    void	endWrite();
    
    Array &	frames()
    {
      return(*rframes);
    }
    
    slice	operator-(Que & q)
    {
      int	over = frameWidth - q.frameWidth;
      
      if ( over < 0 )
	{
	  return(slice(0
		       ,frameWidth,1
		       )
		 );
	}
      else
	{
	  return(slice(over>>1
		       ,q.frameWidth,1
		       )
		 );
	}
    }

    Status	put();
    Status	del();

    Que *
    relocate();

    void	dump(
		     ostream &	out
		     ) const;

  }
  ;

}
#endif
