/*===========================================================================*/
/*
 * This file is part of libpersist - a c++ library for object persistence
 *
 * Copyright (C) 2006  Elaine Tsiang YueLien
 *
 * libpersist is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to
 * Free Software Foundation, Inc.
 * 51 Franklin Street, Fifth Floor
 * Boston, MA  02110-1301, USA
 *
 *===========================================================================*/
/*                                                                           */
/* Persistence::GoBase object database class implementation                   */
/*                                                                           */
/*===========================================================================*/
#ifdef __GNUG__
#pragma implementation
#endif

#include	<GoBase.H>

#include	<sstream>
#include	<iostream>

extern "C"
{
#include	<gdbm.h>
#include	<signal.h>
#include	<pthread.h>

  // for sleep
#include	<unistd.h>
  // for memcpy
#include	<string.h>
  // for free
#include	<stdlib.h>
}

namespace	Persistence
{
  BasesClosed::BasesClosed(
			   int	sn
			   ) throw()
    : Exception("Bases Closed"
		,"GoBase"
		,"throwIfInterrupted"
		)
    , signum(sn)
  {
    os() << strsignal(signum) << std::endl;
  }

  OpenFailed::OpenFailed(
			 const char *	path
			 ) throw()
    : Exception("Open Failed"
		,"GoBase"
		,"GoBase"
		)
  {
    os() << path << std::endl;
  }
  //------------------------------------------------------------------------------
  volatile sig_atomic_t	GoBase::HandlingSignal(0);
  volatile sig_atomic_t	GoBase::ClosingBases(0);
  int			GoBase::Interrupt(0);
  listBases *		GoBase::OpenBases = new listBases;
  pthread_mutex_t &	GoBase::InterruptLock = *(new pthread_mutex_t);

  //------------------------------------------------------------------------------
  void
  GoBase::closeBases()
  {
    if ( !OpenBases->empty() )
      {
	for (
	     iBases i = OpenBases->begin()
	       ;
	     i != OpenBases->end()
	       ;
	     ++i
	     )
	  {
	    GoBase * o = *i;

	    // close the object base without locking InterruptLock
	    o->close();

	    // delete it without ~GoBase()
	    operator delete(static_cast<void *>(o));
	  }
      }
  }

  //------------------------------------------------------------------------------
  void	GoBase::signalHandler(
			      int signum
			      )
  {
    if ( HandlingSignal || ClosingBases )
      {
	// try again later
	raise(signum);
	return;
      }
    HandlingSignal = 1;

    //   doNotInterrupt();

    Interrupt = signum;

    switch( Interrupt )
      {
      case SIGFPE	:
      case SIGBUS	:
      case SIGSEGV	:
	break;

      default		:
	HandlingSignal = 0;
	return;

      }
				
    closeBases();

    struct sigaction sa;
    sa.sa_handler = SIG_DFL;
    sigaction(signum
	      ,&sa
	      ,0
	      );
    HandlingSignal = 0;
    raise(signum);
  }
  
  void
  GoBase::handleInterrupts()
  {
    HandlingSignal = 0;

    struct sigaction sa;
    sa.sa_handler = &signalHandler;
    sa.sa_flags = SA_RESTART;

    sigaction(SIGFPE
	      ,&sa
	      ,0
	      );
    sigaction(SIGBUS
	      ,&sa
	      ,0
	      );
    sigaction(SIGSEGV
	      ,&sa
	      ,0
	      );
    sigaction(SIGPIPE
	      ,&sa
	      ,0
	      );
    sigaction(SIGTERM
	      ,&sa
	      ,0
	      );
    sigaction(SIGINT
	      ,&sa
	      ,0
	      );
    sigaction(SIGQUIT
	      ,&sa
	      ,0
	      );
    sigaction(SIGTSTP
	      ,&sa
	      ,0
	      );

  }

  void
  GoBase::doNotInterrupt()
  {
    sigset_t sm;
    sigemptyset(&sm);
    sigaddset(&sm, SIGFPE);
    sigaddset(&sm, SIGBUS);
    sigaddset(&sm, SIGSEGV);
    sigaddset(&sm, SIGPIPE);
    sigaddset(&sm, SIGTERM);
    sigaddset(&sm, SIGINT);
    sigaddset(&sm, SIGQUIT);
    sigaddset(&sm, SIGTSTP);
    pthread_sigmask(SIG_BLOCK, &sm, 0);
  }

  int
  GoBase::whatInterrupted()
  {
    pthread_mutex_lock(&InterruptLock);
    int was = Interrupt;
    Interrupt = 0;
    pthread_mutex_unlock(&InterruptLock);
    return(was);
  }

  void
  GoBase::throwIfInterrupted()
  {
    pthread_mutex_lock(&InterruptLock);
    if ( Interrupt == 0 )
      {
	pthread_mutex_unlock(&InterruptLock);
	return;
      }

    // this is still not water tight between threads
    // recommend this be called only on the interruptable thread
    ClosingBases = 1;
    while ( HandlingSignal )
      ;
    Interrupt = 0;
    closeBases();
    ClosingBases = 0;
    pthread_mutex_unlock(&InterruptLock);
    throw(BasesClosed(Interrupt));
  }

  //------------------------------------------------------------------------------
  void
  GoBase::open()
  {
    Path	gpath;
    size_t	t;

    gpath = path + ".byId";
    for ( t=maxtrys; t; t-- )
      {
	gdbmById = gdbm_open(const_cast<char *>(gpath.cstr())
			     ,gdbmBlocksize
			     ,access
			     ,00660
			     ,NULL
			     );
	if ( !gdbmById )
	  {
	    // std::cout << "waiting for access to " << path.cstr() << std::endl;
	    sleep(timeout);
	  }
	else
	  break;
      }

    if ( !gdbmById )
      {
	gdbmByName = 0;
	throw(OpenFailed(gpath.cstr()));
      }

    gpath = path + ".byName";
    for ( t=maxtrys; t; t-- )
      {
	gdbmByName = gdbm_open(const_cast<char *>(gpath.cstr())
			       ,gdbmBlocksize
			       ,access
			       ,00660
			       ,NULL
			       );
	if ( !gdbmByName)
	  {
	    // std::cout << "waiting for access to " << path.cstr() << std::endl;
	    sleep(timeout);
	  }
	else
	  break;
      }
    if ( !gdbmByName )
      {
	gdbm_close(gdbmById);
	gdbmById = 0;
	throw(OpenFailed(gpath.cstr()));
      }

    pthread_mutex_lock(&InterruptLock);
    if ( OpenBases->empty() )
      defaultOBase(this);
    OpenBases->push_back(this);
    pthread_mutex_unlock(&InterruptLock);
  }

  GoBase::GdbmObjectBase(
			 const char * 	nam
			 ,const char *	roo
			 ,const int	rw
			 ,const size_t	timout
			 ,const size_t	maxtris
			 )
    :ObjectBase()
    ,root(roo)
    ,name(nam)
    ,access(rw)
    ,timeout(timout)
    ,maxtrys(maxtris)
  {
    path = root + "/";
    path += nam;

    pthread_mutex_lock(&InterruptLock);
    if ( OpenBases->empty() )
      {
	handleInterrupts();
      }
    pthread_mutex_unlock(&InterruptLock);
    
    open();
  }

  //------------------------------------------------------------------------------
  GoBase::GdbmObjectBase(
			 GoBase & ob
			 )
    : ObjectBase()
  {
    ob.startWrite();
    root = ob.root;
    name = ob.name;
    path = ob.path;
    access = ob.access;
    timeout = ob.timeout;
    maxtrys = ob.maxtrys;

    ob.close();				// amounts to re-opening
    open();

    ob.gdbmById = gdbmById;
    ob.gdbmByName = gdbmByName;
    ob.endWrite();
  }

  //------------------------------------------------------------------------------
  void
  GoBase::close()
  {
    startWrite();
    if ( gdbmById )
      gdbm_close(gdbmById);

    if ( gdbmByName)
      gdbm_close(gdbmByName);

    gdbmById = 0;
    gdbmByName = 0;
    endWrite();
    
    OpenBases->remove(this);
  }

  GoBase::~GdbmObjectBase()
  {
    pthread_mutex_lock(&InterruptLock);
    close();
    pthread_mutex_unlock(&InterruptLock);
  }

  //------------------------------------------------------------------------------
  Obj *	
  GoBase::get(
	      const Id		id
	      )
  {
    datum	key, val;

    key.dptr = (char *)(&id);
    key.dsize = sizeof(id);
    startRead();
    val = gdbm_fetch(
		     gdbmById
		     ,key
		     );
    endRead();
    Obj * o = 0;
    if ( val.dptr )
      {
	// transfer to ::new
	std::ptrdiff_t headDiff = *reinterpret_cast<std::ptrdiff_t *>(val.dptr);
	size_t objSize = val.dsize - sizeof(std::ptrdiff_t);
	void * v = Obj::operator new(objSize);
	memcpy(reinterpret_cast<char *>(v)
	       ,val.dptr + sizeof(std::ptrdiff_t)
	       ,objSize
	       );
	o = Obj::base(v, headDiff);
	free(val.dptr);
      }
    return(o);
  }

  //------------------------------------------------------------------------------
  Id
  GoBase::getId(
		const char *	nam
		)
  {
    LongName	nm(nam);

    datum	key, val;

    key.dptr = const_cast<char *>(nm.cstr());
    key.dsize = nm.size();
    startRead();
    val = gdbm_fetch(
		     gdbmByName
		     ,key
		     );
    endRead();
    
    Id i = 0;
    if ( val.dptr )
      {
	i = *(reinterpret_cast<Id *>(val.dptr));
	free(val.dptr);
      }
    return(i);
  }

  //------------------------------------------------------------------------------
  Obj *
  GoBase::get(
	      const char *	nam
	      )
  {
    Id i = getId(nam);

    if ( 0 == i )
      return(0);
    else
      return(get(i));
  }

  //------------------------------------------------------------------------------
  Status
  GoBase::put(
	      Obj &		obj
	      )
  {
    // prohibit creating >1 objects under same id 
    Obj * o = get(obj.id());
    if ( o
	 &&
	 ( o->clid() == obj.clid() )
	 &&
	 ( Name(o->cname()) != obj.cname() )
	 )
      {
	Status	sta(
		    "Error"
		    ,"GoBase"
		    ,"put"
		    ,obj.cname()
		    );
	sta.os() << "object of different name " << o->cname() << 
	  " has same id " << obj.id() << std::endl;
	return(sta);
      }
    // prohibit creating >1 objects under same name
    Id i = getId(obj.cname());
    if ( i
	 &&
	 (i != obj.id() )
	 )
      {
	Status	sta(
		    "Error"
		    ,"GoBase"
		    ,"put"
		    ,obj.cname()
		    );
	sta.os() << "object of different id " << i <<
	  " has same name " << obj.cname() << std::endl;
	return(sta);
      }

    datum	key, val;

    i = obj.id();
    // store object under ById
    key.dptr = reinterpret_cast<char *>(&i);
    key.dsize = sizeof(Id);

    // put virtualBase datum in front of object
    val.dsize = sizeof(std::ptrdiff_t) + obj.putSize();
    char	vals[val.dsize];
    val.dptr = vals;
    *(reinterpret_cast<std::ptrdiff_t *>(vals)) = obj.virtualBase();
    memcpy(vals + sizeof(std::ptrdiff_t)
	   ,obj.head()
	   ,obj.putSize()
	   );

    startWrite();
    if ( gdbm_store(gdbmById
		    ,key
		    ,val
		    ,GDBM_REPLACE
		    )
	 )
      {
	endWrite();
	Status	sta(
		    "Error"
		    ,"GoBase"
		    ,"put"
		    ,obj.cname()
		    );

	sta.os() << "gdbm_store ById to " << path.cstr() <<
	  "of object " << obj.cname() << 
	  " failed" << std::endl;
	return(sta);
      }
    endWrite();

    // store id under name
    LongName nm(obj.cname());
    key.dptr = const_cast<char *>(nm.cstr());
    key.dsize = nm.size();

    val.dptr = reinterpret_cast<char *>(&i);
    val.dsize = sizeof(Id);
    
    startWrite();
    if ( gdbm_store(gdbmByName
		    ,key
		    ,val
		    ,GDBM_REPLACE
		    )
	 )
      {
	endWrite();
	Status	sta(
		    "Error"
		    ,"GoBase"
		    ,"put"
		    ,obj.cname()
		    );
	sta.os() << "gdbm_store ByName to " << path.cstr() <<
	  "of object " << obj.cname() << 
	  " failed" << std::endl;
	return(sta);
      }
    endWrite();


    std::cout << "saved " << obj.cname() << std::endl;
    return(true);
  }
  
  //------------------------------------------------------------------------------
  Status
  GoBase::del(
	      Obj &		obj
	      )
  {
    Id		i(obj.id());
    datum	key;
    // for debugging
    Name	nm(obj.cname());	

    startWrite();
    key.dptr = (char *)&i;
    key.dsize = sizeof(i);
    if ( gdbm_exists(
		     gdbmById
		     ,key
		     )
	 )
      {
	if ( 0 > gdbm_delete(
			     gdbmById
			     ,key
			     )
	     )
	  {
	    endWrite();
	    Status	sta(
			    "Error"
			    ,"GoBase"
			    ,"del"
			    ,obj.cname()
			    );
	    sta.os() << "deletion of object by Id " << obj.id() << " failed" << std::endl;
	    return(sta);
	  }
	else
	  {
	    LongName	nm(obj.cname());
	    key.dptr = const_cast<char *>(nm.cstr());
	    key.dsize = nm.size();

	    if ( gdbm_exists(
			     gdbmByName
			     ,key
			     )
		 )
	      {
		if ( 0 > gdbm_delete(
				     gdbmByName
				     ,key
				     )
		     )
		  {
		    endWrite();
		    Status	sta(
				    "Error"
				    ,"GoBase"
				    ,"del"
				    ,obj.cname()
				    );
		    sta.os() << "deletion of '" << obj.cname() << "' by name failed" << std::endl;
		    return(sta);
		  }
	      }
	    else
	      {
		endWrite();
		Status	sta(
			    "Error"
			    ,"GoBase"
			    ,"del"
			    ,obj.cname()
			    );
		sta.os() << "no name corresponds to id for '" << obj.cname() << "'" << std::endl;
		return(sta);
	      }
	  }
      }
    else
      {
	endWrite();
	Status	sta(
		    "Error"
		    ,"GoBase"
		    ,"del"
		    ,obj.cname()
		    );
	sta.os() << "object " << obj.id() << " not in base" << std::endl;
	return(sta);
      }
    endWrite();

    std::cout << "deleted " << nm.cstr() << std::endl;
    return(true);
  }

  //------------------------------------------------------------------------------
  Obj	*
  GoBase::first()
  {
    startRead();
    datum key = gdbm_firstkey(gdbmById);
    endRead();

    if ( key.dptr )
      {
	Id i = *(reinterpret_cast<Id *>(key.dptr));
	free(key.dptr);
	return(get(i));
      }
    else
      return(0);
  }

  //------------------------------------------------------------------------------
  Obj	*
  GoBase::next(
	       const Obj &	obj
	       )
  {
    Id		i = obj.id();	
    datum	key;

    key.dptr = (char *)&i;
    key.dsize = sizeof(i);
    startRead();
    key = gdbm_nextkey(gdbmById
		       ,key
		       );
    endRead();

    if ( key.dptr )
      {
	Id i = *(reinterpret_cast<Id *>(key.dptr));
	free(key.dptr);
	return(get(i));
      }
    else
      return(0);
  }

  //------------------------------------------------------------------------------
  Status
  GoBase::check()
  {
    map<Id, tally>			objCounts;
    typedef map<Id, tally>::iterator	iCounts;
    tally				total;
    datum				key;
    Id					oid;
    char 				lnm[256];
    
    Status	sta(
		    "logging"
		    ,"GoBase"
		    ,"check"
		    );
    sta.succeeded(true);

    startRead();
    for ( key = gdbm_firstkey(gdbmById);
	  key.dptr != 0;
	  key = gdbm_nextkey(gdbmById, key)
	  )
      {
	Obj *	obj = get(*(reinterpret_cast<Id *>(key.dptr)));
	free(key.dptr);
	oid = obj->id();
	key.dptr = reinterpret_cast<char *>(&oid);
	Id 	i = getId(
			  obj->cname()
			  );
	if ( !i )
	  {
	    sta.succeeded(false);
	    sta.os() << "check ERROR: object " << obj->cname() <<
	      " not mapped to its id " << obj->id() <<
	      std::endl;
	  }

	if ( i != obj->id() )
	  {
	    sta.succeeded(false);
	    sta.os() << "check ERROR: id " << i <<
	      " for object " << obj->cname() <<
	      " != object's id " << obj->id() <<
	      std::endl;
	  }

	if ( obj->clid() )
	  {
	    objCounts[obj->clid()].count++;
	    objCounts[obj->clid()].size += obj->putSize();
	  }
	total.count++;
	total.size += obj->putSize();

	if ( (total.count % 1000) == 1 )
	  {
	    sta.os() << "nobjs = " << total.count << "  "
		<< "size  = " << total.size << std::endl;
	  }
      }

    nobjs = total.count;
    sta.os() << "TOTAL = " << total.count << "  "
	<< "size  = " << total.size << std::endl;

    for ( (key = gdbm_firstkey(gdbmByName)),
	    nnames = 0
	    ;
	  key.dptr != 0
	    ;
	  ++nnames,
	    key = gdbm_nextkey(gdbmByName, key)
	  )
      {
	datum val = gdbm_fetch(
			       gdbmByName
			       ,key
			       );

	Id i = *(reinterpret_cast<Id *>(val.dptr));
	free(val.dptr);
	strncpy(lnm, key.dptr, key.dsize);
	lnm[key.dsize] = '\0';
	free(key.dptr);
	key.dptr = lnm;
	
	Obj * obj = get(i);
	if ( !obj )
	  {
	    sta.succeeded(false);
	    sta.os() << "check ERROR: name " << key.dptr <<
	      " not mapped to any id " <<
	      std::endl;
	  }
	else
	  {
	    LongName	nm(obj->cname());
	    if ( nm != lnm )
	      {
		sta.succeeded(false);
		sta.os() << "check ERROR: name " << key.dptr <<
		  " for object " << obj->id() <<
		  " != object's name " << obj->cname() <<
		  std::endl;
	      }
	  }
	
	if ( (nnames % 1000) == 1 )
	  {
	    sta.os() << "nnames = " << nnames << std::endl;
	  }
      }
    sta.os() << "TOTAL = " << total.count << " names" << std::endl;

    for ( iCounts i = objCounts.begin()
	    ;
	  i != objCounts.end()
	    ;
	  ++i
	  )
      {
	sta.os() << "class " << i->first << ": " <<
	  (i->second).count << " objs = " <<
	  float((i->second).size)/1000.0 << "kBs" <<
	  std::endl;

      }

    endRead();
    return(sta);
  }

  //------------------------------------------------------------------------------
  Status
  GoBase::reorg()
  {
    Status	sta(
		    "logging"
		    ,"GoBase"
		    ,"reorg"
		    );
    sta.succeeded(true);

    startWrite();
    if ( -1 == gdbm_reorganize(gdbmById) )
      {
	sta.os() << gdbm_strerror(gdbm_errno) <<
	  " failure on reorganizing dbmById of " << path.cstr() <<
	  std::endl;
      }
    else
      {
	sta.os() << "finished reorganizing dbmById of " << path.cstr() <<
	  std::endl;
      }

    if ( -1 == gdbm_reorganize(gdbmByName) )
      {
	sta.os() << gdbm_strerror(gdbm_errno) <<
	  " failure on reorganizing dbmByName of " << path.cstr() <<
	  std::endl;
      }
    else
      {
	sta.os() << "finished reorganizing dbmByName of " << path.cstr() <<
	  std::endl;
      }
    endWrite();
    return(sta);
  }

  //------------------------------------------------------------------------------
  GoBase &
  GoBase::operator=(
		    GoBase &		src
		    )
  {
    src.startRead();
    startWrite();
    for ( datum key = gdbm_firstkey(src.gdbmById);
	  key.dptr != 0;
	  key = gdbm_nextkey(src.gdbmById, key)
	  )
      {
	Obj *	obj = src.get(*(reinterpret_cast<Id *>(key.dptr)));
	free(key.dptr);
	
	if ( put(*obj).failed() )
	  {
	    return(*this);
	  }
      }

    endWrite();
    src.endRead();

    return(*this);
  }

}
