/*===========================================================================*/
/*
 * This file is part of libogg++ - a c++ library for the Ogg transport format
 *
 * Copyright (C) 2006, 2007, 2008 Elaine Tsiang YueLien
 *
 * libogg++ is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc.
 * 51 Franklin Street, Fifth Floor
 * Boston, MA  02110-1301, USA
 *
 *===========================================================================*/
/** @file Transport.H                                                        *
 *                                                                           *
 *  Ogg::Transport to application interface                                  *
 *                                                                           */
/*===========================================================================*/
#ifndef TransportToApplicationInterface
#define	TransportToApplicationInterface

#ifdef __GNUG__
#pragma interface
#endif

namespace	Ogg
{
  class Transport;
  class Page;
}

#include	<Ogg/Ogg.H>
#include	<Ogg/Logical.H>
#include	<Ogg/Exception.H>

namespace	Ogg
{

  /// @defgroup transport 	Transport Interface
  //@{
  
  /// @brief Transport paging interface
  ///
  /// @ingroup derivables
  ///
  /// Do not derive from Page.
  class Page
  {
  public:
    void
    data(void *
	 ,size_t
	 );

    void *
    data();		///< data area of page
    
    size_t
    size() const;	///< size of data area on reading

    bool
    ending() const;	///< Is a logical stream ending in this page?

    Position
    granulePosition() const;	///< of this page

    bool
    isHeader()const ;	///< of this page

    long
    pageNo() const;	///< of this page

    long
    serialNo() const;	///< of the logical stream of this page

    long
    pageOrderNo() const;	///< for debugging

  };

  /// @brief Transport interface
  ///
  /// @ingroup derivables
  ///
  /// Derive from this class
  /// and override Transport::sendCallback and Transport::recvCallback
  /// to provide the actual transport logic.
  class Transport
  {
  private:
    void *		impl_;

    /// Copy and assignment are verboten
    Transport(const Transport &);

    Transport &
    operator=(const Transport &);

  protected:
    /// @brief Despite appearance, derived classes should NOT call this
    Transport(
	      int dummy
	      )
      : impl_(0)
    {}

  public:
    void *
    impl()
    {
      return(impl_);
    }

    static
    const unsigned char	CurrentVersion = 0x00;

    /*----- Exceptions thrown by a Transport thread ---------------------------------*/

    ///@brief Exception thrown by Transport::~Transport()
    class NoDeletion : public Exception
    {
    public:
      NoDeletion() throw()
      : Exception("Sorry, you can't delete Transport instances")
      {
	*this << std::endl;
      }

      NoDeletion(
		 const NoDeletion & ex
		 ) throw()
	: Exception(ex)
      {}

      ~NoDeletion() throw()
      {}
    }
    ;

    ///@brief Exception thrown by a reading Transport::loop
    ///
    /// Thrown after Transport::loop sends an end of stream packet
    /// which carries an Error::PrematureEnd to each of the reading Logicals
    class PrematureEndOfOggStream : public Exception
    {
    public:
      PrematureEndOfOggStream() throw()
      : Exception("Premature end of transport stream.")
      {
	*this << std::endl;
      }

      PrematureEndOfOggStream(
			      const PrematureEndOfOggStream & ex
			      ) throw()
	: Exception(ex)
      {}
      
      ~PrematureEndOfOggStream() throw()
      {}
    }
    ;

    /*- end Exceptions thrown by a Transport thread ---------------------------------*/

    /// @brief An argument of false means decapsulating
    Transport(
	      bool	encapsulating
	      );

    /// @brief Disallows deletion. Throws NoDeletion exception.
    virtual
    ~Transport();

    /// @brief Get the next unselected serial no.
    ///
    /// For encapsulation, returns a valid unused serial number
    /// greater or equal to its argument.
    /// For decapsulation, returns the serial number of
    /// a detected logical stream greater than its argument
    long
    serialNo(long);

    /// @brief Returns the Logical with serialNo.
    Logical *
    logical(
	    long	serialNo
	    );

    /// @brief Encapsulation/decapsulation loop in Transport thread.
    ///
    /// On encapsulation, accepts initiations of Logical streams for sending.
    /// Enforces page ordering according to Ogg::operator<(Page, Page).
    ///
    /// On decapsulation, makes ready first header packet for each logical stream.
    /// To be safe, instantiate the reading Logical instances before you loop.
    ///
    /// Call this within a try block to catch exceptions.
    /// To force exit from the loop, call terminate from another thread
    /// or from within the call back methods.
    ///
    void
    loop();

    /// @brief Called by thread other than Transport or in call backs.
    ///
    /// Sends command to Transport to exit loop
    /// at end of an Ogg stream.
    ///
    void
    terminate();

    /// @brief Send call back.
    ///
    /// @ingroup overrides
    ///
    /// Override with actual send logic.
    ///
    /// Default: Dumps to standard out.
    /// Terminates transport upon removal of all Logical instances.
    ///
    virtual
    void
    sendCallback(
		 Page &
		 );

    /// @brief Receive call back.
    ///
    /// @ingroup overrides
    ///
    /// Override with actual receive logic.
    ///
    /// Default: Reads from standard in.
    /// Terminates transport upon end of file.
    ///
    /// 
    /// Page::size() gives limit to size of data that can be received.
    /// Returns actual size received.
    virtual
    size_t
    recvCallback(
		 Page &
		 );

    /// @brief Transport page ordering function
    ///
    /// @ingroup overrides
    ///
    /// Default page ordering is by
    /// granule position,
    /// then headers first (all headers have granule position of 0?)
    /// then page no.
    /// then serial no.
    ///
    /// The Pages should be for the same encapsulating Transport.
    virtual
    bool
    laterThan (
	       const Page &		p1
	       ,const Page &		p2
	       );

  }
  ;

  //@} transport
};
#endif
