/*===========================================================================*/
/*
 * This file is part of libogg++ - a c++ library for transport of the Ogg format
 *
 * Copyright (C) 2006, 2007  Elaine Tsiang YueLien
 *
 * libogg++ is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Lesser General Public
 * License as published by the Free Software Foundation; either
 * version 2.1 of the License, or (at your option) any later version.
 * 
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Lesser General Public License for more details.
 * 
 * You should have received a copy of the GNU Lesser General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc.
 * 51 Franklin Street, Fifth Floor
 * Boston, MA  02110-1301, USA
 *
 *===========================================================================*/
/** @file multiThreadedDemux.C                                               *
/*                                                                           */
/* multiThreadedSeek - multi-threaded seeking                                */
/*                                                                           */
/*===========================================================================*/
#include	<Ogg/Ogg.H>
#include	<Debug.H>
#include	<Thread.H>


extern "C" 
{
#include	<string.h>
}

namespace	Examples
{
  using namespace	Ogg;

  class VorbisParser : public Logical, public Thread
  {
  public:
    VorbisParser(
		 Transport & transport
		 )
      : Logical(transport)
    {}

    bool
    selectCallback(
		   Packet &	pkt
		   )
    {
      const  char * header = static_cast<const  char *>(pkt.data());
      const unsigned char * uheader = static_cast<const unsigned char *>(pkt.data());

      if ( (pkt.size() < 30)
	   ||
	   (uheader[0] != 0x01)
	   ||
	   (strncmp (header+1, "vorbis", 6))
	   )
	return(false);
      else
	return(true);
    }

    void
    run()
    {
      try
	{
	  size_t total = 0;
	  Reader & r = reader();
	  total += r->size();

	  // print first header packet
	  debug(true,
		"Vorbis : serial no ", serialNo()
		,"got first header"
		);

	  size_t inc = 1;
	  Position granulePosition = r.granulePosition();
	  do
	    {
	      // seek by granule position
	      if ( r.granulePosition() >= (granulePosition + inc) )
		{
		  // print subsequent packet #, position
		  debug(true,
			"Vorbis : got ",r->packetNo(),"th packet, "
			,"granule position = "
			,r.granulePosition()
			);

		  inc += inc;
		  granulePosition = r.granulePosition() + inc;
		  debug(true,
			"Vorbis : seeking granule position ", granulePosition
			);
		  r = granulePosition;
		  // print subsequent packet #, position
		  debug(true,
			"Vorbis : got ",r->packetNo(),"th packet, "
			,"granule position = "
			,r.granulePosition()
			);
		  granulePosition = r.granulePosition();
		}
	      else
		++r;

	      if ( r.ending() )
		debug(true,
		      "Vorbis : got ",r->packetNo(),"th packet, "
		      ,"granule position = "
		      ,r.granulePosition()
		      );

	    }
	  while ( !r.ending() );

	}
      catch ( exception ex
	      )
	{
	  debug(true,
		ex.what()
		);
	  throw;
	}

      return;
    }
      
  }
  ;
  
  class TheoraParser : public Logical, public Thread
  {
  public:
    TheoraParser(
		 Transport & transport
		 )
      : Logical(transport)
    {}


    bool
    selectCallback(
		   Packet &	pkt
		   )
    { 
      const  char * header = static_cast<const  char *>(pkt.data());
      const unsigned char * uheader = static_cast<const unsigned char *>(pkt.data());

      if ( (pkt.size() < 41)
	   ||
	   (uheader[0] != 0x80)
	   ||
	   (strncmp (header+1, "theora", 6))
	   )
	return(false);
      else
	return(true);
    }

    void
    run()
    {
      try
	{
	  size_t total = 0;
	  Reader & r = reader();
	  total += r->size();

	  // print first header packet
	  debug(true,
		"Theora : serial no ", serialNo()
		,"got first header"
		);

	  
	  size_t inc = 1;
	  PacketNo lastPktNo = r->packetNo();
	  do
	    {
	      // seek by packet increment;
	      if ( r->packetNo() >= (lastPktNo + inc + 7) )
		{
		  inc += inc;
		  lastPktNo = r->packetNo();
		  r += inc;
		}
	      else
		++r;
	      // print subsequent packet #, position
	      debug(true,
		    "Theora : got ",r->packetNo(),"th packet, "
		    ,"granule position = "
		    ,r.granulePosition()
		    );
	    }
	  while ( !r.ending() );

	}

      catch ( exception ex
	      )
	{
	  debug(true,
		ex.what()
		);
	  throw;
	}

      return;
    }
  }
  ;

  class Muxer: public Transport, public Thread
  {
  public:
    Muxer()
      : Transport(false)
    {}

    void
    run()
    {
      loop();
    }

    size_t
    recvCallback(
		 Page &	pg
		 )
    {
      size_t count = Transport::recvCallback(pg);

      if ( std::cin.eof()
	   ||
	   (count <= 0 )
	   )
	{
	  terminate();
	}

      return(count);
    }
  }
  ;

}

int main(int argc, const char* argv[])
{
  using namespace	Examples;
  using namespace	Ogg;

  Muxer * transport = new Muxer;
  VorbisParser * vorparser = new VorbisParser(*transport);
  TheoraParser * theoparser = new TheoraParser(*transport);

  transport->start();
  vorparser->start();
  theoparser->start();

  vorparser->finish();
  theoparser->finish();
  transport->finish();

  exit(0);
}
