/*
Copyright (C) 2015 John Tse

This file is part of Libknit.

Libknit is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

Libknit is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with Libknit.  If not, see <http://www.gnu.org/licenses/>.
*/

#include "_knit.h"

KNIT knit_new() {
	return calloc(1, sizeof(struct knit_ctx));
}

void knit_free(KNIT knit) {
	free(knit);
}

void knit_set(KNIT knit, uint32_t flags, ...) {
	va_list ap;

	va_start(ap, flags);

	if (flags & KNIT_KEY) {
		knit->k        = va_arg(ap, void *);
		knit->k_length = va_arg(ap, size_t);
	}

	if (flags & KNIT_IV) {
		knit->iv        = va_arg(ap, void *);
		knit->iv_length = va_arg(ap, size_t);
	}

	if (flags & KNIT_EKB)
		knit->ekb = va_arg(ap, size_t);

	if (flags & _KNIT_CIPHER)
		knit->cipher = flags & _KNIT_CIPHER;

	if (flags & _KNIT_HASH)
		knit->hash = flags & _KNIT_HASH;

	if (flags & _KNIT_MAC)
		knit->mac = flags & _KNIT_MAC;

	if (flags & _KNIT_MODE)
		knit->mode = flags & _KNIT_MODE;

	va_end(ap);
}

uint8_t knit_do(KNIT knit, uint8_t op, void *x, size_t x_length, ...) {
	uint8_t r;
	va_list ap;

	uint8_t **z;
	size_t *z_length;

	va_start(ap, x_length);

	switch (op) {
		case KNIT_ENCRYPT:
		case KNIT_DECRYPT:
			z        = va_arg(ap, uint8_t **);
			z_length = va_arg(ap, size_t *);

			if (op == KNIT_ENCRYPT)
				r = _knit_encrypt(knit, x, x_length, z, z_length);
			else
				r = _knit_decrypt(knit, x, x_length, z, z_length);

			break;
		case KNIT_HASH:
		case KNIT_MAC:
			z = va_arg(ap, uint8_t **);

			if (op == KNIT_HASH)
				r = _knit_hash(knit, x, x_length, z);
			else
				r = _knit_mac(knit, x, x_length, z);

			break;
		default:
			r = KNIT_ERROR_NO_OP;
	}

	va_end(ap);

	return r;
}

uint8_t * knit_error(uint8_t errnum) {
	switch (errnum) {
		case KNIT_ERROR_OK:
			return _KNIT_ERROR_OK;
		case KNIT_ERROR_NO_OP:
			return _KNIT_ERROR_NO_OP;
		case KNIT_ERROR_NO_ALGO:
			return _KNIT_ERROR_NO_ALGO;
		case KNIT_ERROR_NO_KEY:
			return _KNIT_ERROR_NO_KEY;
		case KNIT_ERROR_NO_IV:
			return _KNIT_ERROR_NO_IV;
		case KNIT_ERROR_MIN_KEY_SIZE:
			return _KNIT_ERROR_MIN_KEY_SIZE;
		default:
			return NULL;
	}
}
