/********************************************************************************
	libeddml - parses eddml files to extract error information
	Copyright (C) 2004 by Pete Rowley
	pete@openrowley.com

    This file is part of the libeddml project.

    libeddml is free software; you can redistribute it and/or modify
    it under the terms of the GNU Lesser General Public License as published by
    the Free Software Foundation; either version 2.1 of the License, or
    (at your option) any later version.

    libeddml is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU Lesser General Public License for more details.

    You should have received a copy of the GNU Lesser General Public License
    along with Foobar; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*********************************************************************************/

/*
	Clients of the interface are advised against direct
	manipulatation of structures since this may
	harm their future compatibility with the interface.
*/

/* includes */
#include <sys/types.h> /* stat */
#include <sys/stat.h> /* "" */
#include <unistd.h> /* "" */

#include <stdlib.h> /* strtol */
#include <limits.h>
#include <dirent.h>
#include <sys/types.h>
#include <errno.h>

#include <stdio.h>
#include <string.h>
#include <ctype.h>

#include <assert.h>

#include <libxml/xmlreader.h>
#include "eddml.h"

/*
   this our secret definition of the
   translator, it is opaque to clients
*/
struct _eddml_translator
{
	/* translator configuration */
	xmlChar **files;
	xmlChar **paths;
	xmlChar **env_paths;

	/* translation configuration */
	xmlChar **errorsets;
	int alphaCode;
	char *errorText;
	long error;
	int match_all;

	/* translation state */
	int filePos;
	int pathPos;
	int envPos;
	int eof;
	xmlTextReaderPtr reader;
	DIR *dir;
	int in_note;
	int in_symbol;

	/* error code translation */
	/* set info */
	xmlChar *set_name;
	xmlChar *set_version;
	xmlChar *set_authority;
	xmlChar *set_docs;

	/* error info */
	xmlChar *error_code;
	xmlChar *error_desc;
	xmlChar *error_severity;
	xmlChar **error_note_title;
	xmlChar **error_note_text;
	xmlChar **error_symbol_lang;
	xmlChar **error_symbol_text;
};

/* return codes */
#define EDDML_SUCCESS 0
#define EDDML_ERROR_GENERAL 5320
#define EDDML_ERROR_CALL_ONCE EDDML_ERROR_GENERAL + 1
#define EDDML_ERROR_INVALID_TARGET EDDML_ERROR_GENERAL + 2

/* macros */
#define IN_SET(t) (t->set_name) /* in a set entity */
#define IN_ERROR(t) (t->error_code) /* in an error entity */
#define IN_NOTE(t) (t->in_note) /* in a note entity */
#define IN_SYMBOL(t) (t->in_symbol) /* in a symbol entity */

/* symbols */
#define EDDML_ARRAY_SIZE  5
#define EDDML_XML_OPTIONS XML_PARSE_RECOVER | XML_PARSE_NOENT | XML_PARSE_DTDATTR
#define EDDML_ENV_PATH "ERRORPATH"

/* forward declarations */
static int int_set_filelist(xmlChar ***target, const char **filepaths, const int isDir, char **bad_paths);
static int int_switch_files(eddml_translator *translator);
static int int_parse_node(eddml_translator *translator);
static void int_free_errorset(eddml_translator *translator);
static void int_parse_errorset(eddml_translator *translator);
static void int_free_error(eddml_translator *translator);
static void int_parse_error(eddml_translator *translator);
static void int_add_array_item(void **array_in, void *item);
static void int_parse_note(eddml_translator *translator);
static void int_parse_symbol(eddml_translator *translator);
static int int_get_data(char **to, char *from);
static void int_iterate_paths(eddml_translator *translator, xmlChar **paths, int *pos);
static void int_free_array(xmlChar ***a);


/* function definitions */

/*
	initialise the library
	--

	return: error code
*/
int
eddml_init(void)
{
	int ret = 0;
	static int call_once = 0;

	if(!call_once)
	{
		xmlInitParser();
		LIBXML_TEST_VERSION;
	}
	else
	{
		ret = EDDML_ERROR_CALL_ONCE;
	}

	return ret;
}

/*
	shutdown and clean up libeddml
*/
void
eddml_done(void)
{
    xmlCleanupParser();
}

/*
	intialize a translator
	--
	arg translator: the translator to initialize

	return: error code
*/
int
eddml_get_translator(eddml_translator **translator)
{
	int ret = 0;

	if(translator != 0)
	{
		*translator = (eddml_translator*) malloc(sizeof(eddml_translator));
		if(*translator != 0)
		{
			memset(*translator, 0, sizeof(eddml_translator));
		}
		else
		{
			ret = ENOMEM;
		}
	}
	else
	{
		ret = EINVAL;
	}

	return ret;
}

int int_set_filelist(xmlChar ***target, const char **filepaths, const int isDir, char **bad_paths)
{
	int ret = 0;
	struct stat buf;
	char **tmp_filepaths = (char**)filepaths;
	int bad_count = 0;

	/* check validity of files */
	do
	{
		ret = stat(*tmp_filepaths, &buf);
		if(0 == ret)
		{
			if(isDir)
			{
				if(!S_ISDIR(buf.st_mode))
				{
					if(bad_paths)
					{
						bad_paths[bad_count++] = *tmp_filepaths;
						bad_paths[bad_count] = 0;
					}
					else
					{
						ret = EDDML_ERROR_INVALID_TARGET;
						break;
					}
				}
			}
			else
			{
				if(!S_ISREG(buf.st_mode))
				{
					if(bad_paths)
					{
						bad_paths[bad_count++] = *tmp_filepaths;
						bad_paths[bad_count] = 0;
					}
					else
					{
						ret = EDDML_ERROR_INVALID_TARGET;
						break;
					}
				}
			}
		}
		else
		{
			if(bad_paths)
			{
				bad_paths[bad_count++] = *tmp_filepaths;
				bad_paths[bad_count] = 0;
				ret = 0;
			}
			else
			{
				ret = errno;
				break;
			}
		}

		tmp_filepaths++;
	}
	while(*tmp_filepaths);

	if(0 == ret)
	{
		/* copy file paths */
		int count = 0;

		tmp_filepaths = (char**)filepaths;

		while(tmp_filepaths[count])
		{
			int_add_array_item((void**)target, xmlCharStrdup(tmp_filepaths[count]));
		}
	}

	return ret;
}

/*
	set translation files to be used
	--
	arg translator: the translator to use
	arg filepaths:
		null terminated array of translation file paths,
		these should remain valid in the scope of this translator

	return: error code
*/
int
eddml_use_translation_files(eddml_translator *translator, const char **filepaths, char **bad_paths)
{
	int ret = EDDML_ERROR_GENERAL;

	if(translator != 0 && filepaths != 0 && *filepaths != 0)
	{
		if(translator->files)
		{
			free(translator->files);
			translator->files = 0;
		}

		ret = int_set_filelist(&translator->files, filepaths, 0, bad_paths);

	}
	else
	{
		ret = EINVAL;
	}

	return ret;
}


/*
	set translation paths to be used
	--
	arg translator: the translator to use
	arg paths:
		null terminated array of paths where translations may be found,
		these should remain valid in the scope of this translator

	return: error code

	maybe one path only
*/
int
eddml_use_translation_paths(eddml_translator *translator, const char **paths, char **bad_paths)
{
	int ret = 0;

	if(translator != 0 && paths != 0 && *paths != 0)
	{
		if(translator->paths)
		{
			free(translator->paths);
			translator->paths = 0;
		}

		int_set_filelist(&translator->paths, paths, 0, bad_paths);
	}
	else
	{
		ret = EINVAL;
	}

	return ret;
}


int eddml_use_translation_defaults(eddml_translator *translator)
{
	char *env;

	/* load environment vars and/or config */
	env = getenv(EDDML_ENV_PATH);
	if(env && *env)
	{
		/* parse paths */
		char *pos;
		char *start = env;;

		pos = strchr(start, ':');
		while(start)
		{
			int end = pos? (int)(pos - start) - 1:0;

			if(pos == 0)
			{
				/* last path */
				int_add_array_item((void**)&translator->env_paths, xmlCharStrdup(start));
				break;
			}
			else
			{
				xmlChar *buf = xmlCharStrndup(start, end);

				int_add_array_item((void**)&translator->env_paths, buf);
			}

			start = (&start[end]) + 2;
			if(*start)
			{
				pos = strchr(start, ':');
			}
		}
	}

	return 0;
}

/*
	initiate translation of error code
	--
	arg translator: the translator to use
	arg errorText: textual error code (may be null)
	arg error: the error code to translate (used if errorText is null)
	arg errorsets: the errorsets to consider (partial string match), null means all
	arg translation: the returned translation

	return: error code
*/
int
eddml_translate(
	eddml_translator *translator, char *errorText, long error,
	const char **errorsets)
{
	int ret = 0;
	int i = 0;

	if(translator != 0)
	{
		translator->error = error;

		if(errorText)
		{
			translator->errorText = strdup(errorText);
			translator->alphaCode = 0;

			/* try to convert to a numeric code */
			char *check;

			translator->error = strtol(errorText, &check, 0);
			if(*check != 0 || translator->error == LONG_MAX || translator->error == LONG_MIN)
			{
				// try to find this code using text match
				translator->error = 0;
				translator->alphaCode = 1;

				/* see if we have the wild card */
				if(errorText[0] == '*')
					translator->match_all = 1;
			}
		}
		else
		{
			translator->match_all = 1;
		}

		while(errorsets && errorsets[i])
		{
			int_add_array_item((void**)&translator->errorsets, xmlCharStrdup(errorsets[i]));
			i++;
		}
	}
	else
	{
		ret = EINVAL;
	}

	return ret;
}


void int_iterate_paths(eddml_translator *translator, xmlChar **paths, int *pos)
{
	if(paths)
	{
		int done;

		do
		{
			done = 1;

			/* are we at the end of the last directory enumeration? */
			if(translator->dir == 0  && paths[*pos])
			{
				translator->dir = opendir(paths[*pos]);
				(*pos)++;
			}

			/* get next file */
			if(translator->dir)
			{
				struct dirent *ent = 0;
				struct stat stat_buf;
				char *filename = 0;

				/* skip dirs */
				do
				{
					ent = readdir(translator->dir);
					if(ent)
					{
						/* despite the cost, we dynamically allocate for the path
						 * some os have no limit on path length, and even those
						 * that do are subject to relative path buffer overflow bugs
						*/
						filename = (char*)malloc(
								strlen(paths[(*pos)-1])
								+ strlen(ent->d_name) + 2);
						strcpy(filename, paths[(*pos)-1]);
						strcat(filename, "/");
						strcat(filename, ent->d_name);

						if(0 == stat(filename, &stat_buf) && S_ISREG(stat_buf.st_mode))
						{
							translator->reader = xmlReaderForFile(
									(char*)filename, NULL, EDDML_XML_OPTIONS);
						}

						free(filename);
					}
				}
				while(ent && 0 == translator->reader);

				if(0 == ent)
				{
					/* end of enumeration, get next path */
					closedir(translator->dir);
					translator->dir = 0;
					done = 0;
				}
			}
		}
		while(!done);
	}
}

/* add in memory documents option */
int
int_switch_files(eddml_translator *translator)
{
	int ret = 0;

	/* null reader means switch required */
	if(translator && translator->reader == 0)
	{
		/* switching files, named files first */
		if(translator->files && translator->files[translator->filePos])
		{
			do
			{
				/* it is possible files may have been deleted since init */
				translator->reader = xmlReaderForFile(
					translator->files[translator->filePos], NULL, EDDML_XML_OPTIONS);
				translator->filePos++;
			}
			while(translator->reader == 0 && translator->files[translator->filePos]);
		}

		/* try paths next */
		if(translator->reader == 0)
		{
			int_iterate_paths(translator, translator->paths, &(translator->pathPos));
		}

		/* try environment set paths next */
		if(translator->reader == 0)
		{
			int_iterate_paths(translator, translator->env_paths, &(translator->envPos));
		}

		/* if we have not got a reader, this is the end of the query */
		if(translator->reader == 0)
		{
			translator->eof = 1;
			ret = -1;
		}
	}

	if(translator == 0)
		ret = -1;

	return ret;
}

void
int_xmlfree(xmlChar **str)
{
	if(*str)
	{
		xmlFree(*str);
		*str = 0;
	}
}

void
int_free_errorset(eddml_translator *translator)
{
	int_xmlfree(&translator->set_name);
	int_xmlfree(&translator->set_version);
	int_xmlfree(&translator->set_authority);
	int_xmlfree(&translator->set_docs);
}


void
int_parse_errorset(eddml_translator *translator)
{
	xmlChar **sets = translator->errorsets ? &(translator->errorsets[2]) : 0;
	xmlChar **first_set = sets;
	int len;

	/* find out if this is one of the errorsets we seek, match only start of string */
	xmlChar *name = xmlTextReaderGetAttribute(translator->reader, "name");

	if(sets)
	{
		while(*sets && (len = xmlStrlen(*sets)) && xmlStrncasecmp(name, *sets, len))
		{
			sets++;
		}
	}

	/* if it is a named set or we are searching all sets */
	if(first_set == 0 || (first_set && first_set[0] == 0) || *sets)
	{
		/* got a match on errorset, fill in info on set */
		translator->set_name = name;
		translator->set_version = xmlTextReaderGetAttribute(translator->reader, "version");
		translator->set_authority = xmlTextReaderGetAttribute(translator->reader, "authority");
		translator->set_docs = xmlTextReaderGetAttribute(translator->reader, "authoritiveDocuments");
	}
	else
	{
		int_xmlfree(&name);
	}
}


void
int_free_array(xmlChar ***a)
{
	int i = 2;

	if(*a)
	{
		while((*a)[i])
		{
			int_xmlfree(&((*a)[i]));
			i++;
		}

		free(*a);
		*a = 0;
	}
}


void
int_free_error(eddml_translator *translator)
{
	int_xmlfree(&translator->error_code);
	int_xmlfree(&translator->error_desc);
	int_xmlfree(&translator->error_severity);
	int_free_array(&translator->error_note_title);
	int_free_array(&translator->error_note_text);
	int_free_array(&translator->error_symbol_lang);
	int_free_array(&translator->error_symbol_text);
}


int
eddml_translation_free(eddml_translator **t)
{
	eddml_translator *translator;

	if(*t == 0)
		return 0;

	translator = *t;

	int_free_error(translator);

	int_free_array(&translator->files);
	int_free_array(&translator->paths);
	int_free_array(&translator->env_paths);
	int_free_array(&translator->errorsets);

	if(translator->reader)
	{
		xmlFreeTextReader(translator->reader);
	}

	if(translator->dir)
	{
		closedir(translator->dir);
		translator->dir = 0;
	}

	if(translator->errorText)
	{
		free(translator->errorText);
		translator->errorText = 0;
	}

	free(translator);
	*t = 0;

	return 0;
}


void
int_parse_error(eddml_translator *translator)
{
	xmlChar *code_str = xmlTextReaderGetAttribute(translator->reader, "code");

	if(code_str)
	{
		int match = 0;

		if(!translator->match_all)
		{
			if(translator->alphaCode)
			{
				match = !xmlStrcasecmp(translator->errorText, code_str);
			}
			else
			{
				char *check;
				long val = strtol(code_str, &check, 0);

				if(*check != 0 || val == LONG_MAX || val == LONG_MIN)
				{
					/* unparseable as value */
				}
				else
				{
					match = val == translator->error;
				}
			}
		}
		else
		{
			match = 1;
		}

		if(match)
		{
			translator->error_code = code_str;
			translator->error_severity = xmlTextReaderGetAttribute(translator->reader, "severity");
			translator->error_desc = xmlTextReaderGetAttribute(translator->reader, "description");
		}
		else
		{
			int_xmlfree(&code_str);
		}
	}
}

void
int_add_array_item(void **array_in, void *item)
{
	assert(array_in);
	int *array = (int*)*array_in; /* convenience */

	/* first deal with memory issues */
	if(*array_in)
	{
		/* is the current array big enough to add a value */
		/* first element gives us number of items, second the array size */
		if(array[0] + 4 > array[1]) /* 4 for # items, size of array, new item, null terminator */
		{
			/* need to grow the array */
			void *tmp = realloc(array, array[1] * 2 * sizeof(int*));
			if(tmp)
			{
				*array_in = tmp;
				array = (int*)*array_in;
				memset(*array_in + (array[1] * sizeof(int*)), 0, array[1] * sizeof(int*));
				array[1] *= 2;
			}
			else
			{
				goto end;
			}
		}
	}
	else
	{
		/* create initial array */
		*array_in = malloc(EDDML_ARRAY_SIZE * sizeof(int*));
		if(*array_in)
		{
			memset(*array_in, 0, EDDML_ARRAY_SIZE * sizeof(int*));
			array = (int*)*array_in;
			array[1] = EDDML_ARRAY_SIZE;
		}
		else
		{
			goto end;
		}
	}

	/* now add item */
	(*(char***)array_in)[array[0] + 2] = (char*)item;
	array[0]++;
	/* terminating null */
	(*(int**)array_in)[array[0] + 2] = 0;


end:
	;
}

void
int_parse_note(eddml_translator *translator)
{
	xmlChar *val = xmlTextReaderGetAttribute(translator->reader, "title");
	if(!val)
	{
		val = xmlCharStrdup("Note");
	}

	int_add_array_item((void**)&translator->error_note_title, val);
}

void
int_parse_symbol(eddml_translator *translator)
{
	xmlChar *val = xmlTextReaderGetAttribute(translator->reader, "language");
	if(!val)
	{
		val = xmlCharStrdup("Symbol");
	}

	int_add_array_item((void**)&translator->error_symbol_lang, val);
}

int
int_parse_node(eddml_translator *translator)
{
	int ret = 1;
	xmlChar *entity_name;

	entity_name = xmlTextReaderName(translator->reader);
	if(entity_name)
	{
		if(!xmlStrcasecmp(entity_name, "errorset"))
		{
			if(IN_SET(translator))
			{
				int_free_error(translator);
				int_free_errorset(translator);
			}
			else
			{
				int_parse_errorset(translator);
			}
		}
		else if(IN_SET(translator) && !xmlStrcasecmp(entity_name, "error"))
		{
			if(IN_ERROR(translator))
			{
				ret = 0;
			}
			else
			{
				int_parse_error(translator);
			}
		}
		else if(IN_ERROR(translator) && !xmlStrcasecmp(entity_name, "note"))
		{
			if(IN_NOTE(translator))
			{
				translator->in_note = 0;
			}
			else
			{
				int_parse_note(translator);
				translator->in_note = 1;
			}
		}
		else if(IN_NOTE(translator) && !xmlStrcasecmp(entity_name, "#text"))
		{
			xmlChar *val = xmlTextReaderValue(translator->reader);
			if(!val)
			{
				val = xmlCharStrdup("");
			}

			int_add_array_item((void**)&translator->error_note_text, val);
		}
		else if(IN_ERROR(translator) && !xmlStrcasecmp(entity_name, "symbol"))
		{
			if(IN_SYMBOL(translator))
			{
				translator->in_symbol = 0;
			}
			else
			{
				int_parse_symbol(translator);
				translator->in_symbol = 1;
			}
		}
		else if(IN_SYMBOL(translator) && !xmlStrcasecmp(entity_name, "#text"))
		{
			xmlChar *val = xmlTextReaderValue(translator->reader);
			if(!val)
			{
				val = xmlCharStrdup("");
			}

			int_add_array_item((void**)&translator->error_symbol_text, val);
		}

		int_xmlfree(&entity_name);
	}

	return ret;
}

/*
	get the next error translation
	--
	arg translation: the translation to use

	return: -1 for no more entries, 0 for success, error codes
*/
int
eddml_next_error(eddml_translator *translator)
{
	int ret = -1;

	int_free_error(translator);

	/* switches: next error, next file, next filepath, next file in path */
	if(translator && !translator->eof)
	{
		ret = int_switch_files(translator);

		if(ret == 0)
		{
			do
			{
				do
				{
					ret = xmlTextReaderRead(translator->reader);
					if(ret != 1)
					{
						xmlFreeTextReader(translator->reader);
						translator->reader = 0;
						int_switch_files(translator);
					}
				}
				while(ret != 1 && translator->eof == 0);

				if(ret == 1)
				{
					ret = int_parse_node(translator);
				}
			}
			while(ret == 1);
		}
	}

	if(translator->eof)
	{
		ret = -1;
	}

	return ret;
}


/*
	get the first error translation
	--
	arg translation: the translation to use

	return: error code
*/
int
eddml_first_error(eddml_translator *translator)
{
	/* reset translation state */
	if(translator)
	{
		translator->filePos = 2; /* 2 to skip array header info */
		translator->pathPos = 2;
		translator->envPos = 2;
		translator->eof = 0;
		translator->reader = 0;
		translator->dir = 0;
	}

	/* get first error */
	return eddml_next_error(translator);
}

int
int_get_data(char **to, char *from)
{
	if(to)
	{
		*to = from;
	}

	return !(to && from);
}

int
eddml_get_name(eddml_translator *t, char **to)
{
	return int_get_data(to, t->set_name);
}

int
eddml_get_version(eddml_translator *t, char **to)
{
	return int_get_data(to, t->set_version);
}

int
eddml_get_authority(eddml_translator *t, char **to)
{
	return int_get_data(to, t->set_authority);
}

int
eddml_get_docs(eddml_translator *t, char **to)
{
	return int_get_data(to, t->set_docs);
}

int
eddml_get_error_code(eddml_translator *t, char **to)
{
	return int_get_data(to, t->error_code);
}

int
eddml_get_error_desc(eddml_translator *t, char **to)
{
	return int_get_data(to, t->error_desc);
}

int
eddml_get_error_severity(eddml_translator *t, char **to)
{
	return int_get_data(to, t->error_severity);
}

int
eddml_get_error_note_titles(eddml_translator *t, char ***to)
{
	if(t->error_note_title == 0)
	{
		*to = 0;
		return 0;
	}

	return int_get_data((char**)to, (char*)((t->error_note_title)+2)); /* 2 to skip state info */
}

int
eddml_get_error_note_bodies(eddml_translator *t, char ***to)
{
	if(t->error_note_text == 0)
	{
		*to = 0;
		return 0;
	}

	return int_get_data((char**)to, (char*)((t->error_note_text)+2)); /* 2 to skip state info */
}

int
eddml_get_error_symbol_languages(eddml_translator *t, char ***to)
{
	if(t->error_symbol_lang == 0)
	{
		*to = 0;
		return 0;
	}

	return int_get_data((char**)to, (char*)((t->error_symbol_lang)+2)); /* 2 to skip state info */
}

int
eddml_get_error_symbol_bodies(eddml_translator *t, char ***to)
{
	if(t->error_symbol_text == 0)
	{
		*to = 0;
		return 0;
	}

	return int_get_data((char**)to, (char*)((t->error_symbol_text)+2)); /* 2 to skip state info */
}
