/*
 * This file contains the public interface of libbubblemon.
 * Libbubblemon is a library for making bubbling load monitors for
 * different platforms.  See for example
 * "http://www.nongnu.org/bubblemon/".
 *
 *  Copyright (C) 1999-2000 Johan Walles - d92-jwa@nada.kth.se
 *  http://www.nongnu.org/libbubblemon/
 *
 *  This program is free software; you can redistribute it and/or modify
 *  it under the terms of the GNU General Public License as published by
 *  the Free Software Foundation; either version 2 of the License, or
 *  (at your option) any later version.
 *
 *  This program is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *  GNU General Public License for more details.
 *
 *  You should have received a copy of the GNU General Public License
 *  along with this program; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Street #330, Boston, MA 02111-1307, USA.
 */

#ifndef BUBBLEMON_H
#define BUBBLEMON_H

/***************************************************************
 *
 * Data types.
 */

// The current state of your bubblemon
typedef void* bubblemon_t;

// An rgba color value
typedef union {
  unsigned int value;
  struct {
    unsigned char r;
    unsigned char g;
    unsigned char b;
    unsigned char a;
  } components;
} bubblemon_color_t;

// An image for on-screen rendering
typedef struct
{
  int width;
  int height;
  
  bubblemon_color_t *pixels;
} bubblemon_picture_t;


/***************************************************************
 *
 * The essentials:
 */

// Create a new bubblemon, start by calling this function
//
// Returns NULL on out of memory
bubblemon_t bubblemon_create(void);

// Render a frame.  Before calling this function, you are responsible
// for that picture->pixels is pointing to a memory area of at least
// picture->width * picture->height * sizeof(bubblemon_color_t) bytes.
//
// Returns:
// 0 on success
// ENOMEM (as defined in <errno.h>) on out of memory
int bubblemon_render(bubblemon_t bubblemon, bubblemon_picture_t *picture);

// Free all resources of a bubblemon previously allocated by
// bubblemon_create().
void bubblemon_destroy(bubblemon_t bubblemon);


/***************************************************************
 *
 * Sensor input that will be visualized:
 */

// Set the water level to 0-100
void bubblemon_setWaterLevel(bubblemon_t bubblemon, int level);

// Set the color intensity to 0-100
void bubblemon_setColorIntensity(bubblemon_t bubblemon, int intensity);

// Set the amount of bubbles to 0-100
void bubblemon_setBubbles(bubblemon_t bubblemon, int bubbles);

// Set the amount of growth to 0-100
void bubblemon_setGrowthIntensity(bubblemon_t bubblemon, int growth);

// Set whether the floater is visible (true / false)
void bubblemon_setFloaterVisible(bubblemon_t bubblemon, int visible);


/***************************************************************
 *
 * Settings
 */

// Get / set the water and air colors
bubblemon_color_t bubblemon_getMaxWaterColor(bubblemon_t bubblemon);
void bubblemon_setMaxWaterColor(bubblemon_t bubblemon,
				bubblemon_color_t color);

bubblemon_color_t bubblemon_getMinWaterColor(bubblemon_t bubblemon);
void bubblemon_setMinWaterColor(bubblemon_t bubblemon,
				bubblemon_color_t color);

bubblemon_color_t bubblemon_getMaxAirColor(bubblemon_t bubblemon);
void bubblemon_setMaxAirColor(bubblemon_t bubblemon,
			      bubblemon_color_t color);

bubblemon_color_t bubblemon_getMinAirColor(bubblemon_t bubblemon);
void bubblemon_setMinAirColor(bubblemon_t bubblemon,
			      bubblemon_color_t color);

// Change the floater image (default is a message-in-a-bottle).  The
// horizontal center of gravity is counted from the top down, so lower
// values are higher up.
//
// Returns:
// 0 on success
// ENOMEM (as defined in <errno.h>) on out of memory.  In this case,
//   the previous image is preserved.
int bubblemon_setFloaterImage(bubblemon_t bubblemon,
			      bubblemon_picture_t *picture,
			      int horizontalCenterOfGravity,
			      int verticalCenterOfGravity);

// Change the color of the stuff growing from the bottom.  The growth
// color is randomized between these two values.
void bubblemon_setGrowthColor(bubblemon_t bubblemon,
			      bubblemon_color_t color1,
			      bubblemon_color_t color2);

// Fetch the current color of the stuff growing from the bottom.  If
// any of the parameters is NULL, that parameter will be ignored.
void bubblemon_getGrowthColor(bubblemon_t bubblemon,
			      bubblemon_color_t *color1,
			      bubblemon_color_t *color2);

#endif  // !defined(BUBBLEMON_H)
